/**
 * @file libcomprex/file.h File structures
 * 
 * $Id: file.h,v 1.32 2003/01/01 06:22:35 chipx86 Exp $
 *
 * @Copyright (C) 2001-2003 The GNUpdate Project.
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Library General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Library General Public License for more details.
 *
 * You should have received a copy of the GNU Library General Public
 * License along with this library; if not, write to the
 * Free Software Foundation, Inc., 59 Temple Place - Suite 330,
 * Boston, MA  02111-1307, USA.
 */
#ifndef _LIBCOMPREX_FILE_H_
#define _LIBCOMPREX_FILE_H_

#include <sys/types.h>
#include <time.h>

#ifdef __cplusplus
extern "C" {
#endif

typedef struct _CxFsNode   CxFile;     /**< A single file.  */
typedef struct _CxFileData CxFileData; /**< File data.      */

#include <libcomprex/fsnode.h>
#include <libcomprex/archive.h>
#include <libcomprex/types.h>

/**
 * The data specific to a file node.
 */
struct _CxFileData
{
	/**
	 * The full physical path to the file. If the file
	 * is non-local, this will be a temporary filename.
	 */
	char *physPath;

	unsigned int compressedSize;   /**< The compressed file size.   */
	unsigned int uncompressedSize; /**< The uncompressed file size. */
};

/**************************************************************************/
/** @name Structure (De)allocation Functions                              */
/**************************************************************************/
/*@{*/

/**
 * Creates a new CxFile structure.
 *
 * @return A new, empty CxFile structure.
 *
 * @see cxDestroyFile()
 */
CxFile *cxNewFile(void);

/**
 * Destroys a CxFile structure.
 *
 * @param file The CxFile to destroy.
 *
 * @see cxNewFile()
 */
void cxDestroyFile(CxFile *file);

/*@}*/

/**************************************************************************/
/** @name Attribute Modification Functions                                */
/**************************************************************************/
/*@{*/

/**
 * Sets the archive that owns the file.
 *
 * This should only be used by libcomprex.
 *
 * @param file    The file.
 * @param archive The archive that owns this file.
 *
 * @see cxGetFileArchive()
 */
void cxSetFileArchive(CxFile *file, CxArchive *archive);

/**
 * Sets the specified file's parent directory.
 *
 * @param file   The file.
 * @param parent The parent directory.
 *
 * @see cxGetFileParent()
 */
void cxSetFileParent(CxFile *file, CxDirectory *parent);

/**
 * Sets the base name of the specified file.
 *
 * @param file The file.
 * @param name The filename.
 *
 * @see cxGetFileName()
 */
void cxSetFileName(CxFile *file, const char *name);

/**
 * Sets the full physical path to the file.
 *
 * If the file is non-local, this should be a temporary file.
 *
 * @param file The file.
 * @param path The physical path to the file.
 *
 * @see cxGetFilePhysicalPath()
 * @see cxGetFilePath()
 */
void cxSetFilePhysicalPath(CxFile *file, const char *path);

/**
 * Sets the mode of the specified file.
 *
 * @param file The file.
 * @param mode The mode.
 *
 * @see cxGetFileMode()
 */
void cxSetFileMode(CxFile *file, mode_t mode);

/**
 * Sets the user ID of the specified file.
 *
 * @param file The file.
 * @param uid  The user ID.
 *
 * @see cxGetFileUid()
 */
void cxSetFileUid(CxFile *file, uid_t uid);

/**
 * Sets the group ID of the specified file.
 *
 * @param file The file.
 * @param gid  The group ID.
 *
 * @see cxGetFileGid()
 */
void cxSetFileGid(CxFile *file, gid_t gid);

/**
 * Sets the compressed size of the specified file.
 *
 * @param file The file.
 * @param size The file's compressed size.
 *
 * @see cxGetFileCompressedSize()
 */
void cxSetFileCompressedSize(CxFile *file, unsigned int size);

/**
 * Sets the uncompressed size of the specified file.
 *
 * @param file The file.
 * @param size The file's uncompressed size.
 *
 * @see cxGetFileSize()
 */
void cxSetFileSize(CxFile *file, unsigned int size);

/**
 * Sets the timestamp of the specified file.
 *
 * @param file The file.
 * @param date The date/time timestamp in seconds since the epoch.
 *
 * @see cxGetFileDate()
 */
void cxSetFileDate(CxFile *file, time_t date);

/**
 * Sets whether or not the file is stored locally.
 *
 * If the file is located online, or within an archive, this should be 0.
 * Otherwise, this should be 1.
 *
 * @param file    The file.
 * @param isLocal 1 if the file is stored locally; 0 otherwise.
 *
 * @see cxIsFileLocal()
 */
void cxSetFileLocal(CxFile *file, char isLocal);

/*@}*/

/**************************************************************************/
/** @name Attribute Retrieval Functions                                   */
/**************************************************************************/
/*@{*/

/**
 * Returns the archive that owns the specified file.
 *
 * @param file The file.
 *
 * @return The archive that owns this file.
 *
 * @see cxSetFileArchive()
 */
CxArchive *cxGetFileArchive(CxFile *file);

/**
 * Returns the specified file's parent directory.
 *
 * @param file The file.
 *
 * @return The specified file's parent directory.
 *
 * @see cxSetFileParent()
 */
CxDirectory *cxGetFileParent(CxFile *file);

/**
 * Returns the name of the specified file.
 *
 * @param file The file.
 *
 * @return The file's name.
 *
 * @see cxSetFileName()
 */
const char *cxGetFileName(CxFile *file);

/**
 * Returns the full path to the specified file.
 *
 * This may include such non-physical elements as FTP sites, archives,
 * etc. in the path.
 *
 * @param file The file.
 *
 * @return The path to the file.
 *
 * @see cxGetFilePhysicalPath()
 * @see cxSetFilePhysicalPath()
 */
const char *cxGetFilePath(CxFile *file);

/**
 * Returns the full physical path to the file.
 *
 * If the file is non-local, this will be a temporary file.
 *
 * @param file The file.
 *
 * @return The physical path to the file.
 *
 * @see cxSetFilePhysicalPath()
 * @see cxGetFilePath()
 */
const char *cxGetFilePhysicalPath(CxFile *file);

/**
 * Returns the mode of the specified file.
 *
 * @param file The file.
 *
 * @return The file's mode.
 *
 * @see cxSetFileMode()
 */
mode_t cxGetFileMode(CxFile *file);

/**
 * Returns the user ID of the specified file.
 *
 * @param file The file.
 *
 * @return The user ID of the specified file.
 *
 * @see cxSetFileUid()
 */
uid_t cxGetFileUid(CxFile *file);

/**
 * Returns the group ID of the specified file.
 *
 * @param file The file.
 *
 * @return The group ID of the specified file.
 *
 * @see cxSetFileGid()
 */
gid_t cxGetFileGid(CxFile *file);

/**
 * Returns the compressed size of the specified file.
 *
 * @param file The file.
 *
 * @return The file's compressed size.
 *
 * @see cxSetFileCompressedSize()
 */
unsigned int cxGetFileCompressedSize(CxFile *file);

/**
 * Returns the uncompressed size of the specified file.
 *
 * @param file The file.
 *
 * @return The file's uncompressed size.
 *
 * @see cxSetFileSize()
 */
unsigned int cxGetFileSize(CxFile *file);

/**
 * Returns the timestamp of the specified file.
 *
 * @param file The file.
 *
 * @return The date/time timestamp in seconds since the epoch.
 *
 * @see cxSetFileDate()
 */
time_t cxGetFileDate(CxFile *file);

/**
 * Returns whether or not the file is stored locally.
 *
 * If the file is located online, or within an archive, this will be 0.
 * Otherwise, this will be 1.
 *
 * @param file The file.
 *
 * @return 1 if the file is stored locally; 0 otherwise.
 *
 * @see cxSetFileLocal()
 */
char cxIsFileLocal(CxFile *file);

/*@}*/

/**************************************************************************/
/** @name Iteration Functions                                             */
/**************************************************************************/
/*@{*/

/**
 * Returns the next file in a list of files.
 *
 * @param file The current file.
 *
 * @return The next file in the list.
 * 
 * @see cxGetFirstFile()
 * @see cxGetPreviousFile()
 */
CxFile *cxGetNextFile(CxFile *file);

/**
 * Returns the previous file in a list of files.
 *
 * @param file The current file.
 * 
 * @return The previous file in the list.
 *
 * @see cxGetFirstFile()
 * @see cxGetNextFile()
 */
CxFile *cxGetPreviousFile(CxFile *file);

/*@}*/

#ifdef __cplusplus
}
#endif

#endif /* _LIBCOMPREX_FILE_H_ */

