/*
 * Decompiled with CFR 0.152.
 */
package org.apache.cassandra.auth;

import com.google.common.collect.ImmutableSet;
import com.google.common.collect.Lists;
import com.google.common.util.concurrent.UncheckedExecutionException;
import java.net.InetAddress;
import java.nio.ByteBuffer;
import java.nio.charset.StandardCharsets;
import java.util.Arrays;
import java.util.Map;
import java.util.Set;
import java.util.concurrent.ExecutionException;
import org.apache.cassandra.auth.AuthCache;
import org.apache.cassandra.auth.AuthCacheMBean;
import org.apache.cassandra.auth.AuthenticatedUser;
import org.apache.cassandra.auth.CassandraRoleManager;
import org.apache.cassandra.auth.DataResource;
import org.apache.cassandra.auth.IAuthenticator;
import org.apache.cassandra.config.DatabaseDescriptor;
import org.apache.cassandra.config.Schema;
import org.apache.cassandra.cql3.QueryOptions;
import org.apache.cassandra.cql3.QueryProcessor;
import org.apache.cassandra.cql3.UntypedResultSet;
import org.apache.cassandra.cql3.statements.SelectStatement;
import org.apache.cassandra.exceptions.AuthenticationException;
import org.apache.cassandra.exceptions.ConfigurationException;
import org.apache.cassandra.exceptions.RequestExecutionException;
import org.apache.cassandra.service.ClientState;
import org.apache.cassandra.service.QueryState;
import org.apache.cassandra.transport.messages.ResultMessage;
import org.apache.cassandra.utils.ByteBufferUtil;
import org.mindrot.jbcrypt.BCrypt;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class PasswordAuthenticator
implements IAuthenticator {
    private static final Logger logger = LoggerFactory.getLogger(PasswordAuthenticator.class);
    private static final String SALTED_HASH = "salted_hash";
    public static final String USERNAME_KEY = "username";
    public static final String PASSWORD_KEY = "password";
    private static final byte NUL = 0;
    private SelectStatement authenticateStatement;
    public static final String LEGACY_CREDENTIALS_TABLE = "credentials";
    private SelectStatement legacyAuthenticateStatement;
    private CredentialsCache cache;

    @Override
    public boolean requireAuthentication() {
        return true;
    }

    protected static boolean checkpw(String password, String hash) {
        try {
            return BCrypt.checkpw((String)password, (String)hash);
        }
        catch (Exception e) {
            logger.warn("Error: invalid password hash encountered, rejecting user", (Throwable)e);
            return false;
        }
    }

    private AuthenticatedUser authenticate(String username, String password) throws AuthenticationException {
        try {
            String hash = (String)this.cache.get(username);
            if (!PasswordAuthenticator.checkpw(password, hash)) {
                throw new AuthenticationException(String.format("Provided username %s and/or password are incorrect", username));
            }
            return new AuthenticatedUser(username);
        }
        catch (UncheckedExecutionException | ExecutionException e) {
            if (e.getCause() instanceof NoSuchCredentialsException) {
                throw new AuthenticationException(String.format("Provided username %s and/or password are incorrect", username));
            }
            if (e.getCause() instanceof RequestExecutionException) {
                logger.trace("Error performing internal authentication", e);
                throw new AuthenticationException(String.format("Error during authentication of user %s : %s", username, e.getMessage()));
            }
            throw new RuntimeException(e);
        }
    }

    private String queryHashedPassword(String username) throws NoSuchCredentialsException {
        try {
            SelectStatement authenticationStatement = this.authenticationStatement();
            ResultMessage.Rows rows = authenticationStatement.execute(QueryState.forInternalCalls(), QueryOptions.forInternalCalls(CassandraRoleManager.consistencyForRole(username), Lists.newArrayList((Object[])new ByteBuffer[]{ByteBufferUtil.bytes(username)})), System.nanoTime());
            if (rows.result.isEmpty()) {
                throw new NoSuchCredentialsException();
            }
            UntypedResultSet result = UntypedResultSet.create(rows.result);
            if (!result.one().has(SALTED_HASH)) {
                throw new NoSuchCredentialsException();
            }
            return result.one().getString(SALTED_HASH);
        }
        catch (RequestExecutionException e) {
            logger.trace("Error performing internal authentication", (Throwable)e);
            throw e;
        }
    }

    private SelectStatement authenticationStatement() {
        if (Schema.instance.getCFMetaData("system_auth", LEGACY_CREDENTIALS_TABLE) == null) {
            return this.authenticateStatement;
        }
        if (this.legacyAuthenticateStatement == null) {
            this.prepareLegacyAuthenticateStatement();
        }
        return this.legacyAuthenticateStatement;
    }

    public Set<DataResource> protectedResources() {
        return ImmutableSet.of((Object)DataResource.table("system_auth", "roles"));
    }

    @Override
    public void validateConfiguration() throws ConfigurationException {
    }

    @Override
    public void setup() {
        String query = String.format("SELECT %s FROM %s.%s WHERE role = ?", SALTED_HASH, "system_auth", "roles");
        this.authenticateStatement = PasswordAuthenticator.prepare(query);
        if (Schema.instance.getCFMetaData("system_auth", LEGACY_CREDENTIALS_TABLE) != null) {
            this.prepareLegacyAuthenticateStatement();
        }
        this.cache = new CredentialsCache(this);
    }

    private void prepareLegacyAuthenticateStatement() {
        String query = String.format("SELECT %s from %s.%s WHERE username = ?", SALTED_HASH, "system_auth", LEGACY_CREDENTIALS_TABLE);
        this.legacyAuthenticateStatement = PasswordAuthenticator.prepare(query);
    }

    @Override
    public AuthenticatedUser legacyAuthenticate(Map<String, String> credentials) throws AuthenticationException {
        String username = credentials.get(USERNAME_KEY);
        if (username == null) {
            throw new AuthenticationException(String.format("Required key '%s' is missing", USERNAME_KEY));
        }
        String password = credentials.get(PASSWORD_KEY);
        if (password == null) {
            throw new AuthenticationException(String.format("Required key '%s' is missing for provided username %s", PASSWORD_KEY, username));
        }
        return this.authenticate(username, password);
    }

    @Override
    public IAuthenticator.SaslNegotiator newSaslNegotiator(InetAddress clientAddress) {
        return new PlainTextSaslAuthenticator();
    }

    private static SelectStatement prepare(String query) {
        return (SelectStatement)QueryProcessor.getStatement((String)query, (ClientState)ClientState.forInternalCalls()).statement;
    }

    private static final class NoSuchCredentialsException
    extends RuntimeException {
        private NoSuchCredentialsException() {
        }
    }

    public static interface CredentialsCacheMBean
    extends AuthCacheMBean {
        public void invalidateCredentials(String var1);
    }

    private static class CredentialsCache
    extends AuthCache<String, String>
    implements CredentialsCacheMBean {
        private CredentialsCache(PasswordAuthenticator authenticator) {
            super("CredentialsCache", DatabaseDescriptor::setCredentialsValidity, DatabaseDescriptor::getCredentialsValidity, DatabaseDescriptor::setCredentialsUpdateInterval, DatabaseDescriptor::getCredentialsUpdateInterval, DatabaseDescriptor::setCredentialsCacheMaxEntries, DatabaseDescriptor::getCredentialsCacheMaxEntries, x$0 -> authenticator.queryHashedPassword(x$0), () -> true);
        }

        @Override
        public void invalidateCredentials(String roleName) {
            this.invalidate(roleName);
        }
    }

    private class PlainTextSaslAuthenticator
    implements IAuthenticator.SaslNegotiator {
        private boolean complete = false;
        private String username;
        private String password;

        private PlainTextSaslAuthenticator() {
        }

        @Override
        public byte[] evaluateResponse(byte[] clientResponse) throws AuthenticationException {
            this.decodeCredentials(clientResponse);
            this.complete = true;
            return null;
        }

        @Override
        public boolean isComplete() {
            return this.complete;
        }

        @Override
        public AuthenticatedUser getAuthenticatedUser() throws AuthenticationException {
            if (!this.complete) {
                throw new AuthenticationException("SASL negotiation not complete");
            }
            return PasswordAuthenticator.this.authenticate(this.username, this.password);
        }

        private void decodeCredentials(byte[] bytes) throws AuthenticationException {
            logger.trace("Decoding credentials from client token");
            byte[] user = null;
            byte[] pass = null;
            int end = bytes.length;
            for (int i = bytes.length - 1; i >= 0; --i) {
                if (bytes[i] != 0) continue;
                if (pass == null) {
                    pass = Arrays.copyOfRange(bytes, i + 1, end);
                } else if (user == null) {
                    user = Arrays.copyOfRange(bytes, i + 1, end);
                }
                end = i;
            }
            if (pass == null) {
                throw new AuthenticationException("Password must not be null");
            }
            if (user == null) {
                throw new AuthenticationException("Authentication ID must not be null");
            }
            this.username = new String(user, StandardCharsets.UTF_8);
            this.password = new String(pass, StandardCharsets.UTF_8);
        }
    }
}

