# encoding: utf-8

require "spec_helper"
require "amq/client/settings"

describe AMQ::Client::Settings do
  describe ".default" do
    it "should provide some default values" do
      AMQ::Client::Settings.default.should_not be_nil
      AMQ::Client::Settings.default[:host].should_not be_nil
    end
  end

  describe ".configure(&block)" do
    it "should merge custom settings with default settings" do
      settings = AMQ::Client::Settings.configure(:host => "tagadab")
      settings[:host].should eql("tagadab")
    end

    it "should merge custom settings from AMQP URL with default settings" do
      settings = AMQ::Client::Settings.configure("amqp://tagadab")
      settings[:host].should eql("tagadab")
    end
  end

  describe ".parse_amqp_url(connection_string)" do
    context "when schema is not one of [amqp, amqps]" do
      it "raises ArgumentError" do
        expect {
          described_class.parse_amqp_url("http://dev.rabbitmq.com")
        }.to raise_error(ArgumentError, /amqp or amqps schema/)
      end
    end


    it "handles amqp:// URIs w/o path part" do
      val = described_class.parse_amqp_url("amqp://dev.rabbitmq.com")

      val[:vhost].should be_nil # in this case, default / will be used
      val[:host].should == "dev.rabbitmq.com"
      val[:port].should == 5672
      val[:scheme].should == "amqp"
      val[:ssl].should be_false
    end

    it "handles amqps:// URIs w/o path part" do
      val = described_class.parse_amqp_url("amqps://dev.rabbitmq.com")

      val[:vhost].should be_nil
      val[:host].should == "dev.rabbitmq.com"
      val[:port].should == 5671
      val[:scheme].should == "amqps"
      val[:ssl].should be_true
    end


    context "when URI ends in a slash" do
      it "parses vhost as an empty string" do
        val = described_class.parse_amqp_url("amqp://dev.rabbitmq.com/")

        val[:host].should == "dev.rabbitmq.com"
        val[:port].should == 5672
        val[:scheme].should == "amqp"
        val[:ssl].should be_false
        val[:vhost].should == ""
      end
    end


    context "when URI ends in /%2Fvault" do
      it "parses vhost as /vault" do
        val = described_class.parse_amqp_url("amqp://dev.rabbitmq.com/%2Fvault")

        val[:host].should == "dev.rabbitmq.com"
        val[:port].should == 5672
        val[:scheme].should == "amqp"
        val[:ssl].should be_false
        val[:vhost].should == "/vault"
      end
    end


    context "when URI is amqp://dev.rabbitmq.com/a.path.without.slashes" do
      it "parses vhost as a.path.without.slashes" do
        val = described_class.parse_amqp_url("amqp://dev.rabbitmq.com/a.path.without.slashes")

        val[:host].should == "dev.rabbitmq.com"
        val[:port].should == 5672
        val[:scheme].should == "amqp"
        val[:ssl].should be_false
        val[:vhost].should == "a.path.without.slashes"
      end
    end

    context "when URI is amqp://dev.rabbitmq.com/a/path/with/slashes" do
      it "raises an ArgumentError" do
        lambda { described_class.parse_amqp_url("amqp://dev.rabbitmq.com/a/path/with/slashes") }.should raise_error(ArgumentError)
      end
    end


    context "when URI has username:password, for instance, amqp://hedgehog:t0ps3kr3t@hub.megacorp.internal" do
      it "parses them out" do
        val = described_class.parse_amqp_url("amqp://hedgehog:t0ps3kr3t@hub.megacorp.internal")

        val[:host].should == "hub.megacorp.internal"
        val[:port].should == 5672
        val[:scheme].should == "amqp"
        val[:ssl].should be_false
        val[:user].should == "hedgehog"
        val[:pass].should == "t0ps3kr3t"
        val[:vhost].should be_nil # in this case, default / will be used
      end
    end
  end
end
