
# IRuby Input

This README is generated from README.ipynb. Please do not edit this file directly.

The `IRuby::Input` class makes it easier for IRuby users to get input from users. For example:


```ruby
name = IRuby.input 'Enter your name'
```

The following input methods are supported on the `IRuby` module:

| method | description |
| -------- | -------- |
| `input(prompt)`   | Prompts the user for a line of input |
| `password(prompt)`   | Prompts the user for a password |
| `form(&block)`   | Presents a form to the user |
| `popup(title,&block)`   | Displays a form to the user as a popup |

## Forms

Forms are groups of inputs to be collected from the user. For example:


```ruby
result = IRuby.form do 
  input :username
  password :password
  button
end
```

The following methods are available to build forms: 

| method | description |
| -------- | -------- |
| `input(key=:input)`   | Prompts the user for a line of input |
| `textarea(key=:textarea),`   | Adds a textarea to the form |
| `password(key=:password)`   | Prompts the user for a password |
| `button(key=:done, color: :blue)`   | Adds a submit button to the form |
| `cancel(prompt='Cancel')`   | Adds a cancel button to the form |
| `text(string)`   | Adds text to the form |
| `html(&block)`   | Inserts HTML from the given [erector block](https://github.com/erector/erector)   |
| `file(key=:file)`   | Adds a file input to the form |
| `date(key=:date)`   | Adds a date picker to the form |
| `select(*options)`   | Adds a dropdown select input to the form |
| `radio(*options)`   | Adds a radio select input to the form |
| `checkbox(*options)`   | Adds checkbox inputs to the form |

## Popups
 
Popups are just forms in a bootstrap modal. They are useful when users **Run All** in a notebook with a lot of inputs. The popups always appear in the same spot, so users don't have to scroll down to find the next input. 

Popups accept a `title` argument, for example: 


```ruby
result = IRuby.popup 'Please enter your name' do 
  input
  button
end
```

## Submit and cancel

The enter key will submit an input or form and the escape key will cancel it. Canceled inputs are returned as `nil`. Inputs are automatically canceled if destroyed. An input can be destroyed by clearing its cell's output. The `cancel` button will cancel a form and all other buttons will submit it. 

After a form destroyed, the cell's output is cleared. Be careful not to prompt for input in a block that has previous output you would like to keep. Output is cleared to prevent forms from interferring with one another and to ensure that inputs are not inadvertently saved to the notebook. 


```ruby
result = IRuby.popup 'Confirm' do 
  text 'Are you sure you want to continue?'
  cancel 'No'
  button 'Yes'
end
```

## Custom keys

Every widget has an entry in the final results hash. A custom key can be passed as the first parameter to the hash. If no key is provided, the widget name is used as the key. The `cancel` widget has no key; it's first parameter is its label. 


```ruby
result = IRuby.form do
  input :username
  password :password
end
```

## Custom labels

Field labels appear to the left of the field. Button labels appear as the text on the button. `cancel` labels are passed as the first argument. All other widgets' labels are set using the `label` parameter. 


```ruby
result = IRuby.form do 
  input :name, label: 'Please enter your name'
  cancel 'None of your business!'
  button :submit, label: 'All done'
end
```

## Defaults

Most inputs will accept a `default` parameter. If no default is given, the deault is `nil`. Since checkboxes can have multiple values selected, you can pass an array of values. To check everything, pass `true` as the default. 


```ruby
result = IRuby.form do 
  checkbox :one, 'Fish', 'Cat', 'Dog', default: 'Fish'
  checkbox :many, 'Fish', 'Cat', 'Dog', default: ['Cat', 'Dog']
  checkbox :all, 'Fish', 'Cat', 'Dog', default: true
  button :submit, label: 'All done'
end
```

## Dates

The `date` widget provides a calendar popup and returns a `Time` object. It's default should also be a `Time` object. 


```ruby
result = IRuby.form do 
  date :birthday
  date :today, default: Time.now
  button
end
```

## Buttons

Buttons do not appear in the final hash unless they are clicked. If clicked, their value is `true`. Here are the various colors a button can be:


```ruby
result = IRuby.form do 
  IRuby::Input::Button::COLORS.each_key do |color|
    button color, color: color
  end
end
```

## Textareas

Textareas are multiline inputs that are convenient for larger inputs. If you need a line return when typing in a textarea, use shift+enter since enter will submit the form.


```ruby
result = IRuby.form do 
  text 'Enter email addresses, one per line (use shift+enter for newlines)'
  textarea :emails
end
```

## Text and HTML

You can insert lines of text or custom html using their respective helpers:


```ruby
result = IRuby.form do 
  html { h1 'Choose a Stooge' }
  text 'Choose your favorite stooge'
  select :stooge, 'Moe', 'Larry', 'Curly'
  button
end
```

## Dropdowns

A `select` is a dropdown of options. Use a `multiple` to allow multiple selections. `multiple` widgets accept an additional `size` parameters that determines the number of rows. The default is 4. 


```ruby
result = IRuby.form do 
  select :stooge, 'Moe', 'Larry', 'Curly'
  select :stooge, 'Moe', 'Larry', 'Curly', default: 'Moe'
  multiple :stooges, 'Moe', 'Larry', 'Curly', default: true, size: 3
  multiple :stooges, 'Moe', 'Larry', 'Curly', default: ['Moe','Curly']
  button
end
```

## Radio selects and checkboxes

Like selects, radio selects and checkboxes take multiple arguments, each one an option. If the first argument is a symbol, it is used as the key. 

Note that the `checkbox` widget will always return `nil` or an array. 


```ruby
result = IRuby.form do 
  radio :children, *(0..12), label: 'How many children do you have?'
  checkbox :gender, 'Male', 'Female', 'Intersex', label: 'Select the genders of your children'
  button
end
```

## Files

Since file widgets capture the enter key, you should include a button when creating forms that contain only a file input:


```ruby
IRuby.form do 
  file :avatar, label: 'Choose an Avatar'
  button :submit
end
```

File widgets return a hash with three keys: 

* `data`: The contents of the file as a string
* `content_type`: The type of file, such as `text/plain` or `image/jpeg`
* `name`: The name of the uploaded file

## Example

Here is an example form that uses every built-in widget. 


```ruby
result = IRuby.form do 
  html { h1 'The Everything Form' }
  text 'Marvel at the strange and varied inputs!'
  date
  file
  input :username
  password
  textarea
  radio *(1..10)
  checkbox 'Fish', 'Cat', 'Dog', label: 'Animals'
  select :color, *IRuby::Input::Button::COLORS.keys
  cancel                     
  button    
end
```

## Writing your own widget

Most form methods are `IRuby::Input::Widget` instances. A `Widget` is an [`Erector::Widget`](https://github.com/erector/erector) with some additional helpers. Here is the `cancel` widget:

```ruby
module IRuby
  module Input
    class Cancel < Widget
      needs :label

      builder :cancel do |label='Cancel'|
        add_button Cancel.new(label: label)
      end

      def widget_css
        ".iruby-cancel { margin-left: 5px; }"
      end

      def widget_js
        <<-JS
          $('.iruby-cancel').click(function(){
            $('#iruby-form').remove();
          });
        JS
      end

      def widget_html
        button(
          @label,
          type: 'button', 
          :'data-dismiss' => 'modal',
          class: "btn btn-danger pull-right iruby-cancel"
        )
      end
    end
  end
end
```

The following methods are available for widgets to use or override: 

| method | description |
| -------- | -------- |
| `widget_js`   | Returns the widget's Javascript |
| `widget_css`   | Returns the widget's CSS |
| `widget_html`   | Returns the widget's   |
| `builder(method,&block)`   | Class method to add form building helpers. |

The following methods are available in the `builder` block:

| method | description |
| -------- | -------- |
| `add_field(field)` | Adds a widget to the form's field area |
| `add_button(button)` | Adds a button to the form's button area |
| `process(key,&block)`   | Register a custom processing block for the given key in the results hash |
| `unique_key(key)` | Returns a unique key for the given key. Use this to make sure that there are no key collisions in the final results hash. |

A canceled form always returns `nil`. Otherwise, the form collects any element with a `data-iruby-key` and non-falsey `data-iruby-value` and passes those to the processor proc registered for the key. See the `File` widget for a more involved example of processing results.
