/* linda.h -- linda parallel programming library
   Copyright (C) 1997, 1998, 1999, 2000, 2001, 2002 Wong Weng Fai <wongwf@comp.nus.edu.sg>

   This file is part of linda.

   linda is free software; you can redistribute it and/or modify it
   under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2, or (at your option)
   any later version.

   linda is distributed in the hope that it will be useful, but
   WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.	 See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with linda; see the file COPYING.  If not, write to
   the Free Software Foundation, 675 Mass Ave, Cambridge, MA 02139, USA.  */

#ifndef G_LINDA_H
#define G_LINDA_H 

#include <glib.h>

G_BEGIN_DECLS

const gchar * linda_version(void);

/* linda_init --- initializes linda.
 * linda_init must not be called directly or indirectly
 * from another glib-function, e.g. as a callback.
 * VTABLE is passed to gthread_init. Normally VTABLE could 
 * be NULL.
 */
void linda_init  (GThreadFunctions       *vtable);

/* linda_end --- frees all alocated memory for linda.
 * Also waits for all spawned threads exiting.
 * the root thread(main thread) should call this.
 */
void linda_end   (void);

/* linda_spawn --- launches a new thread that drives F.
 * Return 0  if spawning is successful .
 * Return -1 fi spawning is failed. 
 */
gint  linda_spawn      (GVoidFunc f);

/* linda_out --- writes out an tuple to tuplespace.
 */
void linda_out       (const gchar *tuple_mask, ...);

/* linda_in, linda_inp and linda_inp_timed --- reads in an tuple from tuplespace.
 * The read tuple in tuplespace is removed.
 * If there is no tuple that matches to TUPLE_MASK,
 * linda_in is blocked. In other hand linda_inp is returned
 * immediately.
 * linda_inp returns non-zero value if a tuple that matches
 * to TUPLE_MASK is existed. returns 0 if not.
 * linda_inp_timed is linda_inp with an timer. linda_inp_timed
 * waits for appearing new tuple matched to TUPLE_MASK in the tuplespace
 * until the time specified by END_TIME. To set END_TIME, 
 * g_get_current_time and g_time_val_add are useful.
 */
void linda_in        (const gchar *tuple_mask, ...);
gint linda_inp       (const gchar *tuple_mask, ...);
gint linda_inp_timed (GTimeVal * end_time, const gchar *tuple_mask, ...);

/* linda_rd, linda_rdp and linda_rdp_timed --- Almost the same as linda_in,
 * linda_inp and linda_inp_timed. But the read tuple in tuplespace is not
 * removed. So another thread can read the same tuple in the 
 * tuplespace.
 */
void linda_rd        (const gchar *tuple_mask, ...);
gint linda_rdp       (const gchar *tuple_mask, ...);
gint linda_rdp_timed (GTimeVal * end_time, const gchar *tuple_mask, ...);

G_END_DECLS

#endif /* Not def: G_LINDA_H */
