package PAR::StrippedPARL::Dynamic;
use 5.008001;
use strict;
use warnings;
our $VERSION = '0.958';

use base 'PAR::StrippedPARL::Base';

our $Data_Pos = tell DATA;

=head1 NAME

PAR::StrippedPARL::Dynamic - Data package containing a dynamic PARL

=head1 SYNOPSIS

  # For details, see PAR::StrippedPARL::Base.
  PAR::StrippedPARL::Dynamic->write_parl($file) or die "Some error...";

=head1 DESCRIPTION

This class is internal to PAR. Do not use it outside of PAR.

This class is basically just a container for a dynamic binary PAR loader
which doesn't include the PAR code like the F<parl> or F<parl.exe>
you are used to. If you're really curious, I'll tell you it is
just a copy of the F<myldr/par> (or F<myldr/par.exe>) file.

The data is appended during the C<make> phase of the PAR build process,
but only if applicable: If you perl is static, you won't get the dynamic
loader.

If the binary data isn't appended during the build process, the class
methods will return the empty list.

=head1 CLASS METHODS

Inherits the methods from L<PAR::StrippedPARL::Base>.

=cut

sub _data_pos {
    my $class = shift;
    return $Data_Pos;
}

=head1 AUTHORS

Steffen Mueller E<lt>smueller@cpan.orgE<gt>,
Audrey Tang E<lt>cpan@audreyt.orgE<gt>

=head1 COPYRIGHT AND LICENSE

Copyright 2006-2009 by Steffen Mueller E<lt>smueller@cpan.orgE<gt>.

This program is free software; you can redistribute it and/or 
modify it under the same terms as Perl itself.

See F<LICENSE>.

=cut

$^V eq 5.42.0 
    or die sprintf("Perl (%s) version (%vd) doesn't match the version (5.42.0) ".
                   "that PAR::Packer was built with; please rebuild PAR::Packer", 
                   $^X, $^V);

1;

__DATA__
M?T5,1@$!`0````````````,`*``!````F!8``#0```#<E0````0`!30`(``'
M`"@`'``;``8````T````-````#0```#@````X`````0````$`````P```!0!
M```4`0``%`$``!<````7````!`````$````!````````````````````$)``
M`!"0```%```````!``$````0D```$)`!`!"0`0"T`@``-`,```8```````$`
M`@```!R0```<D`$`')`!`/````#P````!@````0````$````+`$``"P!```L
M`0``3````$P````$````!````%#E=&0XB@``.(H``#B*```$`0``!`$```0`
M```$````+W5S<B]L:6)E>&5C+VQD+F5L9E]S;P``!P````0````!````3F5T
M0E-$``!`#*H[!`````0````#````4&%8```````'````"0````4```!.971"
M4T0``&5A<FUV-VAF`````&$```!K```````````````P````#0``````````
M````.`````X```!``````````"\``````````P```$X```!C`````````!\`
M``!-````!@```!T````4`````````$,```!8````7P`````````7````/```
M`%H```!D`````````&<```!I````+@```!(```!9````10```!4```!>````
M1P````````!*````1@```#4```!=````)0```"D`````````!P```$\`````
M````"P```"L```!J````7````$P````@````4````&@`````````80``````
M```8````+0```$$````````````````````1````,@``````````````8@``
M`"H`````````"0`````````C````.0````````!1````!0```$@````;````
M.P````````````````````````!+`````````#H```!4````90````````!F
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````$P``````````````
M````````````````````````````````(@```"$`````````*``````````*
M````)@```"<````L```````````````,`````````!P````:````-````!D`
M`````````````!``````````,P```#$````W````'@``````````````````
M````````````!````````````````````$(````D````````````````````
M%@`````````_````1`````@````````````````````^`````````%,`````
M`````````%<``````````````#T```!6`````````%(`````````20````\`
M`````````````%4```!;````-@```&``````````````````````````````
M``````"8%@````````,`"P``````&)`!```````#`!$`D@``````````````
M$@````P``````````````!(````4```````````````2````<P,``&`A```(
M````$@`+`!P``````````````"(````K```````````````2````/`,`````
M````````$@```(0#``#$D@$``````!``%0`Q```````````````2````D@,`
M`#`L```8````$@`+`#@``````````````!(```!)`P`````````````2````
M0```````````````$@```*T#``#0-@``6````!(`"P"#```````````````2
M````P0,``$23`0``````$``5`$@``````````````!(```"Y`P``Q)(!````
M```0`!0`6```````````````$@```&<``````````````!(```!M````````
M```````2````=```````````````$@```'X``````````````!(```"(````
M```````````2````P`,``$23`0``````$``5`)$``````````````!(```!0
M`@`````````````2````EP``````````````$@```)X``````````````!(`
M``"L```````````````2````O0``````````````$0```,H`````````````
M`!(```#,`P``Y",``"0````2``L`V0,```@D``!0````$@`+`.4#``!D+0``
MR`8``!(`"P#8```````````````6````ZP``````````````$@```%L#````
M`````````!(```#V```````````````2````_P```$"3`0`$````$0`5``<!
M`````````````!(````0`0`````````````2````,@,`````````````$@``
M`!P!`````````````!(````C`0`````````````2````\@,``(1#```(`0``
M$@`+`#8!`````````````!$```!(`0`````````````2````6`$`````````
M````$@```&8!`````````````!(```!K`0`````````````1````>@$`````
M````````$@```/T#``#`D@$`!````!$`%`")`0`````````````2````E0$`
M````````````$@```*H!`````````````!(```"U`0`````````````2````
M;@,``$23`0``````$``5`/,"`````````````!(```#(`0`````````````2
M````0`0``)@6````````$@`+`-8!`````````````!(```#>`0``````````
M```B````]@$`````````````$@````("`````````````!(````'!```1),!
M```````0`!4`#P0``%@D```X!0``$@`+``@"`````````````"(````>`@``
M```````````2````*0(`````````````$@```!P$``",1```4`$``!(`"P`Z
M`@`````````````2````)@0``,22`0``````$``5`#($``",(```7````!(`
M"P`Z!```K$8``+0"```2``L`00(`````````````$@```#\$``"8%@``````
M`!(`"P!(`@`````````````2````1P0``&@A``!\`@``$@`+`%0$``!(+```
M'`$``!(`"P!/`@`````````````2````9@0``)`I``"4````$@`+`%@"````
M`````````!(```!I`@`````````````2````W`(`````````````$@```'`"
M`````````````!(```!T!```)"H``&0````2``L`?P(`````````````$@``
M`(@"`````````````!(```".`@`````````````2````)0,`````````````
M$@```),"`````````````!(````8`P``!),!``0````1`!4`FP(`````````
M````$@```%`#`````````````!(```"B`@`````````````2````?`0``"A#
M``!<````$@`+`*D"`````````````!(```"X`@`````````````1````#0,`
M`'R2`0`$````$0`4`(4$```L-```I`(``!(`"P`O`0`````````````2````
MO0(`````````````$@```&(#`````````````!(`````;&EB<&5R;"YS;P!R
M96%L;&]C`&UE;6UO=F4`7U]C>&%?9FEN86QI>F4`<FUD:7(`<W1R8VAR`'-T
M<FYC;7``<W1R<F-H<@!B;V]T7T1Y;F%,;V%D97(`<&5R;%]C;VYS=')U8W0`
M86)O<G0`=6YL:6YK`%]?;'-T870U,`!P97)L7V9R964`7U]S=&%T-3``7V5X
M:70`;65M8W!Y`'!E<FQ?9&5S=')U8W0`7U]S=&%C:U]C:&M?9F%I;`!03%]D
M;U]U;F1U;7``4&5R;%]S=E]S971P=@!03%]C=7)R96YT7V-O;G1E>'0`<&5R
M;%]A;&QO8P!C;&]S961I<@!E;G9I<F]N`')E861L:6YK`%]?;W!E;F1I<C,P
M`&UE;7-E=`!097)L7W-A9F5S>7-M86QL;V,`7U]S=&%C:U]C:&M?9W5A<F0`
M4&5R;%]G=E]F971C:'!V`%]?<W!R:6YT9E]C:&L`<F5A9`!03%]M96UO<GE?
M=W)A<`!097)L7W-Y<U]I;FET,P!097)L7V=E=%]S=@!097)L7V-R;V%K7VYO
M8V]N=&5X=`!097)L7VYE=UA3`%!E<FQ?871F;W)K7W5N;&]C:P!097)L7W-Y
M<U]T97)M`%]?97)R;F\`7U]D97)E9VES=&5R7V9R86UE7VEN9F\`4&5R;%]M
M9U]S970`;6MD:7(`7U]R96=I<W1E<E]F<F%M95]I;F9O`'!E<FQ?<&%R<V4`
M4&5R;%]N97=84U]F;&%G<P!A8V-E<W,`9V5T<&ED`&UE;6-H<@!V9G!R:6YT
M9@!097)L7V%T9F]R:U]L;V-K`&UE;6-M<`!097)L7W-V7W-E='!V;@!P97)L
M7W)U;@!L<V5E:P!O<&5N`'-P<FEN=&8`9V5T=6ED`'-T<FQE;@!?7W1L<U]G
M971?861D<@!?7W-&`'-T<F-M<`!L:6)M+G-O+C``;&EB8W)Y<'0N<V\N,0!S
M=')D=7``;&EB<'1H<F5A9"YS;RXQ`'!T:')E861?871F;W)K`&QI8F,N<V\N
M,3(`7U]P<F]G;F%M90!?7W!S7W-T<FEN9W,`7U]G971P=W5I9#4P`%]?<WES
M8V%L;`!?7W-T<&-P>5]C:&L`871E>&ET`%]L:6)C7VEN:70`<W1R=&]K`%]?
M<F5A9&1I<C,P`%]E;F0`<&%R7V-U<G)E;G1?97AE8P!?7V)S<U]S=&%R=%]?
M`&=E=%]U<V5R;F%M95]F<F]M7V=E='!W=6ED`'!A<E]C;&5A;G5P`%]E9&%T
M80!?7V)S<U]E;F1?7P!P87)?8F%S96YA;64`<&%R7V1I<FYA;64`<&%R7VUK
M=&UP9&ER`'-H85]U<&1A=&4`;7E?<&%R7W!L`%]?96YD7U\`<&%R7VEN:71?
M96YV`'-H85]F:6YA;`!?7V)S<U]S=&%R=`!X<U]I;FET`&UA:6X`7U]S=&%R
M=`!P87)?9FEN9'!R;V<`<&%R7W-E='5P7VQI8G!A=&@`<&%R7V5N=E]C;&5A
M;@!P87)?9&EE`'-H85]I;FET`%A37TEN=&5R;F%L<U]005)?0D]/5``O=7-R
M+W!K9R]L:6(O<&5R;#4O-2XT,BXP+V5A<FUV-VAF+6YE=&)S9"UT:')E860M
M;75L=&DO0T]213HO=7-R+W!K9R]L:6(`$)`!`!<````4D`$`%P```""2`0`7
M````,)(!`!<```!`D@$`%P```$R2`0`7````5)(!`!<```!DD@$`%P```&R2
M`0`7````=)(!`!<```!XD@$`%P```'R2`0`7````@)(!`!<```"0D@$`%P``
M`)22`0`7````G)(!`!<```"@D@$`%P```*22`0`7````J)(!`!<```"LD@$`
M%P```+22`0`7````N)(!`!<```#`D@$`%P```"22`0`5!P``+)(!`!43```T
MD@$`%2$``#B2`0`1)@``/)(!`!(F``!(D@$`%3$``%"2`0`5-0``6)(!`!4[
M``!<D@$`%4$``&"2`0`51@``:)(!`!55``!PD@$`%64``!B1`0`6`P``')$!
M`!8$```@D0$`%@4``"21`0`6!P``*)$!`!8(```LD0$`%@D``#"1`0`6"P``
M-)$!`!8-```XD0$`%@X``#R1`0`6#P``0)$!`!81``!$D0$`%A4``$B1`0`6
M%@``3)$!`!87``!0D0$`%A@``%21`0`6&0``6)$!`!8:``!<D0$`%AP``&"1
M`0`6'0``9)$!`!8>``!HD0$`%A\``&R1`0`6(```<)$!`!8B``!TD0$`%B<`
M`'B1`0`6*```?)$!`!8I``"`D0$`%BL``(21`0`6+```B)$!`!8M``",D0$`
M%BX``)"1`0`6+P``E)$!`!8R``"8D0$`%C,``)R1`0`6-```H)$!`!8V``"D
MD0$`%C@``*B1`0`6.0``K)$!`!8Z``"PD0$`%CT``+21`0`6/@``N)$!`!9`
M``"\D0$`%D$``,"1`0`60@``Q)$!`!9#``#(D0$`%D8``,R1`0`61P``T)$!
M`!9(``#4D0$`%DH``-B1`0`63@``W)$!`!90``#@D0$`%E,``.21`0`65@``
MZ)$!`!97``#LD0$`%E@``/"1`0`66@``])$!`!9;``#XD0$`%EP``/R1`0`6
M70```)(!`!9>```$D@$`%F````B2`0`680``#)(!`!9B```0D@$`%F0``!22
M`0`6:```&)(!`!9I```<D@$`%FH```3@+>4$X)_E#N"/X`CPON60?0$``,:/
MXA?*C.*0_;SE`,:/XA?*C.*(_;SE`,:/XA?*C.*`_;SE`,:/XA?*C.)X_;SE
M`,:/XA?*C.)P_;SE`,:/XA?*C.)H_;SE`,:/XA?*C.)@_;SE`,:/XA?*C.)8
M_;SE`,:/XA?*C.)0_;SE`,:/XA?*C.)(_;SE`,:/XA?*C.)`_;SE`,:/XA?*
MC.(X_;SE`,:/XA?*C.(P_;SE`,:/XA?*C.(H_;SE`,:/XA?*C.(@_;SE`,:/
MXA?*C.(8_;SE`,:/XA?*C.(0_;SE`,:/XA?*C.((_;SE`,:/XA?*C.(`_;SE
M`,:/XA?*C.+X_+SE`,:/XA?*C.+P_+SE`,:/XA?*C.+H_+SE`,:/XA?*C.+@
M_+SE`,:/XA?*C.+8_+SE`,:/XA?*C.+0_+SE`,:/XA?*C.+(_+SE`,:/XA?*
MC.+`_+SE`,:/XA?*C.*X_+SE`,:/XA?*C.*P_+SE`,:/XA?*C.*H_+SE`,:/
MXA?*C.*@_+SE`,:/XA?*C.*8_+SE`,:/XA?*C.*0_+SE`,:/XA?*C.*(_+SE
M`,:/XA?*C.*`_+SE`,:/XA?*C.)X_+SE`,:/XA?*C.)P_+SE`,:/XA?*C.)H
M_+SE`,:/XA?*C.)@_+SE`,:/XA?*C.)8_+SE`,:/XA?*C.)0_+SE`,:/XA?*
MC.)(_+SE`,:/XA?*C.)`_+SE`,:/XA?*C.(X_+SE`,:/XA?*C.(P_+SE`,:/
MXA?*C.(H_+SE`,:/XA?*C.(@_+SE`,:/XA?*C.(8_+SE`,:/XA?*C.(0_+SE
M`,:/XA?*C.((_+SE`,:/XA?*C.(`_+SE`,:/XA?*C.+X^[SE`,:/XA?*C.+P
M^[SE`,:/XA?*C.+H^[SE`,:/XA?*C.+@^[SE`,:/XA?*C.+8^[SE`,:/XA?*
MC.+0^[SE`,:/XA?*C.+(^[SE`,:/XA?*C.+`^[SE`,:/XA?*C.*X^[SE`,:/
MXA?*C.*P^[SE`,:/XA?*C.*H^[SE`,:/XA?*C.*@^[SE`,:/XA?*C.*8^[SE
M`,:/XA?*C.*0^[SE`,:/XA?*C.*(^[SE`!"@X0(`H.$'T,WC,P``ZD0PG^5$
M()_E`S"/X'!`+>D\4)_E`F"3YP50C^`&`%7A<("]*`%@1N(`0*#C!6!&X"9A
MH.$$,)7D,_\OX08`5.$!0(3B^O__&G"`O>A4>@$`%`$``$QY`0!D()_E9#"?
MY600G^4"((_@`P"2YP$PDN<#`%#A'O\O(?!!+>D!<$/B`'!'X`A0@.(`0*#C
MIW&@X0<``.H(8!7E"%"%X@`PEN4S_R_A!P!4X0%`A.(``(;E\(&]"`0P5>6@
M`%/C]/__"BC__^OX>0$`.`$``!P!``!P@9_E`&!1XH!`+>D(@(_@"-!-XDL`
M``I<(9_E7#&?Y0(@F.<`8(+E"""6Y0-PF.<`,);E`""'Y0`@D^4``%+C.@``
M"C@1G^4!$)CG`""!Y0`@D^4`,-+E``!3XP4```H!((+B+P!3XP`@@04!,-+D
M``!3X_K__QH``%#C````"OC^_^N3___K^#"?Y0,PF.<``%/C-0``"NPPG^7L
M4)_E`Y"8YP50C^`)`%7A`9!),@6023``0*`S*9&@,00``"H$,)7D,_\OX0D`
M5.$!0(3B^O__&K@`G^6X4)_E``"/X.#^_^NP,)_E!5"/X`.`F.<(`%7A`8!(
M,@6`2#``0*`S*(&@,00``"H$,)7D,_\OX00`6.$!0(3B^O__&@`@E^4`$);E
M!`"6Y8$+`.NU_O_K3#"?Y60@G^4#,)CG`B"/X``@@^7+___J5#"?Y1/`H.,$
M`*#C`!"@XP,PC^`"(*#C`,"-Y?O^_^L!`*#CV/[_ZX3__^O'___J@'D!`&`!
M```T`0``:`$``%@!```D`0``['<!`$S^__^H=P$`0`$``$1Z`0"(,```\$<M
MZ0"04.+$,)_E`S"/X"L```J\()_E`C"3YP"`D^4``%CC*```"@!0V>4!<*#A
M``!5XST`51,!4*`3`%"@`Q$``!H`0)CE``!4XPP```H(8*#A!`"@X04@H.$)
M$*#AD_[_ZP``4.,"```:!3#4YST`4^,+```*!$"VY0``5./S__\:!`"@X?"'
MO>@)4*#A`3#UY0``4^,]`%,3^___&@E01>#F___J`5"%X@A@1N`%0(3@1F&@
MX0!@A^4$`*#A\(>]Z`E`H.'M___J"$"@X>O__^K8=P$`-`$``&`AG^5@,9_E
M`B"/X/!#+>D`4%#B`S"2YZ303>(`,)/EG#"-Y0`PH.,"```*`##5Y0``4^,*
M```:,"&?Y2@QG^4"((_@`S"2YP`@D^6<,)WE`B`SX``PH.-!```:I-"-XO"#
MO>B7_O_K`$!0XO'__PK\@)_E#7"@X0B`C^`$`*#A`O__ZP``4.,G```*#3#0
MY0U@@.(N`%/C`@``&@$PUN4``%/C]/__"@TPT.4N`%/C(@``"@4`H.'H_O_K
M`)"@X08`H.'E_O_K``")X`(`@.+H_O_K!C"@X04@H.$($*#A`&"@X=3^_^L'
M$*#A!@"@X5/^_^L!`'#C!@"@X0,```H(,)WE#SH#X@$)4^,2```*0O[_ZP8`
MH.$W_O_K!`"@X=K^_^L``%#CU___&@0`H.%>_O_K!0"@X1W^_^N]___J`3#6
MY2X`4^/9__\:`C#6Y0``4^/(__\*U?__ZJ?__^OK___J1?[_ZP!W`0`\`0``
MR'8!`#`O``"0()_ED#"?Y0(@C^`00"WIH-!-X@,PDN<`0*#A#1"@X0`PD^6<
M,(WE`#"@XR#^_^L``%#C``"@$P0``!H(,)WE#SH#X@H*4^,!"5,3"@``"D@@
MG^5`,)_E`B"/X`,PDN<`()/EG#"=Y0(@,^``,*#C!P``&J#0C>(0@+WH`A"@
MXP0`H.%M_O_K$`]OX:`"H.'N___J&_[_ZXAU`0`\`0``.'4!```0H.'X`)_E
M,$`MZ0';3>+P,)_E#-!-X@``C^`$4(WB`2N@XP,PD.<%`*#A`#"3Y00TC>4`
M,*#CV?W_ZP$P0.(#`%7A"```*@4@C>(`$*#C`@``Z@(`4.$`$,#E`@``"@$P
M<.4O`%/C^?__"B\0H.,%`*#AUOW_ZP``4.,0```*!0!0X0`PH(,`,,"%!0"@
MX5#^_^MP()_E:#"?Y0(@C^`#,)+G`""3Y00TG>4"(#/@`#"@XQ$``!H!VXWB
M#-"-XC"`O>A$()_E.#"?Y0(@C^`#,)+G`""3Y00TG>4"(#/@`#"@XP4``!HD
M`)_E``"/X`';C>(,T(WB,$"]Z#7^_^K7_?_KU'0!`#P!``!0=`$`('0!`'`L
M``#P0"WI`&"@X?@`G^4!VTWB]#"?Y0S03>(``(_@!'"-X@%0H.'_(P#C`S"0
MYP<0H.'8`)_E`#"3Y00TC>4`,*#C``"/X,_]_^L`P%#B%P``V@%`3.(!.X3B
M"#"#X@,PC>`$-%/E``!3XR,``!H``%3C#@``"@0@H.$O$*#C!P"@X07^_^L`
M`%#C"```"H``G^4``(_@)_[_ZP0PH.$'(*#A!A"@X0``D.4'_O_K!@``ZF``
MG^4``(_@'O[_ZP4@H.$&$*#A``"0Y:+]_^M(()_E-#"?Y0(@C^`#,)+G`""3
MY00TG>4"(#/@`#"@XP0``!H!VXWB#-"-XO"`O>@,0*#AV___ZI#]_^N\<P$`
M/`$``#PL``!P=`$`3'0!``!S`0#P1RWI`4"@X0`PT>4(T$WB#!*?Y0!0H.$(
M(I_E/0!3XP$0C^`!0(0"_(&?Y0(@D><$`*#A"("/X``@DN4$((WE`""@X_#]
M_^L-$*#A`&"@X04`H.&=_O_K`)!0XD$```KI_?_K``!6X3@``)J\,9_E`'"=
MY0.`F.<'<:#A`##5Y0``4^,]`%,3`3"@$P`PH`,%```*!3"@X0$@\^4``%+C
M/0!2$_O__QH%,$/@`@"&X@!@F.4#`(#@V?W_ZP``4.,'`(;G%@``"@`0G>4%
M`*#A`""8Y0$PT.0!$9+G``!3XST`4Q,`,,'E!@``"@$@@>(!,-#D`A"@X0``
M4^,]`%,3`3#"Y/G__QH!(*#A/3"@XP`PP>4!,-3D``!3XP$PXN7[__\:""&?
MY?@PG^4"((_@`S"2YP`@D^4$,)WE`B`SX``PH.,U```:"-"-XO"'O>@!D$GB
M`3#4Y```4^,!,.GE^___&N[__^K`,)_E`X"8YP!PF.4`,)?E``!3XR0```H'
M,*#A!""SY0D0H.$!D(GB``!2X_K__QH#$('B`1&@X9`PG^4#,(_@`""3Y0``
M4N,/```:`0"@X0$@H.,`((/EE_W_ZP"@4.+5__\*!Q"@X0EQH.$'(*#AVOS_
MZP"@B.4!,(GB`""@XP,ABN<`D(WEH?__Z@<`H.'/_/_K``!0XP"@H.$``(CE
MQ?__"@EQH.'R___J`Y"@X0@0H./?___J`?W_ZY1R`0`\`0``@'(!`#0!``"(
M<0$`-',!`$`@G^5`,)_E0`"?Y0(@C^!P0"WI``"/X`-@DN=N_?_K+%"?Y2Q`
MG^4%4(_@!B"@X01`C^`%,*#A!!"@X7!`O>@``)#E&?W_ZFQP`0`@`0``D'$!
M`&AI```$*0``7`"?Y3!`+>D``(_@%-!-XE!0G^59_?_K!5"/X`!`H.$``)#E
MWO__ZP`0E.4X()_E``"@XS0PG^4,$(WE,!"?Y0,PC^`"()7G`1"/X`0`C>4`
M,(WE#`"=Y1C]_^L4T(WB,("]Z$!Q`0`$<`$`;`$``'@K``"L*`````"@XQ[_
M+^%$(I_E1#*?Y?!/+>D"((_@2]Y-X@!0H.$,T$WB,`*?Y0,PDN<88(WB``"/
MX`%PH.$$$$;B`#"3Y;0TC>4`,*#CW/W_ZR\0H.,`0*#A!0"@X8+\_^L``%#C
M$```"O0!G^4%$*#A``"/X![__^L%`*#AY"&?Y=0QG^4"((_@`S"2YP`@D^6T
M-)WE`B`SX``PH.-L```:2]Z-X@S0C>+PC[WH!P"@X??\_^NP$9_E`1"/X*#\
M_^L`H%#BY___"@`PVN6<D9_EG(&?Y0``4^.8<9_E"9"/X`B`C^`'<(_@/@``
M"@``5.,$```*"A"@X00`H.$(_?_K``!0XRX```H*`*#A^_S_ZP$@0.(`,*#A
M`B"*X`"PH.$"`%KA`#"*,`$`BC(`$*`S`P``.@@``.H#`%#A`!##Y0(```H!
M('/E+P!2X_G__PH*`*#AZ?S_ZP"PH.$%`*#AYOS_ZPL`@.`!`(#B`0M0X[S_
M_RJTL(WB_!"?Y0<PH.$!*Z#C`1"/X`L`H.$$$(WE`!"@XPA0C>4`H(WE@/S_
MZP80H.$+`*#A3?S_ZP``4.,#```:(#"=Y0\Z`^(""5/C%```"K00G^4``*#C
M`1"/X%O\_^L`H%#BHO__"@`PVN4``%/CP/__&@``5.,5```*`##4Y2X`4^,"
M```*`;"@XPF@H.'4___J`3#4Y0``4^/K__\*^/__Z@$0H.,+`*#AC/S_ZP``
M4./E__\:4`"?Y0L0H.$``(_@K?[_ZPL`H.&3_/_KC/__Z@&PH.,(H*#AP?__
MZC'\_^N0;P$`/`$``&0H```T*```)&\!`/PG```\)P``."<``-0G``"0)@``
MZ"8``'`F```00"WI+Q"@XP!`H.']^__K``!0XP%`@!($`*#A$$"]Z'G\_^H0
M0"WI`$!0X@D```H`,-3E``!3XP8```J-_/_K`0"`X@$+4.,&``"*!`"@X1!`
MO>CV_?_J$`"?Y1!`O>@``(_@:/S_Z@``H.,0@+WH,"4``,@DG^7(-)_E`B"/
MX'!`+>D(T$WBO&2?Y0,PDN<-0*#AM%2?Y09@C^``,)/E!#"-Y0`PH.,%4(_@
M!!"@X08`H.$@_?_K``!0XPX```J,-)_E`!"=Y0,PE><`()/E`3&"X`0`D^4`
M`%#C`0&"Y_'__PH$,(/B!""3Y0``4N,$((/D^___&NO__^I49)_E!F"/X`00
MH.$&`*#A"OW_ZP``4.,.```*-#2?Y0`0G>4#,)7G`""3Y0$Q@N`$`)/E``!0
MXP$!@N?Q__\*!#"#X@0@D^4``%+C!""#Y/O__QKK___J`&2?Y09@C^`$$*#A
M!@"@X?3\_^L``%#C#@``"MPSG^4`$)WE`S"5YP`@D^4!,8+@!`"3Y0``4.,!
M`8+G\?__"@0P@^($()/E``!2XP0@@^3[__\:Z___ZJQCG^4&8(_@!!"@X08`
MH.'>_/_K``!0XPX```J$,Y_E`!"=Y0,PE><`()/E`3&"X`0`D^4``%#C`0&"
MY_'__PH$,(/B!""3Y0``4N,$((/D^___&NO__^I88Y_E!F"/X`00H.$&`*#A
MR/S_ZP``4.,.```*+#.?Y0`0G>4#,)7G`""3Y0$Q@N`$`)/E``!0XP$!@N?Q
M__\*!#"#X@0@D^4``%+C!""#Y/O__QKK___J!&.?Y09@C^`$$*#A!@"@X;+\
M_^L``%#C#@``"M0RG^4`$)WE`S"5YP`@D^4!,8+@!`"3Y0``4.,!`8+G\?__
M"@0P@^($()/E``!2XP0@@^3[__\:Z___ZK!BG^4&8(_@!!"@X08`H.&<_/_K
M``!0XPX```I\,I_E`!"=Y0,PE><`()/E`3&"X`0`D^4``%#C`0&"Y_'__PH$
M,(/B!""3Y0``4N,$((/D^___&NO__^I<8I_E!F"/X`00H.$&`*#AAOS_ZP``
M4.,.```*)#*?Y0`0G>4#,)7G`""3Y0$Q@N`$`)/E``!0XP$!@N?Q__\*!#"#
MX@0@D^4``%+C!""#Y/O__QKK___J"&*?Y09@C^`$$*#A!@"@X7#\_^L``%#C
M#@``"LPQG^4`$)WE`S"5YP`@D^4!,8+@!`"3Y0``4.,!`8+G\?__"@0P@^($
M()/E``!2XP0@@^3[__\:Z___ZK1AG^4&8(_@!!"@X08`H.%:_/_K``!0XPX`
M``IT,9_E`!"=Y0,PE><`()/E`3&"X`0`D^4``%#C`0&"Y_'__PH$,(/B!""3
MY0``4N,$((/D^___&NO__^I@89_E!F"/X`00H.$&`*#A1/S_ZP``4.,.```*
M'#&?Y0`0G>4#,)7G`""3Y0$Q@N`$`)/E``!0XP$!@N?Q__\*!#"#X@0@D^4`
M`%+C!""#Y/O__QKK___J#`&?Y000H.$``(_@+_S_ZP`04.("```*^`"?Y0``
MC^!V_?_K\`"?Y000H.$``(_@)OS_ZP`04.("```*W`"?Y0``C^!M_?_KU`"?
MY000H.$``(_@'?S_ZP`04.(2```*P`"?Y0``C^!D_?_KN""?Y60PG^4"((_@
M`S"2YP`@D^4$,)WE`B`SX``PH.,0```:F!"?Y9@`G^4!$(_@``"/X`C0C>)P
M0+WH5/W_ZH0`G^4$$*#A``"/X`3\_^L`$%#BZ?__"G``G^4``(_@2_W_Z^7_
M_^K4^O_KI&P!`#P!``"@)0``>&P!`#0!``!$)0``]"0``*@D``!8)```$"0`
M`%@C``!L(P``(",``-0B```$(@``+"(````B```<(@``Z"````PB``!`(0``
M0&@!`.@A``!8(0``N"$``$@A``!\()_E?#"?Y0(@C^!X`)_E!.`MY0S03>(#
M,)+G``"/X`T0H.$`,)/E!#"-Y0`PH./6^__K``!0XP4```H``-#E``!0XS``
M4!,!`*`#``"@$P$`(.(T()_E*#"?Y0(@C^`#,)+G`""3Y00PG>4"(#/@`#"@
MXP$``!H,T(WB!/"=Y)GZ_^ML9P$`/`$``+0@```89P$`3,"?Y0\`+>D,P(_@
M1!"?Y03@+>4,T$WB`1"<YQ0@C>(T,)_E-`"?Y0`0D>4$$(WE`!"@XP,PC^``
M((WE$!"=Y0``D^>P`(#BV?K_Z_\`H.-!^O_KV&8!`#P!``"L9@$`9`$``/!/
M+>D!V$WB@.&?Y1S03>)\P9_E`A"@XP[@C^``$(WE`1B-X@`@H.,,P)[G`#"@
MXQ00@>(`D*#A`,"<Y0#`@>4`P*#CT/K_ZP%@4.(`@,'BR"^@X0(P)N`",%/@
M<S#_Y@(P(^`",%/@`B#"X`-@5N`"@,C@``!8XST``+H4L9_E"*"@XQ!QG^4,
M4(WB"["/X`&@0.,'<(_@`!"@XP8@H.$(,*#A"0"@X0`0C>6V^O_K`0!QXP$`
M<`,P```*"B"@X040H.$)`*#A;?K_ZP$`<.,?```*!P!0XR```)H(`$#B`$"5
MX`,``#H<``#J`4!$X@4`5.$9```Z"""@XP<0H.$$`*#AE/K_ZP``4./V__\:
M!4!$X`8`E.#$'ZC@?,"?Y0$XC>)H()_E%#"#X@S`C^`"()SG`,"2Y0`@D^4,
MP#+@`""@XQ```!H!V(WB'-"-XO"/O>A(`)_E``"/X([__^L!:%;B`(#(X@$`
M>.,!"'8#R/__&@``X.,`$*#AYO__Z@L`H.&$___KR___ZACZ_^MD9@$`/`$`
M`,`?``!@9P$`4&4!``0?```00"WI@OK_ZWOZ_^L``%#C``"0%1"`O>CT()_E
M\$`MZ0(@C^#L8)_E%-!-XN@PG^4($(WB!F"/X`!0H.$#,)+G!@"@X0`PD^4,
M,(WE`#"@XR;[_^L`0%#B`@``"@`PU.4``%/C#@``&K`@G^6H,)_E`B"/X`,P
MDN<`()/E##"=Y0(@,^``,*#C(```&I``G^4%$*#A``"/X!30C>+P0+WH7OS_
MZ@4`H.%?^O_K`'"@X00`H.%<^O_K``"'X`(`@.)?^O_K7#"?Y5P0G^4%(*#A
M`S"/X`!`C>4!$(_@`$"@X4CZ_^M$()_E+#"?Y0(@C^`#,)+G`""3Y0PPG>4"
M(#/@`#"@XP00H`$&`*`!X?__"LWY_^NT9`$`C!X``#P!``!D9`$`+!X``!`=
M```,'0``Z&,!`#P6G^4\QI_E/":?Y0$0C^`,P(_@\$\MZ0PPC.("()'G1]]-
MX@"0H.$84(SB`""2Y10AC>4`(*#C&'"-X@<`D^@PX(WB,,",XB1`C>(48(WB
M^*6?Y0<`A^@*H(_@#P"UZ`\`KN@#`)7H`P".Z`<`G.@'`(3H!A"@X=0%G^4`
M`(_@S?K_ZP!04.("```*`##5Y0``4^,)`0`:B___ZP!04.+M```*!0"@X1+Z
M_^N``*#A`0"`XA7Z_^L`(-7E``!2XP"`H.$)```*B+6?Y0A`H.$+L(_@!`"@
MX0L0H.'\^?_K`2#UY0)`A.(``%+C^/__&F0%G^4T4(WB``"/X`(``.H$`)7D
M``!0XPT```H&$*#AJ?K_ZP!`4.+X__\*`##4Y0``4^/U__\*./O_ZP``4./R
M__\*!`"@X=/Y_^L`0%#B$P``&@!0H.-&/XWB!3&#X`!!$^4``%3C#```"@&P
MA>(+L8?@`##4Y0``4^,'```*!`"@X0%0A>(D^__K``!0XS<``!H$0)OD``!4
MX_3__QH`0*#C!`"@X=;Y_^L`4*#A"`"@X=/Y_^L`4(7@`5N%X@10A>(%`*#A
MU/G_ZYPTG^6<%)_E!""@X0,PC^``,(WED#2?Y0$0C^`$@(WE`S"/X`!PH.&Z
M^?_K!QV@XP<`H.&*^?_K`0!PXX4```I($(WB!P"@X2[Y_^L`0%#B&0``"E0$
MG^4',*#A4!2?Y0!0H.,`()GE``":YP$0C^"P`(#B+_G_ZS@DG^4,-)_E`B"/
MX`,PDN<`()/E%#&=Y0(@,^``,*#C^0``&@4`H.%'WXWB\(^]Z`0`H.&'^?_K
M`$!0XK3__PK&___J4#"=Y0\Z`^(!"5/CX?__&F2`G>63^?_K``!8X=W__QI0
M,)WE4S#HYP<-4^/9__\:!0"@X9?Y_^L&$*#A`%"@X;@#G^4``(_@/?K_ZP`0
M4.*1```*``"9Y4O\_^L`@%#BC0``"@@`H.&$^?_K`P!0XP!@H.$,``":A!.?
MY00`0.(``(C@`1"/X(7Y_^L``%#C!0``&@0`5N.[```*!F"(X`4P5N4O`%/C
MMP``"D#^_^L`8%#B!```&@@`H.$&$*#A7_G_ZP"`4.)U```:-!.?Y30#G^4!
M$(_@,).?Y0``C^`L@Y_E8OO_ZRACG^4\^?_K)".?Y20SG^4)D(_@(!.?Y0(@
MC^`#,(_@!""-Y0$0C^`'(*#A"("/X`9@C^```(WE!0"@X4SY_^L-``#J%/G_
MZP`PD.41`%/C#@``&BCY_^L!0(3B"3"@X0<@H.$($*#A4`"-Z0#`H.$%`*#A
M`,"-Y3WY_^L'':#C!0"@X0WY_^L!`'#C[/__"@<`H.&F^/_KH`*?Y040H.$`
M`(_@./O_ZP4`H.&T_O_KA___ZOCX_^L`,)#E$0!3XW7__PH`0)GE\_C_ZS@2
MG^4',*#A`%"@XP0@H.$`P)#E`0":YU@2G^6P`(#B`,"-Y0$0C^"F^/_K=?__
MZ@6PH.%&/XWB"S&#X/0`$^4``%#C(```"@&`B^((483@!A"@X0BPH.',^?_K
M`#!0X@(```H`,-/E``!3XP0``!H$`)7D``!0XQ,```H!@(CB\O__ZO7X_^L`
M4%#BZ/__"OC^_^J'_O_KW"&?Y7PQG^4"((_@`S"2YP`@D^44,9WE`B`SX``P
MH.-5```:!0"@X4??C>+P3[WHY/C_Z@T`H..H49_E`OG_ZU,@H.,%4(_@`("@
MX>S^_^H`@)GE;___Z@#^_^L`H%'B`)"@X87__[H*(%#B`&"-Y>!@C>(`,,KB
M"`"@X=KX_^L&(*#C!A"@X0@`H.&4^/_K!@!0XP(```I,`9_E``"/X-7]_^O@
M()WE0SR@XT$S1.,#`%+A<?__&K0@UN%(-03C`P!2X6W__QHR(%GBZ$"-X@`P
MRN(`$*#C"`"@X0`0C>7!^/_K*""@XP@`H.$$$*#A>_C_ZR@`4.,"```*[`"?
MY0``C^"\_?_KY`"?Y0`@H./@,)_EX!"?Y0``C^`0(<WE`S"/X`0`C>4!$(_@
M!R"@X04`H.$`0(WEM/C_ZWK__^H$8)GE``!6XT3__PH&`*#AM_C_ZP,`4.-`
M__^:G!"?Y00`0.(``(;@`1"/X+GX_^L``%#C!H"@`3C__^HT^/_KE&,!``AE
M`0`\`0``2&,!`!0<``#4'```'!L``/`;``#8&P``,!H``&0!``#4&P``9&$!
M`)0;``!<&P``(!H``*@9``"<&```"!L``*P;``#$&P``D!@````;``"0&```
M<!D``-1>`0"4&```"!D``*08``!@&0``*!8``(@8```T&```3"*?Y4PRG^4"
M((_@2`*?Y?!`+>D,T$WB`S"2YP``C^`-8*#A`#"3Y00PC>4`,*#C@?C_ZP80
MH.$`,)#E'`*?Y60@T^4``(_@92##Y2CY_^L``%#C`@``"@`PT.4Q`%/C9```
M"O#[_^OT`9_E]$&?Y0``C^!P^/_K!$"/X`,PH.,!(*#C!!"@X0!0H.$``)#E
M#/C_ZP!`4.(*```*##"4Y<0AG^4``)7E`!"3Y0(@C^#I]__K##"4Y0`0D^4(
M,)'E`043XU<``!J@`9_EH$&?Y0``C^!8^/_K!$"/X`,PH.,!(*#C!!"@X0!0
MH.$``)#E]/?_ZP!`4.()```*=`&?Y080H.$``(_@^?C_ZP`@4.(\```*##"4
MY0``E>4`$)/ES??_ZU`!G^50<9_E``"/X$'X_^M($9_E`2"@XP=PC^`!$(_@
M`%"0Y0!`H.$%`*#AZ/?_ZRPAG^4"((_@`!"@X04`H.&\]__K`#"4Y1@!G^4`
M(*#C!A"@X0``C^!E(,/EV_C_ZP``4.,(`(?E$0``"@<=H./O]__K`2"`X@`P
MH.$1`%#C`0!2$R(``(K<()_EH#"?Y0(@C^`#,)+G`""3Y00PG>4"(#/@`#"@
MXQ@``!H,T(WB\("]Z`0`E^75_?_K``!0XP@`A^7H__\:H`"?Y0``C^#"]__K
M`3#0Y0``4^.8__\*EO__ZH@PG^4,()3E`S"/X`0PD^4``)+E`!"3Y;?Y_^N^
M___J``"5Y<;W_^ND___JA??_ZUP@G^5<`)_E`B"/X```C^`&`)+I`!"1Y:OW
M_^O07`$`/`$``.A=`0#4%0``E%T!`&@7```X%P``-%T!`!07``#0%```V%P!
M`,!=`0"P%@``L!8``%`4```D6P$`(!8``.Q<`0"\7`$`\!4``!!`+>D`0*#A
MK/S_ZP``5.,``%`3$("]"``PU.4``%/C$("]"`0`H.%"^__K./O_ZQP0G^4$
M(*#C`1"/X##W_^L``%#C$("]&`0`H.$00+WHM?C_ZA04``#P3RWI4]]-XN3K
MG^4$,(WBX"N?Y1P0@.(.X(_@7,"`X@!`H.$#`*#A`B">YP`@DN5$(8WE`""@
MXP0@D>0!`%SA,B^_Y@0@@.3Z__\:_`"#X@T@H.$"$*#A!,"RY3@PD>4DX)'E
M``!2X0P0D>4.,"/@`3`CX`PP(^#C/Z#A0#""Y?/__QH@4)3IF3D'XP0`G>6"
M.D7C`&"4Y1!PE.4.$"S@`P"`X`40`>`'`(#@99&@X0X0(>`(()WEY@V`X&:!
MH.$!`(#@"1`LX`,@@N`&$`'@#B""X`P0(>`"$('@#""=Y>`=@>`#((+@#*""
MX`@@*>``(`+@8`&@X0D@(N`*((+@$*"=Y>$M@N`#H(K@":"*X`"0*.`!D`G@
M81&@X0B0*>`*D(G@%*"=Y>*=B>`#H(K@"*"*X`&`(.`"@`C@8B&@X0"`*.`"
ML"'@"H"(X!B@G>4)L`O@Z8V(X`.@BN``H(K@:0&@X0&0*^``L"+@"I")X!R@
MG>4(L`O@Z)V)X`.@BN`!H(K@:!&@X0*`*^`!L"#@"H"(X""@G>4)L`O@Z8V(
MX`.@BN`"H(K@:2&@X0"0*^`"L"'@"I")X"2@G>4(L`O@Z)V)X`.@BN``H(K@
M:`&@X0&`*^``L"+@"H"(X"B@G>4)L`O@Z8V(X`.@BN`!H(K@:1&@X0*0*^`!
ML"#@"I")X"R@G>4(L`O@Z)V)X`.@BN`"H(K@:"&@X0"`*^`"L"'@"H"(X#"@
MG>4)L`O@Z8V(X`.@BN``H(K@:0&@X0&0*^``L"+@"I")X#2@G>4(L`O@Z)V)
MX`.@BN`!H(K@:!&@X0*`*^`!L"#@"H"(X#B@G>4)L`O@Z8V(X`.@BN`"H(K@
M:2&@X0"0*^`"L"'@"I")X#R@G>4(L`O@Z)V)X`.@BN``H(K@:`&@X0&`*^``
ML"+@"H"(X$"@G>4)L`O@Z8V(X`.@BN`!H(K@:1&@X0*0*^`!L"#@"I")X$2@
MG>4(L`O@Z)V)X`.@BN`"H(K@:"&@X0"`*^`*@(C@2*"=Y>F-B.`#H(K@``"*
MX`*@(>`)H`K@:9&@X0&@*N``H(K@3`"=Y>BMBN`#`(#@`0"`X`D0(N`($`'@
M:(&@X0(0(>``$('@4`"=Y>H=@>`#,(#@"``IX`H``.`"((/@"0`@X*$[#N,"
M`(#@5""=Y=D^1N-JH:#A`R""X.$-@.`)D(+@6""=Y0,@@N`(((+@"H`HX`&`
M*.`(@(G@89&@X5P0G>7@C8C@`Q"!X`H0@>`)H"K@`*`JX&`!H.$*H(+@8""=
MY>BMBN`#((+@"2""X`"0*>`(D"G@:(&@X0F0@>!D$)WEZIV)X`,0@>``$('@
M"``@X`H`(.!JH:#A``""X&@@G>7I#8#@`R""X`@@@N`*@"C@"8`HX&F1H.$(
M@('@;!"=Y>"-B.`#$('@"A"!X`F@*N``H"K@8`&@X0J@@N!P()WEZ*V*X`,@
M@N`)((+@`)`IX`B0*>!H@:#A"9"!X'00G>7JG8G@`Q"!X``0@>`(`"#@"@`@
MX&JAH.$``(+@>""=Y>D-@.`#((+@""""X`J`*.`)@"C@:9&@X0B`@>!\$)WE
MX(V(X`,0@>`*$('@":`JX`"@*N!@`:#A"J""X(`@G>7HK8K@`R""X`D@@N``
MD"G@")`IX&B!H.$)D('@A!"=Y>J=B>!JL:#A`Q"!X``0@>`(`"#@"@`@X``@
M@N#IK8+@B""=Y0,@@N`(((+@"X`HX`D`*.!ID:#A`!"!X(P`G>7JC8'@"1`K
MX`,`@.`*$"'@"P"`X)"PG>5JH:#A`1""X`.PB^#H'8'@";"+X`J0*>`(("G@
M:(&@X0(@@.#AG8+@E""=Y0,@@N`*((+@"*`JX`$`*N"8H)WE81&@X0``B^`#
MH(K@Z0V`X`B@BN`!@"C@"8`HX&F1H.$(@(+@G""=Y>"-B."DL)WE`R""X`$@
M@N`)$"'@`!`AX&`!H.$!H(K@`!`IX.BMBN`($"'@`2""X&B!H.'J'8+@H""=
MY0,@@N`(,"#@"2""X&J1H.$#H"K@W#P+XQL_2.,*((+@`["+X`F@B.$`L(O@
MJ`"=Y>$M@N`!H`K@`P"`X&$1H.$(`(#@"8`(X`B@BN$!@(GA"Z"*X`*`".`!
ML`G@XJV*X`N`B.&PL)WE``"(X&(AH.'JC8#@K`"=Y0.PB^`#`(#@`;"+X`F0
M@.`"`('A"@``X`(0`>`!$(#A:J&@X0F0@>`*$(+AZ`V)X+20G>4($`'@:(&@
MX0.0B>`"D(G@"B`"X`(@@>$($(KA"[""X``0`>`((`K@X+V+X`(0@>&X()WE
M"1"!X+R0G>5@`:#A`R""X`J@@N`#D(G@`""(X0B0B>`+(`+@`(`(X.L=@>`(
M@(+A:[&@X0J@B.`+((#A88&@X>&MBN`!(`+@P!"=Y0,0@>``$('@"P``X```
M@N$(((OA"0"`X`H@`N`(D`O@Z@V`X`D@@N'(D)WE`1""X,0@G>5JH:#A`Y")
MX`,@@N`(D(G@"[""X`H@B.$`(`+@"H`(X.`=@>`(@(+A8`&@X0NPB.``((KA
M88&@X>&]B^`!(`+@S!"=Y0,0@>`*$('@`*`*X`J@@N$(((#A"9"*X`L@`N`(
MH`#@ZYV)X`H@@N'4H)WE`1""X-`@G>5KL:#A`Z"*X`,@@N`(H(K@``""X`L@
MB.$)(`+@"X`(X.D=@>`(@(+A:9&@X0``B.`)((OAX8V`X`$@`N!A`:#AV!"=
MY0,0@>`+$('@";`+X`NP@N$`((GA"J"+X`@@`N``L`G@Z*V*X`L@@N'@L)WE
M`1""X-P@G>5H@:#A`["+X`,@@N``L(O@"9""X`@@@.$*(`+@"```X.H=@>``
M`(+A:J&@X0F0@.`*((CA80&@X>&=B>`!(`+@Y!"=Y0,0@>`($('@"H`(X`B`
M@N$`((KA"X"(X`D@`N``L`K@Z8V(X`L@@N'LL)WE`2""X.@0G>5ID:#A`["+
MX`,0@>``L(O@"J"!X`D0@.$($`'@"0``X.@M@N```('A"J"`X&B!H.'B#8K@
M\*"=Y0@0B>$#,(K@`A`!X`DP@^!B(:#A")`)X`*@B.$)$('A`*`*X`L0@>`"
ML`C@X!V!X`N@BN%@D:#A`S"*X/0`G>7AK8/@UC$,XV(Z3.,#`(#@8;&@X0B`
M@.#X`)WE`P"`X`(`@.`)("+@`2`BX/P0G>4"@(C@"R`IX.J-B.`*("+@`Q"!
MX&JAH.$)$('@"I`KX`B0*>`"`(#@"9"!X`01G>4`(9WEZ`V`X`,0@>!H@:#A
M"A"!X`,@@N`(H"K@"R""X`"@*N#@G8G@"J""X`@AG>5@`:#AZ:V*X`,@@N`(
M((+@`(`HX`F`*.!ID:#A"("!X`P1G>7JC8C@`Q"!X``0@>`)`"#@"@`@X&JA
MH.$``(+@$"&=Y>@-@.`#((+@"2""X`J0*>`(D"G@:(&@X0F0@>`4$9WEX)V)
MX`,0@>`*$('@"*`JX`"@*N!@`:#A"J""X!@AG>7IK8K@`R""X`@@@N``@"C@
M"8`HX&F1H.$(@('@'!&=Y>J-B.`#$('@`!"!X`D`(.`*`"#@:J&@X0``@N`@
M(9WEZ`V`X`,@@N`)((+@"I`IX`B0*>!H@:#A"9"!X"01G>7@G8G@`Q"!X`H0
M@>`(H"K@`*`JX&`!H.$*H(+@*"&=Y>FMBN`#((+@""""X`"`*.`)@"C@:9&@
MX0B`@>`L$9WEZHV(X`,0@>``$('@"0`@X`H`(.!JH:#A``""X#`AG>7H#8#@
M`R""X`D@@N`*D"G@")`IX&B!H.$)$('@-)&=Y>`=@>`#D(G@"I")X`B@*N``
MH"K@8`&@X0H@@N`XH9WEX2V"X`.@BN`(H(K@`(`HX`&`*.!A$:#A")")X#R!
MG>7BG8G@`X"(X`"`B.`!`"#@`@`@X&(AH.$``(K@`G"'X.D-@.`0<(3E8,&,
MX`C`A.5`P9WE`S",X`8P@^`!,(/@`A`AX`D0(>!ID:#A`1"(X`D@(N#@'8'@
M`"`BX`(P@^!(()_E`5"%X.$=@^`X,)_E`B"/X`10A.4)X([@`!"$Y0S@A.4#
M,)+G`""3Y40QG>4"(#/@`#"@XP$``!I3WXWB\(^]Z%?T_^O$60$`/`$``"!.
M`0`00"WI8`"@X\_T_^L`(*#C`<,"XXD;"N-%QT;CS1].XP#`@.7^S`WC!!"`
MY;K(2>-V%`7C","`Y3(00>,4((#E#!"`Y?`1#N,8((#ETA-,XUP@@.40$(#E
M$("]Z/!'+>D"D*#A%#"0Y8(AH.$`4*#A`4"@X0(PD^`8()#E%#"`Y1R`A>)<
M`)#E`2""(JDN@N`8((7E``!0XPX```I`8&#B``"(X`8`6>$)8*"Q!B"@X2'T
M_^M<,)7E`S"&X%PPA>5``%/C\(>]&`4`H.$&D$G@!D"$X,G\_^L_`%GC'@``
MVD"02>(I8Z#A`7"&X@=SA.`$$*#A0$"$X@@PH.$`X)'E$!"!X@S`$>40,(/B
M"``1Y00@$>4$`%'A$.`#Y0S``^4(``/E!"`#Y?/__QH%`*#A`4"@X;'\_^L'
M`%3AZ___&@:32>`)(*#A!Q"@X0@`H.'Y\__K7)"%Y?"'O>@$<*#A]___ZO!!
M+>D<<('B%""1Y0%`H.$88)'E`%"@X=(QY><RC[_F`2"#XG\0X.,X`%+C-F^_
MY@(`A^`#$,?G0```VD`@8N(`$*#C`?3_ZP0`H.&2_/_K4`#`\BPPA.(/"D?T
M#PI#]#PPA.(/"D/T3#"$X@\'0_0$`*#A5&"$Y5B`A.6&_/_K`S#4Y00`H.$`
M,,7ELC#4X0$PQ>4`,)3E(S2@X0(PQ>4`,)3E`S#%Y0<PU.4$,,7EMC#4X04P
MQ>4$,)3E(S2@X08PQ>4$,)3E!S#%Y0LPU.4(,,7ENC#4X0DPQ>4(,)3E(S2@
MX0HPQ>4(,)3E"S#%Y0\PU.4,,,7EOC#4X0TPQ>4,,)3E(S2@X0XPQ>4,,)3E
M#S#%Y1,PU.40,,7ELC'4X1$PQ>40,)3E(S2@X1(PQ>40,)3E$S#%Y?!!O>B*
M\__J."!BX@`0H./`\__KQ___ZEPPG^400"WI`S"/X%1`G^4D(-/E!$"/X```
M4N,0@+T81""?Y0$0H.,D$,/E`C"4YP``4^,"```*,#"?Y0,`G^=@\__K*#"?
MY0,PE.<``%/C$("]"!P`G^400+WH``"/X,KS_^H<30$`%$L!`!@!``!@3`$`
M4`$``&A%```\$)_E/#"?Y0$0C^`#,(_@`"#1Y0``4N,>_R\1*""?Y0$`H.,`
M`,'E`C"3YP``4^,>_R\!%`"?Y000@>(``(_@O//_ZJ!,`0"@2@$`5`$```Q%
M``!4PI_E5#*?Y7!`+>D,P(_@(-!-XDA2G^44`(WE%`"-XA`0C>40$(WB#""-
MY0P@C>(#,)SG!5"/X``PD^4<,(WE`#"@XXCS_^L8(I_E&#*?Y0(@E><#`)7G
M`A"@X8[S_^L(,I_E`S"5YP!`T^4``%3C4@``"O@!G^4``(_@SO/_ZQ1@G>4`
M()#E!`"&X@<!<.,#8(;B1C/2Y0,P@^-&,\+E8P``B@`!H.%C\__KQ"&?Y1!`
MG>4"((_@O!&?Y0$0C^`$`(+E`#"@X0`@E.4`((#EJ"&?Y0(@C^`$((#EH"&?
MY10`G>4"()7G`0!0XPP0@^4`()+E"""#Y5(``-H$$(3B#,"#X@!!A.`$X)'D
M`0!4X03@K.7[__\:`P"`XF1!G^4`$*#C`!"-Y08@H.$`$8/G!$"/X%`1G^4,
M`)3E`1"/X&KS_^L``%#C*0``"CQ1G^488(WB!5"/X`P`E>48\__K!A"@X0!`
MH.$D`9_E``"/X#_T_^L``%#C(```"A0QG^4#,(_@#`"3Y?[R_^M%\__K!"&?
MY<0PG^4"((_@`S"2YP`@D^4<,)WE`B`SX``PH.,H```:!`"@X2#0C>)P@+WH
M"//_Z]0@G^4"((_@``!0XPP`@N4=```*WO+_Z\``G^4``(_@=//_ZP`PD.7N
M1</EG___Z@P`E.57\__KTO__Z@@`E>4``%#C`0``"H3[_^O9___JC`"?Y080
MH.$``(_@$_3_ZP``4.,(`(7ETO__"O7__^IP,)_E<`"?Y0,0E><``(_@#O/_
MZP0`H..R___J`0"@XY[R_^O<\O_K3$H!`#P!```D2@$`3`$``%P!```H`0``
M#$L!`,A+`0!`!0``)`4``$@!``!,2P$`^-C__RA+`0`\`@``_$H!`,!(`0"T
M2@$`I$D!`"P!``!$`0``^`(``'!S7W-T<FEN9W,@;6ES<VEN9PH`+W1M<"\`
M```N````4$%27U1-4$1)4@``5$5-4$1)4@!435``55-%4@````!54T523D%-
M10`````E<R\E<P```"]P<F]C+V-U<G!R;V,O97AE````1'EN84QO861E<CHZ
M8F]O=%]$>6YA3&]A9&5R`$EN=&5R;F%L<SHZ4$%2.CI"3T]4`````%!!4E]4
M14U0`````%!!4E]04D]'3D%-10`````Z````)7,E<R5S``!015),-4Q)0@``
M``!015),3$E"`%!%4DPU3U!4`````%!%4DQ)3P``4$%27TE.251)04Q)6D5$
M`%!!4E]34$%73D5$`%!!4E]#3$5!3@```%!!4E]$14)51P```%!!4E]#04-(
M10```%!!4E]'3$]"04Q?1$5"54<`````4$%27T=,3T)!3%]435!$25(```!0
M05)?1TQ/0D%,7U1%35``4$%27T=,3T)!3%]#3$5!3@`````Q````;'-E96L@
M9F%I;&5D`````')E860@9F%I;&5D`$Q$7TQ)0E)!4EE?4$%42`!365-414T`
M`"4P,G@`````)7,E<R5S)7,`````<&%R+0`````E<SH@8W)E871I;VX@;V8@
M<')I=F%T92!S=6)D:7)E8W1O<GD@)7,@9F%I;&5D("AE<G)N;STE:2D*```E
M<SH@<')I=F%T92!S=6)D:7)E8W1O<GD@)7,@:7,@=6YS869E("AP;&5A<V4@
M<F5M;W9E(&ET(&%N9"!R971R>2!Y;W5R(&]P97)A=&EO;BD*``!0051(````
M`'!A<FP`````+G!A<@````!S:&]R="!R96%D```E<R5S8V%C:&4M)7,E<P``
M)7,E<W1E;7`M)74E<P```"5S)7-T96UP+25U+25U)7,`````&````'!E<FP`
M````,````%!!4CHZ4&%C:V5R.CI615)324].`````#$N,#8S````56YA8FQE
M('1O(&-R96%T92!C86-H92!D:7)E8W1O<GD`````)7,Z(&-R96%T:6]N(&]F
M('!R:79A=&4@8V%C:&4@<W5B9&ER96-T;W)Y("5S(&9A:6QE9"`H97)R;F\]
M)6DI"@`````M90``+2T``&5V86P@)V5X96,@+W5S<B]B:6XO<&5R;"`@+5,@
M)#`@)'LQ*R(D0")])PII9B`P.R`C(&YO="!R=6YN:6YG('5N9&5R('-O;64@
M<VAE;&P*<&%C:V%G92!?7W!A<E]P;#L*;7D@*"1005)?34%'24,L("1&24Q%
M7V]F9G-E=%]S:7IE+"`D8V%C:&5?;F%M95]S:7IE*3L*0D5'24X@>PHD4$%2
M7TU!1TE#(#T@(EQN4$%2+G!M7&XB.PHD1DE,15]O9F9S971?<VEZ92`](#0[
M("`@(R!P86-K*").(BD*)&-A8VAE7VYA;65?<VEZ92`](#0P.PHD4$M:25!?
M34%'24,@/2`B4$M<,#`S7#`P-"(["GT*<W5B(&9I;F1?<&%R7VUA9VEC"GL*
M;7D@*"1F:"D@/2!`7SL*;7D@)&-H=6YK7W-I>F4@/2`V-"`J(#$P,C0["FUY
M("1B=68["FUY("1S:7IE(#T@+7,@)&9H.PIM>2`D<&]S(#T@*"1S:7IE+3$I
M("T@*"1S:7IE+3$I("4@)&-H=6YK7W-I>F4[("`@("`@(R!.3U1%.B`D<&]S
M(&ES(&$@;75L=&EP;&4@;V8@)&-H=6YK7W-I>F4*=VAI;&4@*"1P;W,@/CT@
M,"D@>PIS965K("1F:"P@)'!O<RP@,#L*<F5A9"`D9F@L("1B=68L("1C:'5N
M:U]S:7IE("L@;&5N9W1H*"1005)?34%'24,I.PII9B`H*&UY("1I(#T@<FEN
M9&5X*"1B=68L("1005)?34%'24,I*2`^/2`P*2!["G)E='5R;B`D<&]S("L@
M)&D["GT*)'!O<R`M/2`D8VAU;FM?<VEZ93L*?0IR971U<FX@+3$["GT*;7D@
M*"1P87)?=&5M<"P@)'!R;V=N86UE+"!`=&UP9FEL92P@)4UO9'5L94-A8VAE
M*3L*14Y$('L@:68@*"1%3E9[4$%27T-,14%.?2D@>PIR97%U:7)E($9I;&4Z
M.E1E;7`["G)E<75I<F4@1FEL93HZ0F%S96YA;64["G)E<75I<F4@1FEL93HZ
M4W!E8SL*;7D@)'1O<&1I<B`]($9I;&4Z.D)A<V5N86UE.CID:7)N86UE*"1P
M87)?=&5M<"D["F]U=',H<7%;4F5M;W9I;F<@9FEL97,@:6X@(B1P87)?=&5M
M<")=*3L*1FEL93HZ1FEN9#HZ9FEN9&1E<'1H*'-U8B![("@@+60@*2`_(')M
M9&ER(#H@=6YL:6YK('TL("1P87)?=&5M<"D["G)M9&ER("1P87)?=&5M<#L*
M:68@*"UD("1P87)?=&5M<"`F)B`D7D\@;F4@)TU35VEN,S(G*2!["FUY("1T
M;7`@/2!N97<@1FEL93HZ5&5M<"@*5$5-4$Q!5$4@/3X@)W1M<%A86%A8)RP*
M1$E2(#T^($9I;&4Z.D)A<V5N86UE.CID:7)N86UE*"1T;W!D:7(I+`I3549&
M25@@/3X@)RYC;60G+`I53DQ)3DL@/3X@,"P**3L*;7D@)&9I;&5N86UE(#T@
M)'1M<"T^9FEL96YA;64["G!R:6YT("1T;7`@/#PB+BXN(CL*>#TQ.R!W:&EL
M92!;(%PD>"`M;'0@,3`@73L@9&\*<FT@+7)F("<D<&%R7W1E;7`G"FEF(%L@
M7"$@+60@)R1P87)?=&5M<"<@73L@=&AE;@IB<F5A:PIF:0IS;&5E<"`Q"G@]
M8&5X<'(@7"1X("L@,6`*9&]N90IR;2`G)&9I;&5N86UE)PHN+BX*8VQO<V4@
M)'1M<#L*8VAM;V0@,#<P,"P@)&9I;&5N86UE.PIM>2`D8VUD(#T@(B1F:6QE
M;F%M92`^+V1E=B]N=6QL(#(^)C$@)B(["G-Y<W1E;2@D8VUD*3L*;W5T<RAQ
M<5M3<&%W;F5D(&)A8VMG<F]U;F0@<')O8V5S<R!T;R!P97)F;W)M(&-L96%N
M=7`Z("1F:6QE;F%M95TI.PI]"GT@?0I"14=)3B!["DEN=&5R;F%L<SHZ4$%2
M.CI"3T]4*"D@:68@9&5F:6YE9"`F26YT97)N86QS.CI005(Z.D)/3U0["F5V
M86P@>PI?<&%R7VEN:71?96YV*"D["FUY("1Q=6EE="`]("$D14Y6>U!!4E]$
M14)51WT["FUY("5S>7,@/2`H"G!A=&A?<V5P("`@(#T^("@D7D\@/7X@+UY-
M4U=I;B\@/R`G.R<@.B`G.B<I+`I?97AE("`@("`@("`]/B`H)%Y/(#U^("]>
M*#\Z35-7:6Y\3U,R?&-Y9W=I;BDO(#\@)RYE>&4G(#H@)R<I+`I?9&5L:6T@
M("`@("`]/B`H)%Y/(#U^("]>35-7:6Y\3U,R+R`_("=<7"<@.B`G+R<I+`HI
M.PI?<V5T7W!R;V=N86UE*"D["F]U=',H<7%;7"1P<F]G;F%M92`]("(D<')O
M9VYA;64B72D["E]S971?<&%R7W1E;7`H*3L*;W5T<RAQ<5M<)$5.5GM005)?
M5$5-4'T@/2`B)$5.5GM005)?5$5-4'TB72D["FUY("@D<W1A<G1?<&]S+"`D
M<W1A<G1?;V9?1DE,15]S96-T:6]N*3L*34%'24,Z('L*;&]C86P@)%-)1WM?
M7U=!4DY?7WT@/2!S=6(@>WT["G5N;&5S<R`H;W!E;B!?1D@L("<\.G)A=R<L
M("1P<F]G;F%M92D@>PIO=71S*'%Q6T-A;B=T(')E860@9G)O;2!F:6QE("(D
M<')O9VYA;64B72D[("`C(&1O;B=T('5S92`D(2!H97)E(&%S(&ET(')E<75I
M<F5S($5R<FYO+G!M"FQA<W0@34%'24,["GT*;7D@)&UA9VEC7W!O<R`](&9I
M;F1?<&%R7VUA9VEC*"I?1D@I.PII9B`H)&UA9VEC7W!O<R`\(#`I('L*;W5T
M<RAQ<5M#86XG="!F:6YD(&UA9VEC('-T<FEN9R`B)%!!4E]-04=)0R(@:6X@
M9FEL92`B)'!R;V=N86UE(ETI.PIL87-T($U!1TE#.PI]"F]U=',H(D9O=6YD
M(%!!4B!M86=I8R!A="!P;W-I=&EO;B`D;6%G:6-?<&]S(BD["G-E96L@7T9(
M+"`D;6%G:6-?<&]S("T@)$9)3$5?;V9F<V5T7W-I>F4@+2!L96YG=&@H(EPP
M0T%#2$4B*2P@,#L*<F5A9"!?1D@L("1B=68L(&QE;F=T:"@B7#!#04-(12(I
M.PII9B`H)&)U9B!N92`B7#!#04-(12(@)B8@)&)U9B!N92`B7#!#3$5!3B(I
M('L*;W5T<R@B3F\@8V%C:&4@;6%R:V5R(&9O=6YD(BD["FQA<W0@34%'24,[
M"GT*;W5T<RAQ<5M#86-H92!M87)K97(@(B1B=68B(&9O=6YD72D["G-E96L@
M7T9(+"`D;6%G:6-?<&]S("T@)$9)3$5?;V9F<V5T7W-I>F4L(#`["G)E860@
M7T9(+"`D8G5F+"`D1DE,15]O9F9S971?<VEZ93L*;7D@)&]F9G-E="`]('5N
M<&%C:R@B3B(L("1B=68I.PIO=71S*")/9F9S970@9G)O;2!S=&%R="!O9B!&
M24Q%<R!I<R`D;V9F<V5T(BD["G-E96L@7T9(+"`D;6%G:6-?<&]S("T@)$9)
M3$5?;V9F<V5T7W-I>F4@+2`D;V9F<V5T+"`P.PHD<W1A<G1?;V9?1DE,15]S
M96-T:6]N(#T@=&5L;"!?1D@["FUY("5R97%U:7)E7VQI<W0["G)E860@7T9(
M+"`D8G5F+"`T.R`@("`@("`@("`@("`@("`@("`@("`@("`@(",@<F5A9"!T
M:&4@9FER<W0@(D9)3$4B"G=H:6QE("@D8G5F(&5Q(")&24Q%(BD@>PIR96%D
M(%]&2"P@)&)U9BP@-#L*<F5A9"!?1D@L("1B=68L('5N<&%C:R@B3B(L("1B
M=68I.PIM>2`D9G5L;&YA;64@/2`D8G5F.PIO=71S*'%Q6U5N<&%C:VEN9R!&
M24Q%("(D9G5L;&YA;64B+BXN72D["FUY("1C<F,@/2`H("1F=6QL;F%M92`]
M?B!S?%XH6V$M9EQD77LX?2DO?'P@*2`_("0Q(#H@=6YD968["FUY("@D8F%S
M96YA;64L("1E>'0I(#T@*"1B=68@/7X@;7PH/SHN*B\I/R@N*BDH7"XN*BE\
M*3L*<F5A9"!?1D@L("1B=68L(#0["G)E860@7T9(+"`D8G5F+"!U;G!A8VLH
M(DXB+"`D8G5F*3L*:68@*&1E9FEN960H)&5X="D@86YD("1E>'0@(7X@+UPN
M*#\Z<&U\<&Q\:7A\86PI)"]I*2!["FUY("1F:6QE;F%M92`](%]S879E7V%S
M*"(D8W)C)&5X="(L("1B=68L(#`W-34I.PHD4$%2.CI(96%V>3HZ1G5L;$-A
M8VAE>R1F=6QL;F%M97T@/2`D9FEL96YA;64["B1005(Z.DAE879Y.CI&=6QL
M0V%C:&5[)&9I;&5N86UE?2`]("1F=6QL;F%M93L*?0IE;'-E('L*)$UO9'5L
M94-A8VAE>R1F=6QL;F%M97T@/2!["F)U9B`]/B`D8G5F+`IC<F,@/3X@)&-R
M8RP*;F%M92`]/B`D9G5L;&YA;64L"GT["B1R97%U:7)E7VQI<W1[)&9U;&QN
M86UE?2LK.PI]"G)E860@7T9(+"`D8G5F+"`T.PI]"FQO8V%L($!)3D,@/2`H
M<W5B('L*;7D@*"1S96QF+"`D;6]D=6QE*2`]($!?.PIR971U<FX@:68@<F5F
M("1M;V1U;&4@;W(@(21M;V1U;&4["FUY("1I;F9O(#T@)$UO9'5L94-A8VAE
M>R1M;V1U;&5].PIR971U<FX@=6YL97-S("1I;F9O.PID96QE=&4@)')E<75I
M<F5?;&ES='LD;6]D=6QE?3L*)$E.0WLD;6]D=6QE?2`]("(O;&]A9&5R+R1I
M;F9O+R1M;V1U;&4B.PII9B`H)$5.5GM005)?0TQ%04Y](&%N9"!D969I;F5D
M*"9)3SHZ1FEL93HZ;F5W*2D@>PIM>2`D9F@@/2!)3SHZ1FEL92T^;F5W7W1M
M<&9I;&4@;W(@9&EE(")#86XG="!C<F5A=&4@=&5M<"!F:6QE.B`D(2(["B1F
M:"T^8FEN;6]D92@I.PHD9F@M/G!R:6YT*"1I;F9O+3Y[8G5F?2D["B1F:"T^
M<V5E:R@P+"`P*3L*<F5T=7)N("1F:#L*?0IE;'-E('L*;7D@)&9I;&5N86UE
M(#T@7W-A=F5?87,H(B1I;F9O+3Y[8W)C?2YP;2(L("1I;F9O+3Y[8G5F?2D[
M"B1I;F9O+3Y[9FEL97T@/2`D9FEL96YA;64["F]P96X@;7D@)&9H+"`G/#IR
M87<G+"`D9FEL96YA;64@;W(@9&EE('%Q6T-A;B=T(')E860@(B1F:6QE;F%M
M92(Z("0A73L*<F5T=7)N("1F:#L*?0ID:64@(D)O;W1S=')A<'!I;F<@9F%I
M;&5D.B!C86XG="!F:6YD(&UO9'5L92`D;6]D=6QE(2(["GTL($!)3D,I.PIR
M97%U:7)E(%A33&]A9&5R.PIR97%U:7)E(%!!4CHZ2&5A=GD["G)E<75I<F4@
M0V%R<#HZ2&5A=GD["G)E<75I<F4@17AP;W)T97(Z.DAE879Y.PI005(Z.DAE
M879Y.CI?:6YI=%]D>6YA;&]A9&5R*"D["G)E<75I<F4@24\Z.D9I;&4["G=H
M:6QE("AM>2`D9FEL96YA;64@/2`H<V]R="!K97ES("5R97%U:7)E7VQI<W0I
M6S!=*2!["G5N;&5S<R`H)$E.0WLD9FEL96YA;65](&]R("1F:6QE;F%M92`]
M?B`O0E-$4$%.+RD@>PII9B`H)&9I;&5N86UE(#U^("]<+G!M8S\D+VDI('L*
M<F5Q=6ER92`D9FEL96YA;64["GT*96QS92!["F1O("1F:6QE;F%M92!U;FQE
M<W,@)&9I;&5N86UE(#U^("]S:71E8W5S=&]M:7IE7"YP;"0O.PI]"GT*9&5L
M971E("1R97%U:7)E7VQI<W1[)&9I;&5N86UE?3L*?0II9B`H)&)U9B!N92`D
M4$M:25!?34%'24,I('L*;W5T<RAQ<5M.;R!Z:7`@9F]U;F0@869T97(@1DE,
M12!S96-T:6]N(&EN(&9I;&4@(B1P<F]G;F%M92)=*3L*;&%S="!-04=)0R`[
M"GT*)'-T87)T7W!O<R`]("AT96QL(%]&2"D@+2`T.R`@("`@("`@("`@("`@
M("`C('-T87)T(&]F('II<`I]"FUY($!P87)?87)G<SL*;7D@*"1O=70L("1B
M=6YD;&4L("1L;V=F:"P@)&-A8VAE7VYA;64I.PID96QE=&4@)$5.5GM005)?
M05!07U)%55-%?3L@(R!S86YI=&EZ92`H4D554T4@;6%Y(&)E(&$@<V5C=7)I
M='D@<')O8FQE;2D*:68@*"$D<W1A<G1?<&]S(&]R("@D05)'5ELP72!E<2`G
M+2UP87(M;W!T:6]N<R<@)B8@<VAI9G0I*2!["FUY("5D:7-T7V-M9"`]('%W
M*`IP("`@8FQI8E]T;U]P87(*:2`@(&EN<W1A;&Q?<&%R"G4@("!U;FEN<W1A
M;&Q?<&%R"G,@("!S:6=N7W!A<@IV("`@=F5R:69Y7W!A<@HI.PII9B`H0$%2
M1U8@86YD("1!4D=66S!=(&5Q("<M+7)E=7-E)RD@>PIS:&EF="!`05)'5CL*
M)$5.5GM005)?05!07U)%55-%?2`]('-H:69T($!!4D=6.PI]"F5L<V4@>R`C
M(&YO<FUA;"!P87)L(&)E:&%V:6]U<@IM>2!`861D7W1O7VEN8SL*=VAI;&4@
M*$!!4D=6*2!["B1!4D=66S!=(#U^("]>+2A;04E-3T),8G%P:75S5'9=*2@N
M*BDO(&]R(&QA<W0["FEF("@D,2!E<2`G22<I('L*<'5S:"!`861D7W1O7VEN
M8RP@)#(["GT*96QS:68@*"0Q(&5Q("=-)RD@>PIE=F%L(")U<V4@)#(B.PI]
M"F5L<VEF("@D,2!E<2`G02<I('L*=6YS:&EF="!`<&%R7V%R9W,L("0R.PI]
M"F5L<VEF("@D,2!E<2`G3R<I('L*)&]U="`]("0R.PI]"F5L<VEF("@D,2!E
M<2`G8B<I('L*)&)U;F1L92`]("=S:71E)SL*?0IE;'-I9B`H)#$@97$@)T(G
M*2!["B1B=6YD;&4@/2`G86QL)SL*?0IE;'-I9B`H)#$@97$@)W$G*2!["B1Q
M=6EE="`](#$["GT*96QS:68@*"0Q(&5Q("=,)RD@>PIO<&5N("1L;V=F:"P@
M(CX^(BP@)#(@;W(@9&EE('%Q6T-A;B=T(&]P96X@;&]G(&9I;&4@(B0R(CH@
M)"%=.PI]"F5L<VEF("@D,2!E<2`G5"<I('L*)&-A8VAE7VYA;64@/2`D,CL*
M?0IS:&EF="A`05)'5BD["FEF("AM>2`D8VUD(#T@)&1I<W1?8VUD>R0Q?2D@
M>PID96QE=&4@)$5.5GLG4$%27U1%35`G?3L*:6YI=%]I;F,H*3L*<F5Q=6ER
M92!005(Z.D1I<W0["B9[(E!!4CHZ1&ES=#HZ)&-M9")]*"D@=6YL97-S($!!
M4D=6.PHF>R)005(Z.D1I<W0Z.B1C;60B?2@D7RD@9F]R($!!4D=6.PIE>&ET
M.PI]"GT*=6YS:&EF="!`24Y#+"!`861D7W1O7VEN8SL*?0I]"FEF("@D;W5T
M*2!["GL*<F5Q=6ER92!)3SHZ1FEL93L*<F5Q=6ER92!!<F-H:79E.CI::7`[
M"G)E<75I<F4@1&EG97-T.CI32$$["GT*;7D@)'!A<B`]('-H:69T*$!!4D=6
M*3L*;7D@)'II<#L*:68@*&1E9FEN960@)'!A<BD@>PIO<&5N(&UY("1F:"P@
M)SPZ<F%W)RP@)'!A<B!O<B!D:64@<7%;0V%N)W0@9FEN9"!P87(@9FEL92`B
M)'!A<B(Z("0A73L*8FQE<W,H)&9H+"`G24\Z.D9I;&4G*3L*)'II<"`]($%R
M8VAI=F4Z.EII<"T^;F5W.PHH("1Z:7`M/G)E861&<F]M1FEL94AA;F1L92@D
M9F@L("1P87(I(#T]($%R8VAI=F4Z.EII<#HZ05I?3TLH*2`I"F]R(&1I92!Q
M<5M%<G)O<B!R96%D:6YG('II<"!A<F-H:79E("(D<&%R(ET["GT*;7D@)6UE
M=&%?<&%R(#T@9&\@>PII9B`H)'II<"!A;F0@;7D@)&UE=&$@/2`D>FEP+3YC
M;VYT96YT<R@G345402YY;6PG*2D@>PHD;65T82`]?B!S+RXJ7G!A<CHD+R]M
M<SL*)&UE=&$@/7X@<R]>7%,N*B\O;7,["B1M971A(#U^("]>("`H6UXZ72LI
M.B`H+BLI)"]M9SL*?0I].PII9B`H9&5F:6YE9"`D<&%R*2!["F]P96X@;7D@
M)'!H+"`G/#IR87<G+"`D<&%R(&]R(&1I92!Q<5M#86XG="!R96%D('!A<B!F
M:6QE("(D<&%R(CH@)"%=.PIM>2`D8G5F.PIR96%D("1P:"P@)&)U9BP@-#L*
M9&EE('%Q6R(D<&%R(B!I<R!N;W0@82!P87(@9FEL95T@=6YL97-S("1B=68@
M97$@)%!+6DE07TU!1TE#.PIC;&]S92`D<&@["GT*0W)E871E4&%T:"@D;W5T
M*2`["FUY("1F:"`]($E/.CI&:6QE+3YN97<H"B1O=70L"DE/.CI&:6QE.CI/
M7T-214%4*"D@?"!)3SHZ1FEL93HZ3U]21%=2*"D@?"!)3SHZ1FEL93HZ3U]4
M4E5.0R@I+`HP-S<W+`HI(&]R(&1I92!Q<5M#86XG="!C<F5A=&4@9FEL92`B
M)&]U="(Z("0A73L*)&9H+3YB:6YM;V1E*"D["G-E96L@7T9(+"`P+"`P.PIM
M>2`D;&]A9&5R.PII9B`H9&5F:6YE9"`D<W1A<G1?;V9?1DE,15]S96-T:6]N
M*2!["G)E860@7T9(+"`D;&]A9&5R+"`D<W1A<G1?;V9?1DE,15]S96-T:6]N
M.PI](&5L<V4@>PIL;V-A;"`D+R`]('5N9&5F.PHD;&]A9&5R(#T@/%]&2#X[
M"GT*:68@*"$D14Y6>U!!4E]615)"051)37T@86YD("1L;V%D97(@/7X@+UXH
M/SHC(7Q<0')E;2DO*2!["G)E<75I<F4@4$%2.CI&:6QT97(Z.E!O9%-T<FEP
M.PI005(Z.D9I;'1E<CHZ4&]D4W1R:7`M/F%P<&QY*%PD;&]A9&5R+"`D,"D[
M"GT*)&9H+3YP<FEN="@D;&]A9&5R*0IO<B!D:64@<7%;17)R;W(@=W)I=&EN
M9R!L;V%D97(@=&\@(B1O=70B.B`D(5T["FEF("@D8G5N9&QE*2!["G)E<75I
M<F4@4$%2.CI(96%V>3L*4$%2.CI(96%V>3HZ7VEN:71?9'EN86QO861E<B@I
M.PII;FET7VEN8R@I.PIR97%U:7)E7VUO9'5L97,H*3L*;7D@0&EN8R`](&=R
M97`@>R`A+T)31%!!3B\@?0IG<F5P('L**"1B=6YD;&4@;F4@)W-I=&4G*2!O
M<@HH)%\@;F4@)$-O;F9I9SHZ0V]N9FEG>V%R8VAL:6)E>'!](&%N9`HD7R!N
M92`D0V]N9FEG.CI#;VYF:6=[<')I=FQI8F5X<'TI.PI]($!)3D,["G-\+RM\
M+WQG+"!S?"\D?'P@9F]R96%C:"!`:6YC.PIM>2`E9FEL97,["B1F:6QE<WLD
M7WTK*R!F;W(@0$1Y;F%,;V%D97(Z.F1L7W-H87)E9%]O8FIE8W1S+"!V86QU
M97,@)4E.0SL*;7D@)&QI8E]E>'0@/2`D0V]N9FEG.CI#;VYF:6=[;&EB7V5X
M='T[("`@("`@("`@(R!86%@@;&EB7V5X="!V<R!D;&5X="`_"FUY("5W<FET
M=&5N.PIF;W)E86-H(&UY("1K97D@*'-O<G0@:V5Y<R`E9FEL97,I('L*;7D@
M*"1F:6QE+"`D;F%M92D["FEF("AD969I;F5D*&UY("1F8R`]("1005(Z.DAE
M879Y.CI&=6QL0V%C:&5[)&ME>7TI*2!["B@D9FEL92P@)&YA;64I(#T@*"1K
M97DL("1F8RD["GT*96QS92!["F9O<F5A8V@@;7D@)&1I<B`H0&EN8RD@>PII
M9B`H)&ME>2`]?B!M?%Y<421D:7)<12\H+BHI)'QI*2!["B@D9FEL92P@)&YA
M;64I(#T@*"1K97DL("0Q*3L*;&%S=#L*?0II9B`H)&ME>2`]?B!M?%XO;&]A
M9&5R+UM>+UTK+R@N*BDD?"D@>PII9B`H;7D@)')E9B`]("1-;V1U;&5#86-H
M97LD,7TI('L**"1F:6QE+"`D;F%M92D@/2`H)')E9BP@)#$I.PIL87-T.PI]
M"FEF("@M9B`B)&1I<B\D,2(I('L**"1F:6QE+"`D;F%M92D@/2`H(B1D:7(O
M)#$B+"`D,2D["FQA<W0["GT*?0I]"GT*;F5X="!U;FQE<W,@9&5F:6YE9"`D
M;F%M93L*;F5X="!I9B`D=W)I='1E;GLD;F%M97TK*SL*;F5X="!I9B`A<F5F
M*"1F:6QE*2!A;F0@)&9I;&4@/7X@+UPN7%$D;&EB7V5X=%Q%)"]I.PIO=71S
M*'-P<FEN=&8H<7%;4&%C:VEN9R!&24Q%("(E<R(N+BY=+"!R968@)&9I;&4@
M/R`D9FEL92T^>VYA;65](#H@)&9I;&4I*3L*;7D@)&-O;G1E;G0["FEF("AR
M968H)&9I;&4I*2!["B1C;VYT96YT(#T@)&9I;&4M/GMB=69].PI]"F5L<V4@
M>PIL;V-A;"`D+R`]('5N9&5F.PIO<&5N(&UY("1T:"P@)SPZ<F%W)RP@)&9I
M;&4@;W(@9&EE('%Q6T-A;B=T(')E860@(B1F:6QE(CH@)"%=.PHD8V]N=&5N
M="`](#PD=&@^.PIC;&]S92`D=&@["E!!4CHZ1FEL=&5R.CI0;V13=')I<"T^
M87!P;'DH7"1C;VYT96YT+"`B/&5M8F5D9&5D/B\D;F%M92(I"FEF("$D14Y6
M>U!!4E]615)"051)37T@86YD("1N86UE(#U^("]<+B@_.G!M?&EX?&%L*20O
M:3L*4$%2.CI&:6QT97(Z.E!A=&-H0V]N=&5N="T^;F5W+3YA<'!L>2A<)&-O
M;G1E;G0L("1F:6QE+"`D;F%M92D["GT*)&9H+3YP<FEN="@B1DE,12(L"G!A
M8VLH)TXG+"!L96YG=&@H)&YA;64I("L@.2DL"G-P<FEN=&8H(B4P.'@O)7,B
M+"!!<F-H:79E.CI::7`Z.F-O;7!U=&5#4D,S,B@D8V]N=&5N="DL("1N86UE
M*2P*<&%C:R@G3B<L(&QE;F=T:"@D8V]N=&5N="DI+`HD8V]N=&5N="D*;W(@
M9&EE('%Q6T5R<F]R('=R:71I;F<@96UB961D960@1DE,12!T;R`B)&]U="(Z
M("0A73L*;W5T<RAQ<5M7<FET=&5N(&%S("(D;F%M92)=*3L*?0I]"FEF("@D
M>FEP*2!["B1Z:7`M/G=R:71E5&]&:6QE2&%N9&QE*"1F:"D@/3T@07)C:&EV
M93HZ6FEP.CI!6E]/2PIO<B!D:64@<7%;17)R;W(@=W)I=&EN9R!Z:7`@<&%R
M="!O9B`B)&]U=")=.PI]"FEF("@D;65T85]P87)[8VQE86Y]*2!["B1F:"T^
M<')I;G0H(EPP0TQ%04XB*3L*?0IE;'-E('L*:68@*"%D969I;F5D("1C86-H
M95]N86UE*2!["FUY("1C='@@/2!$:6=E<W0Z.E-(02T^;F5W*#$I.PIS965K
M("1F:"P@,"P@,#L*)&-T>"T^861D9FEL92@D9F@I.PIS965K("1F:"P@,"P@
M,CL*)&-A8VAE7VYA;64@/2`D8W1X+3YH97AD:6=E<W0["GT*)&-A8VAE7VYA
M;64@/2!S=6)S='(@)&-A8VAE7VYA;64L(#`L("1C86-H95]N86UE7W-I>F4[
M"B1C86-H95]N86UE("X](")<,"(@>"`H)&-A8VAE7VYA;65?<VEZ92`M(&QE
M;F=T:"`D8V%C:&5?;F%M92D["B1F:"T^<')I;G0H)&-A8VAE7VYA;64L"B)<
M,$-!0TA%(BD["GT*;7D@)&]F9G-E="`]("1F:"T^=&5L;"`M(&QE;F=T:"@D
M;&]A9&5R*3L*)&9H+3YP<FEN="AP86-K*"=.)RP@)&]F9G-E="DL"B1005)?
M34%'24,I.PHD9F@M/F-L;W-E(&]R(&1I92!Q<5M%<G)O<B!W<FET:6YG('1R
M86EL97(@;V8@(B1O=70B.B`D(5T["F-H;6]D(#`W-34L("1O=70["F5X:70[
M"GT*>PIL87-T('5N;&5S<R!D969I;F5D("1S=&%R=%]P;W,["E]F:7A?<')O
M9VYA;64H*3L*<F5Q=6ER92!005(["E!!4CHZ2&5A=GDZ.E]I;FET7V1Y;F%L
M;V%D97(H*3L*>PIR97%U:7)E($9I;&4Z.D9I;F0["G)E<75I<F4@07)C:&EV
M93HZ6FEP.PI]"FUY("1F:"`]($E/.CI&:6QE+3YN97<[("`@("`@("`@("`@
M("`@("`@("`@("`@("`@("`C($%R8VAI=F4Z.EII<"!O<&5R871E<R!O;B!A
M;B!)3SHZ2&%N9&QE"B1F:"T^9F1O<&5N*&9I;&5N;RA?1D@I+"`G<B<I(&]R
M(&1I92!Q<5MF9&]P96XH*2!F86EL960Z("0A73L*07)C:&EV93HZ6FEP.CIS
M971#:'5N:U-I>F4H+7,@7T9(*3L*;7D@)'II<"`]($%R8VAI=F4Z.EII<"T^
M;F5W.PHH)'II<"T^<F5A9$9R;VU&:6QE2&%N9&QE*"1F:"P@)'!R;V=N86UE
M*2`]/2!!<F-H:79E.CI::7`Z.D%:7T]+*"DI"F]R(&1I92!Q<5M%<G)O<B!R
M96%D:6YG('II<"!A<F-H:79E("(D<')O9VYA;64B73L*07)C:&EV93HZ6FEP
M.CIS971#:'5N:U-I>F4H-C0@*B`Q,#(T*3L*<'5S:"!`4$%2.CI,:6)#86-H
M92P@)'II<#L*)%!!4CHZ3&EB0V%C:&5[)'!R;V=N86UE?2`]("1Z:7`["F]U
M=',H(D5X=')A8W1I;F<@>FEP+BXN(BD["FEF("AD969I;F5D("1%3E9[4$%2
M7U1%35!]*2![(",@<VAO=6QD(&)E('-E="!A="!T:&ES('!O:6YT(0IF;W)E
M86-H(&UY("1M96UB97(@*"`D>FEP+3YM96UB97)S("D@>PIN97AT(&EF("1M
M96UB97(M/FES1&ER96-T;W)Y.PIM>2`D;65M8F5R7VYA;64@/2`D;65M8F5R
M+3YF:6QE3F%M93L*;F5X="!U;FQE<W,@)&UE;6)E<E]N86UE(#U^(&U["EX*
M+S]S:&QI8B\**#\Z)$-O;F9I9SHZ0V]N9FEG>V%R8VAN86UE?2\I/PHH6UXO
M72LI"B0*?7@["FUY("1E>'1R86-T7VYA;64@/2`D,3L*;7D@)&1E<W1?;F%M
M92`]($9I;&4Z.E-P96,M/F-A=&9I;&4H)$5.5GM005)?5$5-4'TL("1E>'1R
M86-T7VYA;64I.PII9B`H+68@)&1E<W1?;F%M92`F)B`M<R!?(#T]("1M96UB
M97(M/G5N8V]M<')E<W-E9%-I>F4H*2D@>PIO=71S*'%Q6U-K:7!P:6YG("(D
M;65M8F5R7VYA;64B('-I;F-E(&ET(&%L<F5A9'D@97AI<W1S(&%T("(D9&5S
M=%]N86UE(ETI.PI](&5L<V4@>PIO=71S*'%Q6T5X=')A8W1I;F<@(B1M96UB
M97)?;F%M92(@=&\@(B1D97-T7VYA;64B+BXN72D["B@D;65M8F5R+3YE>'1R
M86-T5&]&:6QE3F%M960H)&1E<W1?;F%M92D@/3T@07)C:&EV93HZ6FEP.CI!
M6E]/2R@I*0IO<B!D:64@<7%;17)R;W(@97AT<F%C=&EN9R!Z:7`@;65M8F5R
M('1O("(D9&5S=%]N86UE(ET["F-H;6]D*#`U-34L("1D97-T7VYA;64I(&EF
M("1>3R!E<2`B:'!U>"(["GT*?0I]"GT*=6YL97-S("@D4$%2.CI,:6)#86-H
M97LD<')O9VYA;65]*2!["F1I92`\/"`B+B(@=6YL97-S($!!4D=6.PI5<V%G
M93H@)#`@6R`M06QI8BYP87(@72!;("U)9&ER(%T@6R`M36UO9'5L92!=(%L@
M<W)C+G!A<B!=(%L@<')O9W)A;2YP;"!="B0P(%L@+4)\+6(@72!;+4]O=71F
M:6QE72!S<F,N<&%R"BX*)$5.5GM005)?4%)/1TY!345](#T@)'!R;V=N86UE
M(#T@)#`@/2!S:&EF="A`05)'5BD["GT*<W5B($-R96%T95!A=&@@>PIM>2`H
M)&YA;64I(#T@0%\["G)E<75I<F4@1FEL93HZ0F%S96YA;64["FUY("@D8F%S
M96YA;64L("1P871H+"`D97AT*2`]($9I;&4Z.D)A<V5N86UE.CIF:6QE<&%R
M<V4H)&YA;64L("@G7"XN*B<I*3L*<F5Q=6ER92!&:6QE.CI0871H.PI&:6QE
M.CI0871H.CIM:W!A=&@H)'!A=&@I('5N;&5S<R@M92`D<&%T:"D[(",@;6MP
M871H(&1I97,@=VET:"!E<G)O<@I]"G-U8B!R97%U:7)E7VUO9'5L97,@>PIR
M97%U:7)E(&QI8CL*<F5Q=6ER92!$>6YA3&]A9&5R.PIR97%U:7)E(&EN=&5G
M97(["G)E<75I<F4@<W1R:6-T.PIR97%U:7)E('=A<FYI;F=S.PIR97%U:7)E
M('9A<G,["G)E<75I<F4@0V%R<#L*<F5Q=6ER92!#87)P.CI(96%V>3L*<F5Q
M=6ER92!%<G)N;SL*<F5Q=6ER92!%>'!O<G1E<CHZ2&5A=GD["G)E<75I<F4@
M17AP;W)T97(["G)E<75I<F4@1F-N=&P["G)E<75I<F4@1FEL93HZ5&5M<#L*
M<F5Q=6ER92!&:6QE.CI3<&5C.PIR97%U:7)E(%A33&]A9&5R.PIR97%U:7)E
M($-O;F9I9SL*<F5Q=6ER92!)3SHZ2&%N9&QE.PIR97%U:7)E($E/.CI&:6QE
M.PIR97%U:7)E($-O;7!R97-S.CI:;&EB.PIR97%U:7)E($%R8VAI=F4Z.EII
M<#L*<F5Q=6ER92!$:6=E<W0Z.E-(03L*<F5Q=6ER92!005(["G)E<75I<F4@
M4$%2.CI(96%V>3L*<F5Q=6ER92!005(Z.D1I<W0["G)E<75I<F4@4$%2.CI&
M:6QT97(Z.E!O9%-T<FEP.PIR97%U:7)E(%!!4CHZ1FEL=&5R.CI0871C:$-O
M;G1E;G0["G)E<75I<F4@871T<FEB=71E<SL*979A;"![(')E<75I<F4@0W=D
M('T["F5V86P@>R!R97%U:7)E(%=I;C,R('T["F5V86P@>R!R97%U:7)E(%-C
M86QA<CHZ571I;"!].PIE=F%L('L@<F5Q=6ER92!!<F-H:79E.CI5;GII<#HZ
M0G5R<W0@?3L*979A;"![(')E<75I<F4@5&EE.CI(87-H.CI.86UE9$-A<'1U
M<F4@?3L*979A;"![(')E<75I<F4@4&5R;$E/.R!R97%U:7)E(%!E<FQ)3SHZ
M<V-A;&%R('T["F5V86P@>R!R97%U:7)E('5T9C@@?3L*?0IS=6(@7W-E=%]P
M87)?=&5M<"!["FEF("AD969I;F5D("1%3E9[4$%27U1%35!](&%N9"`D14Y6
M>U!!4E]414U0?2`]?B`O*"XK*2\I('L*)'!A<E]T96UP(#T@)#$["G)E='5R
M;CL*?0IF;W)E86-H(&UY("1P871H("@**&UA<"`D14Y6>R1??2P@<7<H(%!!
M4E]435!$25(@5$U01$E2(%1%35!$25(@5$5-4"!435`@*2DL"G%W*"!#.EQ<
M5$5-4"`O=&UP("X@*0HI('L*;F5X="!U;FQE<W,@9&5F:6YE9"`D<&%T:"!A
M;F0@+60@)'!A=&@@86YD("UW("1P871H.PIM>2`D=7-E<FYA;64["FUY("1P
M=W5I9#L*979A;"![*"1P=W5I9"D@/2!G971P=W5I9"@D/BD@:68@9&5F:6YE
M9"`D/CM].PII9B`H(&1E9FEN960H)E=I;C,R.CI,;V=I;DYA;64I("D@>PHD
M=7-E<FYA;64@/2`F5VEN,S(Z.DQO9VEN3F%M93L*?0IE;'-I9B`H9&5F:6YE
M9"`D<'=U:60I('L*)'5S97)N86UE(#T@)'!W=6ED.PI]"F5L<V4@>PHD=7-E
M<FYA;64@/2`D14Y6>U5315).04U%?2!\?"`D14Y6>U5315)]('Q\("=365-4
M14TG.PI]"FUY("1S=&UP9&ER(#T@(B1P871H)'-Y<WM?9&5L:6U]<&%R+2(N
M=6YP86-K*")(*B(L("1U<V5R;F%M92D["FUK9&ER("1S=&UP9&ER+"`P-S4U
M.PIM>2`D8V%C:&5?9&ER.PII9B`H)$5.5GM005)?0TQ%04Y]*2!["B1C86-H
M95]D:7(@/2`B=&5M<"TD)"(["GT*96QS92!["F]P96X@;7D@)&9H+"`B/#IR
M87<B+"`D<')O9VYA;64@;W(@9&EE('%Q6T-A;B=T(')E860@(B1P<F]G;F%M
M92(Z("0A73L*:68@*"AM>2`D;6%G:6-?<&]S(#T@9FEN9%]P87)?;6%G:6,H
M)&9H*2D@/CT@,"D@>PIS965K("1F:"P@)&UA9VEC7W!O<PHM("1&24Q%7V]F
M9G-E=%]S:7IE"BT@;&5N9W1H*")<,$-!0TA%(BDL(#`["FUY("1B=68["G)E
M860@)&9H+"`D8G5F+"!L96YG=&@H(EPP0T%#2$4B*3L*:68@*"1B=68@97$@
M(EPP0T%#2$4B*2!["G-E96L@)&9H+"`D;6%G:6-?<&]S"BT@)$9)3$5?;V9F
M<V5T7W-I>F4*+2!L96YG=&@H(EPP0T%#2$4B*0HM("1C86-H95]N86UE7W-I
M>F4L(#`["G)E860@)&9H+"`D8G5F+"`D8V%C:&5?;F%M95]S:7IE.PHD8G5F
M(#U^(',O7#`O+V<["B1C86-H95]D:7(@/2`B8V%C:&4M)&)U9B(["GT*?0IC
M;&]S92`D9F@["GT*:68@*"$D8V%C:&5?9&ER*2!["B1C86-H95]D:7(@/2`B
M=&5M<"TD)"(["B1%3E9[4$%27T-,14%.?2`](#$["GT*)'-T;7!D:7(@+CT@
M(B1S>7-[7V1E;&EM?21C86-H95]D:7(B.PIM:V1I<B`D<W1M<&1I<BP@,#<U
M-3L*)$5.5GM005)?5$5-4'T@/2`D<W1M<&1I<CL*;&%S=#L*?0HD<&%R7W1E
M;7`@/2`D,2!I9B`D14Y6>U!!4E]414U0?2!A;F0@)$5.5GM005)?5$5-4'T@
M/7X@+R@N*RDO.PI]"G-U8B!?<V%V95]A<R!["FUY("@D;F%M92P@)&-O;G1E
M;G1S+"`D;6]D92D@/2!`7SL*;7D@)&9U;&QN86UE(#T@(B1P87)?=&5M<"\D
M;F%M92(["G5N;&5S<R`H+64@)&9U;&QN86UE*2!["FUY("1T96UP;F%M92`]
M("(D9G5L;&YA;64N)"0B.PIM>2`D9F@["BAO<&5N("1F:"P@)SXZ<F%W)RP@
M)'1E;7!N86UE*0HF)B`H<')I;G0@)&9H("1C;VYT96YT<RD*)B8@*&-L;W-E
M("1F:"D*;W(@9&EE('%Q6T5R<F]R('=R:71I;F<@(B1T96UP;F%M92(Z("0A
M73L*8VAM;V0@)&UO9&4L("1T96UP;F%M92!I9B!D969I;F5D("1M;V1E.PIR
M96YA;64H)'1E;7!N86UE+"`D9G5L;&YA;64I(&]R('5N;&EN:R@D=&5M<&YA
M;64I.PI]"G)E='5R;B`D9G5L;&YA;64["GT*<W5B(%]S971?<')O9VYA;64@
M>PII9B`H9&5F:6YE9"`D14Y6>U!!4E]04D]'3D%-17T@86YD("1%3E9[4$%2
M7U!23T=.04U%?2`]?B`O*"XK*2\I('L*)'!R;V=N86UE(#T@)#$["GT*)'!R
M;V=N86UE('Q\/2`D,#L*:68@*"1%3E9[4$%27U1%35!](&%N9"!I;F1E>"@D
M<')O9VYA;64L("1%3E9[4$%27U1%35!]*2`^/2`P*2!["B1P<F]G;F%M92`]
M('-U8G-T<B@D<')O9VYA;64L(')I;F1E>"@D<')O9VYA;64L("1S>7-[7V1E
M;&EM?2D@*R`Q*3L*?0II9B`H(21%3E9[4$%27U!23T=.04U%?2!O<B!I;F1E
M>"@D<')O9VYA;64L("1S>7-[7V1E;&EM?2D@/CT@,"D@>PII9B`H;W!E;B!M
M>2`D9F@L("<\)RP@)'!R;V=N86UE*2!["G)E='5R;B!I9B`M<R`D9F@["GT*
M:68@*"US("(D<')O9VYA;64D<WES>U]E>&5](BD@>PHD<')O9VYA;64@+CT@
M)'-Y<WM?97AE?3L*<F5T=7)N.PI]"GT*9F]R96%C:"!M>2`D9&ER("AS<&QI
M="`O7%$D<WES>W!A=&A?<V5P?5Q%+RP@)$5.5GM0051(?2D@>PIN97AT(&EF
M(&5X:7-T<R`D14Y6>U!!4E]414U0?2!A;F0@)&1I<B!E<2`D14Y6>U!!4E]4
M14U0?3L*)&1I<B`]?B!S+UQ1)'-Y<WM?9&5L:6U]7$4D+R\["B@H)'!R;V=N
M86UE(#T@(B1D:7(D<WES>U]D96QI;7TD<')O9VYA;64D<WES>U]E>&5](BDL
M(&QA<W0I"FEF("US("(D9&ER)'-Y<WM?9&5L:6U])'!R;V=N86UE)'-Y<WM?
M97AE?2(["B@H)'!R;V=N86UE(#T@(B1D:7(D<WES>U]D96QI;7TD<')O9VYA
M;64B*2P@;&%S="D*:68@+7,@(B1D:7(D<WES>U]D96QI;7TD<')O9VYA;64B
M.PI]"GT*<W5B(%]F:7A?<')O9VYA;64@>PHD,"`]("1P<F]G;F%M92!\?#T@
M)$5.5GM005)?4%)/1TY!345].PII9B`H:6YD97@H)'!R;V=N86UE+"`D<WES
M>U]D96QI;7TI(#P@,"D@>PHD<')O9VYA;64@/2`B+B1S>7-[7V1E;&EM?21P
M<F]G;F%M92(["GT*;7D@)'!W9"`]("AD969I;F5D("9#=V0Z.F=E=&-W9"D@
M/R!#=V0Z.F=E=&-W9"@I"CH@*"AD969I;F5D("97:6XS,CHZ1V5T0W=D*2`_
M(%=I;C,R.CI'971#=V0H*2`Z(&!P=V1@*3L*8VAO;7`H)'!W9"D["B1P<F]G
M;F%M92`]?B!S+UXH/SU<+EPN/UQ1)'-Y<WM?9&5L:6U]7$4I+R1P=V0D<WES
M>U]D96QI;7TO.PHD14Y6>U!!4E]04D]'3D%-17T@/2`D<')O9VYA;64["GT*
M<W5B(%]P87)?:6YI=%]E;G8@>PII9B`H("1%3E9[4$%27TE.251)04Q)6D5$
M?2LK(#T](#$@*2!["G)E='5R;CL*?2!E;'-E('L*)$5.5GM005)?24Y)5$E!
M3$E:141](#T@,CL*?0IF;W(@*'%W*"!34$%73D5$(%1%35`@0TQ%04X@1$5"
M54<@0T%#2$4@4%)/1TY!344@*2`I('L*9&5L971E("1%3E9[(E!!4E\D7R)]
M.PI]"F9O<B`H<7<O(%1-4$1)4B!414U0($-,14%.($1%0E5'("\I('L*)$5.
M5GLB4$%27R1?(GT@/2`D14Y6>R)005)?1TQ/0D%,7R1?(GT@:68@97AI<W1S
M("1%3E9[(E!!4E]'3$]"04Q?)%\B?3L*?0II9B`H)$5.5GM005)?5$5-4'TI
M('L*9&5L971E("1%3E9[4$%27T-,14%.?3L*?0I]"G-U8B!O=71S('L*<F5T
M=7)N(&EF("1Q=6EE=#L*:68@*"1L;V=F:"D@>PIP<FEN="`D;&]G9F@@(D!?
M7&XB.PI]"F5L<V4@>PIP<FEN="`B0%]<;B(["GT*?0IS=6(@:6YI=%]I;F,@
M>PIR97%U:7)E($-O;F9I9SL*<'5S:"!`24Y#+"!G<F5P(&1E9FEN960L(&UA
M<"`D0V]N9FEG.CI#;VYF:6=[)%]]+"!Q=R@*87)C:&QI8F5X<"!P<FEV;&EB
M97AP('-I=&5A<F-H97AP('-I=&5L:6)E>'`*=F5N9&]R87)C:&5X<"!V96YD
M;W)L:6)E>'`**3L*?0IP86-K86=E(&UA:6X["G)E<75I<F4@4$%2.PIU;G-H
M:69T($!)3D,L(%PF4$%2.CIF:6YD7W!A<CL*4$%2+3YI;7!O<G0H0'!A<E]A
M<F=S*3L*9&EE('%Q6W!A<BYP;#H@0V%N)W0@;W!E;B!P97)L('-C<FEP="`B
M)'!R;V=N86UE(CH@3F\@<W5C:"!F:6QE(&]R(&1I<F5C=&]R>5QN70IU;FQE
M<W,@+64@)'!R;V=N86UE.PID;R`D<')O9VYA;64["D-/4D4Z.F5X:70H)#$I
M(&EF("@D0"`]?B]>7U1+7T58251?7"@H7&0K*5PI+RD["F1I92`D0"!I9B`D
M0#L*?3L*)#HZ7U]%4E)/4B`]("1`(&EF("1`.PI]"D-/4D4Z.F5X:70H)#$I
M(&EF("@D.CI?7T524D]2(#U^+UY?5$M?15A)5%]<*"A<9"LI7"DO*3L*9&EE
M("0Z.E]?15)23U(@:68@)#HZ7U]%4E)/4CL*,3L*7U]%3D1?7PH`<&5R;'AS
M:2YC`````$-!0TA%```!&P,[``$``!\```!PC/__&`$``,B,__\T`0``0(W_
M_U0!``#HCO__I`$``,2/___(`0``/)'___0!``#@D?__&`(``/B2__]0`@``
M&)3__X`"``!4EO__L`(``+"6___4`@``*)?___0"```PE___"`,``*R9__]`
M`P``T)G__V`#```@FO__B`,``%B?__^X`P``[)___]@#``!0H/___`,``/BA
M__\T!```$*+__TP$```LH___?`0``/2I___$!```F*S__^P$``#PK/__-`4`
M`/"X__]H!0``3+G__X`%``!4NO__I`4``*2[__^$`0``(+S__W`!``!TO/__
M#`4``!```````````7I2``)\#@$;#`T`&````!@```!0B___6`````!(#A"$
M!(4#A@*.`1P````T````C(O__W@`````4@X8A`:%!88$AP.(`HX!&````%0`
M``#DB___J`$```!&#@B'`HX!1`X0`!````!P````J+K__U0`````````'```
M`(0````8NO__?`````!$#@B$`HX!:L[$#@`````@````I````#R-___<````
M`$(.((0(A0>&!H<%B`2)`XH"C@$H````R````/2-__]X`0```$@.'(0'A0:&
M!8<$B`.)`HX!1@[``6(*#AQ""R````#T````0(___Z0`````2`X(A`*.`4(.
MJ`%R"@X(0@L``#0````8`0``P(___Q@!````1@X,A`.%`HX!0@Z,"$0.F`@"
M5`H.#$(+6@H.#$+.Q<0.`$(+````+````%`!``"@D/__(`$```!"#A2$!84$
MA@.'`HX!1@Z4"$0.H`@"<`H.%$(+````+````(`!``"0D?__/`(```!"#B"$
M"(4'A@:'!8@$B0.*`HX!1@XH`IX*#B!""P``(````+`!``"<D___7`````!*
M#A"$!(4#A@*.`5;.QL7$#@``'````-0!``#4D___>`````!$#@R$`X4"C@%$
M#B!H#@P0````]`$``"R4__\(`````````#0````(`@``()3__WP"````1@XD
MA`F%"(8'AP:(!8D$B@.+`HX!1`[4"40.X`D"0`H.)$(+````'````$`"``!D
MEO__)`````!"#@B$`HX!3L[$#@`````D````8`(``&B6__]0`````$(."(0"
MC@%6"L[$#@!""T0*SL0.`$0++````(@"``"0EO__.`4```!(#A"$!(4#A@*.
M`4(.&`-$`@H.$$+.QL7$#@!""P``'````+@"``"8F___E`````!*#@2.`4(.
M$'(*#@1""P`@````V`(```R<__]D`````$0.$(`$@0."`H,!1@X4C@5"#B`T
M````_`(``$R<__^H`0```$(.)(0)A0B&!X<&B`6)!(H#BP*.`4(.I(`$1`[`
M@`0"H`H.)$(+`!0````T`P``O)W__Q@`````0@X(A`*.`2P```!,`P``O)W_
M_QP!````1`X4A`6%!(8#AP*.`48.*'@*#A1"SL?&Q<0.`$(+`$0```!\`P``
MJ)[__\@&````3`XDA`F%"(8'AP:(!8D$B@.+`HX!1@[``@,>`0H.)$(+`T8!
M"@XD0L[+RLG(Q\;%Q`X`0@L``"0```#$`P``**7__Z0"````2@X4A`6%!(8#
MAP*.`4(.(`+>"@X40@L<````[`,``*2G__]8`````$(."(0"C@%FSL0.````
M`"0````,!```8+?__[0"````1@X0A`2%`X8"C@%$#C`"U`H.$$(+```P````
M-`0``+2G__\`#````$(.)(0)A0B&!X<&B`6)!(H#BP*.`4(.\`(#\@4*#B1"
M"P``%````&@$``"`L___7`````!"#@B$`HX!(````(`$``#$L___"`$```!"
M#B"$"(4'A@:'!8@$B0.*`HX!+````*0$``"HM/__4`$```!"#AB$!H4%A@2'
M`X@"C@$"G`K.R,?&Q<0.`$(+`````````%A&``#<10````````$````!````
M`0```,0"```!````S@(```$```#C`@```0````(#```/````FP0``!D````0
MD`$`&P````0````:````%)`!`!P````$````!````'@!```%````8`L```8`
M``"P!```"@```.0$```+````$````!4``````````P````R1`0`"````$`(`
M`!0````1````%P```%P1```1````1!```!(````8`0``$P````@```#[__]O
M````"/K__V\7````````````````````````````````````````````````
M`````````!R0`0```````````&P3``!L$P``;!,``&P3``!L$P``;!,``&P3
M``!L$P``;!,``&P3``!L$P``;!,``&P3``!L$P``;!,``&P3``!L$P``;!,`
M`&P3``!L$P``;!,``&P3``!L$P``;!,``&P3``!L$P``;!,``&P3``!L$P``
M;!,``&P3``!L$P``;!,``&P3``!L$P``;!,``&P3``!L$P``;!,``&P3``!L
M$P``;!,``&P3``!L$P``;!,``&P3``!L$P``;!,``&P3``!L$P``;!,``&P3
M``!L$P``;!,``&P3``!L$P``;!,``&P3``!L$P``;!,``&P3``!L$P``;!,`
M`&P3``!L$P``;!,``!B0`0`````````````````0D`$`````````````````
M0),!````````````%)`!``````#`D@$`````````````````')`!```````$
MDP$``````'R2`0`L-````),!`("2`0`*4$%2+G!M"@````!T20``?$D`````
M``"`20``A$D``(Q)````2@``E$D```````"820``H$D```````"T3```1T-#
M.B`H3F5T0E-$(&YB,R`R,#(S,3`P."D@,3`N-2XP`$=#0SH@*&YB,R`R,#(S
M,3`P."D@,3`N-2XP`"1.971"4T0Z(&-R=#`N4RQV(#$N-"`R,#$X+S$Q+S(V
M(#$W.C,W.C0T(&IO97)G($5X<"`D`"1.971"4T0Z(&-R=#`M8V]M;6]N+F,L
M=B`Q+C(W(#(P,C(O,#8O,C$@,#8Z-3(Z,3<@<VMR;&P@17AP("0`)$YE=$)3
M1#H@8W)T:2Y3+'8@,2XQ(#(P,3(O,#@O,3,@,#(Z-#DZ,#0@;6%T="!%>'`@
M)``D3F5T0E-$.B!C<G1B96=I;BYC+'8@,2XQ-R`R,#$X+S$R+S(X(#$X.C$W
M.C$Q(&-H<FES=&]S($5X<"`D`"1.971"4T0Z(&-R=&5N9"Y3+'8@,2XR(#(P
M,3,O,3(O,C`@,34Z,S4Z-#@@:F]E<F<@17AP("0`\<\"````````````````
M``@````(````"`````@````;`````0``````````*BHJ($YO($QA8F5L(%!R
M;W9I9&5D("HJ*@``\<\"``````````````````@````(````"`````@````;
M`````0``````````*BHJ($YO($QA8F5L(%!R;W9I9&5D("HJ*@!!,@```&%E
M86)I``$H````!3<M00`&"@=!"`$)`@H##`$2!!0!%0$7`Q@!&@(<`2(!1`$`
M+G-H<W1R=&%B`"YI;G1E<G``+FYO=&4N;F5T8G-D+FED96YT`"YN;W1E+FYE
M=&)S9"YP87@`+FYO=&4N;F5T8G-D+FUA<F-H`"YH87-H`"YD>6YS>6T`+F1Y
M;G-T<@`N<F5L+F1Y;@`N<F5L+G!L=``N=&5X=``N<F]D871A`"YE:%]F<F%M
M95]H9'(`+F5H7V9R86UE`"YI;FET7V%R<F%Y`"YF:6YI7V%R<F%Y`"YJ8W(`
M+F1Y;F%M:6,`+F=O=``N9&%T80`N8G-S`"YC;VUM96YT`"Y354Y77V-T9@`N
M05)-+F%T=')I8G5T97,`+FYO:6YI=```````````````````````````````
M``````````````````````````L````!`````@```!0!```4`0``%P``````
M`````````0`````````3````!P````(````L`0``+`$``!@`````````````
M``0`````````)@````<````"````1`$``$0!```4```````````````$````
M`````#<````'`````@```%@!``!8`0``(```````````````!`````````!*
M````!0````(```!X`0``>`$``#@#```&``````````0````$````4`````L`
M```"````L`0``+`$``"P!@``!P````,````$````$````%@````#`````@``
M`&`+``!@"P``Y`0``````````````0````````!@````"0````(```!$$```
M1!```!@!```&``````````0````(````:0````D```!"````7!$``%P1```0
M`@``!@```!,````$````"````&T````!````!@```&P3``!L$P``+`,`````
M````````!`````0```!R`````0````8```"8%@``F!8``,@R````````````
M``0`````````>`````$````"````8$D``&!)``#70``````````````$````
M`````(`````!`````@```#B*```XB@``!`$`````````````!`````````".
M`````0````(````\BP``/(L``-0$``````````````0`````````F`````X`
M```#````$)`!`!"0```$```````````````$````!````*0````/`````P``
M`!20`0`4D```!```````````````!`````0```"P`````0````,````8D`$`
M&)````0```````````````0`````````M0````8````#````')`!`!R0``#P
M````!P`````````$````"````+X````!`````P````R1`0`,D0``<`$`````
M````````!`````0```##`````0````,```!\D@$`?)(``$@`````````````
M``0`````````R0````@````#`````),!`,22``!$``````````````!`````
M`````,X````!````,`````````#$D@``/0```````````````0````$````?
M`````0```#```````````9,``!\!``````````````$````!````UP````$`
M`````````````""4``"/```````````````$`````````.$````#``!P````
M``````"OE```,P```````````````0````````#Q`````0````$```!$DP$`
MXI0```````````````````$``````````0````,``````````````.*4``#Y
3```````````````!````````````
