package PAR::StrippedPARL::Dynamic;
use 5.008001;
use strict;
use warnings;
our $VERSION = '0.958';

use base 'PAR::StrippedPARL::Base';

our $Data_Pos = tell DATA;

=head1 NAME

PAR::StrippedPARL::Dynamic - Data package containing a dynamic PARL

=head1 SYNOPSIS

  # For details, see PAR::StrippedPARL::Base.
  PAR::StrippedPARL::Dynamic->write_parl($file) or die "Some error...";

=head1 DESCRIPTION

This class is internal to PAR. Do not use it outside of PAR.

This class is basically just a container for a dynamic binary PAR loader
which doesn't include the PAR code like the F<parl> or F<parl.exe>
you are used to. If you're really curious, I'll tell you it is
just a copy of the F<myldr/par> (or F<myldr/par.exe>) file.

The data is appended during the C<make> phase of the PAR build process,
but only if applicable: If you perl is static, you won't get the dynamic
loader.

If the binary data isn't appended during the build process, the class
methods will return the empty list.

=head1 CLASS METHODS

Inherits the methods from L<PAR::StrippedPARL::Base>.

=cut

sub _data_pos {
    my $class = shift;
    return $Data_Pos;
}

=head1 AUTHORS

Steffen Mueller E<lt>smueller@cpan.orgE<gt>,
Audrey Tang E<lt>cpan@audreyt.orgE<gt>

=head1 COPYRIGHT AND LICENSE

Copyright 2006-2009 by Steffen Mueller E<lt>smueller@cpan.orgE<gt>.

This program is free software; you can redistribute it and/or 
modify it under the same terms as Perl itself.

See F<LICENSE>.

=cut

$^V eq 5.40.2 
    or die sprintf("Perl (%s) version (%vd) doesn't match the version (5.40.2) ".
                   "that PAR::Packer was built with; please rebuild PAR::Packer", 
                   $^X, $^V);

1;

__DATA__
M?T5,1@$!`0````````````,`*``!````J!8``#0````XE`````0`!30`(``'
M`"@`&P`:``8````T````-````#0```#@````X`````0````$`````P```!0!
M```4`0``%`$``!<````7````!`````$````!````````````````````;(X`
M`&R.```%```````!``$```!LC@``;(X!`&R.`0"T`@``&`,```8```````$`
M`@```'B.``!XC@$`>(X!`/````#P````!@````0````$````+`$``"P!```L
M`0``3````$P````$````!````%#E=&1XAP``>(<``'B'```,`0``#`$```0`
M```$````+W5S<B]L:6)E>&5C+VQD+F5L9E]S;P``!P````0````!````3F5T
M0E-$````\N$U!`````0````#````4&%8```````'````"0````4```!.971"
M4T0``&5A<FUV-VAF`````&$```!K```````````````P````#0``````````
M````.`````X```!``````````"\``````````P```$X```!C`````````!\`
M``!-````!@```!T````4`````````$,```!8````7P`````````7````/```
M`%H```!D`````````&<```!I````+@```!(```!9````10```!4```!>````
M1P````````!*````1@```#4```!=````)0```"D`````````!P```$\`````
M````"P```"L```!J````7````$P````@````4````&@`````````80``````
M```8````+0```$$````````````````````1````,@``````````````8@``
M`"H`````````"0`````````C````.0````````!1````!0```$@````;````
M.P````````````````````````!+`````````#H```!4````90````````!F
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````$P``````````````
M````````````````````````````````(@```"$`````````*``````````*
M````)@```"<````L```````````````,`````````!P````:````-````!D`
M`````````````!``````````,P```#$````W````'@``````````````````
M````````````!````````````````````$(````D````````````````````
M%@`````````_````1`````@````````````````````^`````````%,`````
M`````````%<``````````````#T```!6`````````%(`````````20````\`
M`````````````%4```!;````-@```&``````````````````````````````
M``````"H%@````````,`"P``````=(X!```````#`!$`D@``````````````
M$@````P``````````````!(````4```````````````2````<P,``/PA```(
M````$@`+`!P``````````````"(````K```````````````2````/`,`````
M````````$@```(0#```@D0$``````!``%0`Q```````````````2````D@,`
M`)PI```@````$@`+`#@``````````````!(```!)`P`````````````2````
M0```````````````$@```*T#``#8,P``9````!(`"P"#```````````````2
M````P0,``(21`0``````$``5`$@``````````````!(```"Y`P``()$!````
M```0`!0`6```````````````$@```&<``````````````!(```!M````````
M```````2````=```````````````$@```'X``````````````!(```"(````
M```````````2````P`,``(21`0``````$``5`)$``````````````!(```!0
M`@`````````````2````EP``````````````$@```)X``````````````!(`
M``"L```````````````2````O0``````````````$0```,H`````````````
M`!(```#,`P``@"0``#`````2``L`V0,``+`D``!@````$@`+`.4#``"D*@``
MI`8``!(`"P#8```````````````6````ZP``````````````$@```%L#````
M`````````!(```#V```````````````2````_P```("1`0`$````$0`5``<!
M`````````````!(````0`0`````````````2````,@,`````````````$@``
M`!P!`````````````!(````C`0`````````````2````\@,``*1````8`0``
M$@`+`#8!`````````````!$```!(`0`````````````2````6`$`````````
M````$@```&8!`````````````!(```!K`0`````````````1````>@$`````
M````````$@```/T#```<D0$`!````!$`%`")`0`````````````2````E0$`
M````````````$@```*H!`````````````!(```"U`0`````````````2````
M;@,``(21`0``````$``5`.P"`````````````!(```#(`0`````````````2
M````0`0``*@6````````$@`+`-8!`````````````!(```#>`0``````````
M```B````]@$`````````````$@````("`````````````!(````'!```A)$!
M```````0`!4`#P0``!`E``#,`0``$@`+``@"`````````````"(````>`@``
M```````````2````*0(`````````````$@```!P$``"\00``=`$``!(`"P`Z
M`@`````````````2````)@0``""1`0``````$``5`#($```<(0``:````!(`
M"P`Z!```#$0``)`"```2``L`00(`````````````$@```#\$``"H%@``````
M`!(`"P!(`@`````````````2````1P0```0B``!\`@``$@`+`%0$``"\*0``
MZ````!(`"P!/`@`````````````2````9@0``-PF``"`````$@`+`%@"````
M`````````!(```!I`@`````````````2````^P(`````````````$@```'`"
M`````````````!(```!T!```7"<``&`````2``L`?P(`````````````$@``
M`(@"`````````````!(```".`@`````````````2````)0,`````````````
M$@```),"`````````````!(````8`P``1)$!``0````1`!4`FP(`````````
M````$@```%`#`````````````!(```"B`@`````````````2````?`0``$!`
M``!D````$@`+`*D"`````````````!(```"X`@`````````````1````#0,`
M`-B0`0`$````$0`4`(4$``!(,0``D`(``!(`"P`O`0`````````````2````
MO0(`````````````$@```&(#`````````````!(`````;&EB<&5R;"YS;P!R
M96%L;&]C`&UE;6UO=F4`7U]C>&%?9FEN86QI>F4`<FUD:7(`<W1R8VAR`'-T
M<FYC;7``<W1R<F-H<@!B;V]T7T1Y;F%,;V%D97(`<&5R;%]C;VYS=')U8W0`
M86)O<G0`=6YL:6YK`%]?;'-T870U,`!P97)L7V9R964`7U]S=&%T-3``7V5X
M:70`;65M8W!Y`'!E<FQ?9&5S=')U8W0`7U]S=&%C:U]C:&M?9F%I;`!03%]D
M;U]U;F1U;7``4&5R;%]S=E]S971P=@!03%]C=7)R96YT7V-O;G1E>'0`<&5R
M;%]A;&QO8P!C;&]S961I<@!E;G9I<F]N`')E861L:6YK`%]?;W!E;F1I<C,P
M`&UE;7-E=`!097)L7W-A9F5S>7-M86QL;V,`7U]S=&%C:U]C:&M?9W5A<F0`
M4&5R;%]G=E]F971C:'!V`%]?<W!R:6YT9E]C:&L`<F5A9`!03%]M96UO<GE?
M=W)A<`!097)L7W-Y<U]I;FET,P!097)L7V=E=%]S=@!097)L7V-R;V%K7VYO
M8V]N=&5X=`!097)L7VYE=UA3`%!E<FQ?871F;W)K7W5N;&]C:P!097)L7W-Y
M<U]T97)M`%]?97)R;F\`7U]D97)E9VES=&5R7V9R86UE7VEN9F\`4&5R;%]M
M9U]S970`;6MD:7(`7U]R96=I<W1E<E]F<F%M95]I;F9O`'!E<FQ?<&%R<V4`
M4&5R;%]N97=84U]F;&%G<P!A8V-E<W,`9V5T<&ED`&UE;6-H<@!V9G!R:6YT
M9@!097)L7V%T9F]R:U]L;V-K`&UE;6-M<`!097)L7W-V7W-E='!V;@!P97)L
M7W)U;@!L<V5E:P!O<&5N`'-P<FEN=&8`9V5T=6ED`'-T<FQE;@!?7W1L<U]G
M971?861D<@!?7W-&`'-T<F-M<`!L:6)M+G-O+C``;&EB8W)Y<'0N<V\N,0!L
M:6)P=&AR96%D+G-O+C$`<'1H<F5A9%]A=&9O<FL`<W1R9'5P`&QI8F,N<V\N
M,3(`7U]P<F]G;F%M90!?7W!S7W-T<FEN9W,`7U]G971P=W5I9#4P`%]?<WES
M8V%L;`!?7W-T<&-P>5]C:&L`871E>&ET`%]L:6)C7VEN:70`<W1R=&]K`%]?
M<F5A9&1I<C,P`%]E;F0`<&%R7V-U<G)E;G1?97AE8P!?7V)S<U]S=&%R=%]?
M`&=E=%]U<V5R;F%M95]F<F]M7V=E='!W=6ED`'!A<E]C;&5A;G5P`%]E9&%T
M80!?7V)S<U]E;F1?7P!P87)?8F%S96YA;64`<&%R7V1I<FYA;64`<&%R7VUK
M=&UP9&ER`'-H85]U<&1A=&4`;7E?<&%R7W!L`%]?96YD7U\`<&%R7VEN:71?
M96YV`'-H85]F:6YA;`!?7V)S<U]S=&%R=`!X<U]I;FET`&UA:6X`7U]S=&%R
M=`!P87)?9FEN9'!R;V<`<&%R7W-E='5P7VQI8G!A=&@`<&%R7V5N=E]C;&5A
M;@!P87)?9&EE`'-H85]I;FET`%A37TEN=&5R;F%L<U]005)?0D]/5``O=7-R
M+W!K9R]L:6(O<&5R;#4O-2XT,"XP+V5A<FUV-VAF+6YE=&)S9"UT:')E860M
M;75L=&DO0T]213HO=7-R+W!K9R]L:6(`;(X!`!<```!PC@$`%P```'R0`0`7
M````A)`!`!<```",D`$`%P```)R0`0`7````H)`!`!<```"HD`$`%P```+"0
M`0`7````P)`!`!<```#(D`$`%P```-"0`0`7````U)`!`!<```#8D`$`%P``
M`-R0`0`7````[)`!`!<```#PD`$`%P```/B0`0`7````_)`!`!<`````D0$`
M%P````21`0`7````")$!`!<````0D0$`%P```!21`0`7````')$!`!<```"`
MD`$`%0<``(B0`0`5$P``D)`!`!4A``"4D`$`$28``)B0`0`2)@``I)`!`!4Q
M``"LD`$`%34``+20`0`5.P``N)`!`!5!``"\D`$`%48``,20`0`550``S)`!
M`!5E``!TCP$`%@,``'B/`0`6!```?(\!`!8%``"`CP$`%@<``(2/`0`6"```
MB(\!`!8)``",CP$`%@L``)"/`0`6#0``E(\!`!8.``"8CP$`%@\``)R/`0`6
M$0``H(\!`!85``"DCP$`%A8``*B/`0`6%P``K(\!`!88``"PCP$`%AD``+2/
M`0`6&@``N(\!`!8<``"\CP$`%AT``,"/`0`6'@``Q(\!`!8?``#(CP$`%B``
M`,R/`0`6(@``T(\!`!8G``#4CP$`%B@``-B/`0`6*0``W(\!`!8K``#@CP$`
M%BP``.2/`0`6+0``Z(\!`!8N``#LCP$`%B\``/"/`0`6,@``](\!`!8S``#X
MCP$`%C0``/R/`0`6-@```)`!`!8X```$D`$`%CD```B0`0`6.@``#)`!`!8]
M```0D`$`%CX``!20`0`60```&)`!`!9!```<D`$`%D(``""0`0`60P``))`!
M`!9&```HD`$`%D<``"R0`0`62```,)`!`!9*```TD`$`%DX``#B0`0`64```
M/)`!`!93``!`D`$`%E8``$20`0`65P``2)`!`!98``!,D`$`%EH``%"0`0`6
M6P``5)`!`!9<``!8D`$`%ET``%R0`0`67@``8)`!`!9@``!DD`$`%F$``&B0
M`0`68@``;)`!`!9D``!PD`$`%F@``'20`0`6:0``>)`!`!9J```$X"WE!."?
MY0[@C^`(\+[EW'L!``#&C^(7RHSBW/N\Y0#&C^(7RHSBU/N\Y0#&C^(7RHSB
MS/N\Y0#&C^(7RHSBQ/N\Y0#&C^(7RHSBO/N\Y0#&C^(7RHSBM/N\Y0#&C^(7
MRHSBK/N\Y0#&C^(7RHSBI/N\Y0#&C^(7RHSBG/N\Y0#&C^(7RHSBE/N\Y0#&
MC^(7RHSBC/N\Y0#&C^(7RHSBA/N\Y0#&C^(7RHSB?/N\Y0#&C^(7RHSB=/N\
MY0#&C^(7RHSB;/N\Y0#&C^(7RHSB9/N\Y0#&C^(7RHSB7/N\Y0#&C^(7RHSB
M5/N\Y0#&C^(7RHSB3/N\Y0#&C^(7RHSB1/N\Y0#&C^(7RHSB//N\Y0#&C^(7
MRHSB-/N\Y0#&C^(7RHSB+/N\Y0#&C^(7RHSB)/N\Y0#&C^(7RHSB'/N\Y0#&
MC^(7RHSB%/N\Y0#&C^(7RHSB#/N\Y0#&C^(7RHSB!/N\Y0#&C^(7RHSB_/J\
MY0#&C^(7RHSB]/J\Y0#&C^(7RHSB[/J\Y0#&C^(7RHSBY/J\Y0#&C^(7RHSB
MW/J\Y0#&C^(7RHSBU/J\Y0#&C^(7RHSBS/J\Y0#&C^(7RHSBQ/J\Y0#&C^(7
MRHSBO/J\Y0#&C^(7RHSBM/J\Y0#&C^(7RHSBK/J\Y0#&C^(7RHSBI/J\Y0#&
MC^(7RHSBG/J\Y0#&C^(7RHSBE/J\Y0#&C^(7RHSBC/J\Y0#&C^(7RHSBA/J\
MY0#&C^(7RHSB?/J\Y0#&C^(7RHSB=/J\Y0#&C^(7RHSB;/J\Y0#&C^(7RHSB
M9/J\Y0#&C^(7RHSB7/J\Y0#&C^(7RHSB5/J\Y0#&C^(7RHSB3/J\Y0#&C^(7
MRHSB1/J\Y0#&C^(7RHSB//J\Y0#&C^(7RHSB-/J\Y0#&C^(7RHSB+/J\Y0#&
MC^(7RHSB)/J\Y0#&C^(7RHSB'/J\Y0#&C^(7RHSB%/J\Y0#&C^(7RHSB#/J\
MY0#&C^(7RHSB!/J\Y0#&C^(7RHSB_/F\Y0#&C^(7RHSB]/F\Y0#&C^(7RHSB
M[/F\Y0#&C^(7RHSBY/F\Y0#&C^(7RHSBW/F\Y0#&C^(7RHSBU/F\Y0`0H.$"
M`*#A!]#-XSD``.I,$)_E#<"@X4@@G^4!$(_@1#"?Y3#8+>D#,(_@`B"1YP2P
M3.("`%/A,*B=*`-0X.$"4(7@`U#%XP1`0^(#4(7@!#"TY3/_+^$%`%3A^___
M&C"HG>B<>`$`%`$``)AW`0!T()_E#<"@X7`0G^5P,)_E`B"/X/#8+>D$L$SB
M`4"2YP,PDN<#`%3A\*B=*`0@U.6@`%+C#P``&@A`A.('<(/B!'!'X`=PQ^,$
M<(?@"%`4Y01@H.$`,)7E,_\OX0<`5.$``(7E\*B="`0PUN4(0(3BH`!3X_3_
M_PHB___K.'@!`#@!```<`0```%!1XG2!G^4-P*#A"("/X(#8+>D$L$SB#-!-
MXDL```I<(9_E`,"5Y5@QG^4"$)CG`""<Y0!0@>4``%+C`V"8YP@PE>4`,(;E
M.@``"C@QG^4#$)CG`""!Y0`@G.4`,-+E``!3XP4```H!((+B+P!3XP`@@04!
M,-+D``!3X_K__QH``%#C````"O#^_^N+___K^#"?Y0,PF.<``%/C-0``"NP@
MG^7L,)_E`B"8YP,PC^`"`%/A"```*@-PX.$"<(?@`W#'XP1`0^(#<(?@!#"T
MY3/_+^$'`%3A^___&K@`G^4``(_@V?[_Z[`@G^6P,)_E`B"8YP,PC^`"`%/A
M"```*@-PX.$"<(?@`W#'XP1`0^(#<(?@!#"TY3/_+^$$`%?A^___&@`@EN4`
M$)7E!`"5Y<T*`.NM_O_K3""?Y60PG^4"()CG`S"/X``P@N7+___J5#"?Y1,@
MH.,$`*#C`""-Y0,PC^`"(*#C`!"@X_/^_^L!`*#CT/[_ZW[__^O'___JM'<!
M`&`!```T`0``:`$``%@!```D`0``&'8!`##^__]``0``T'4!`%1X`0"4+0``
M#<"@X=0PG^7PVRWI`(!0X@2P3.(!<*#A`S"/X"L```J\()_E`C"3YP"0D^4`
M`%GC*```"@`PV.4]4%/B`5"@$P``4^,`4*`#``!5XQ$``!H`0)GE``!4XPP`
M``H)8*#A!2"@X0@0H.$$`*#AB/[_ZP``4.,"```:!3#4YST`4^,+```*!$"V
MY0``5./S__\:!`"@X?"KG>@(4*#A`3#UY0``4^,]`%,3^___&@A01>#F___J
M`5"%X@E@1N`%0(3@1F&@X0!@A^4$`*#A\*N=Z`A`H.'M___J"4"@X>O__^KX
M=0$`-`$```W`H.&0,)_E\-@MZ02P3.*(<)_E"-!-XB1`2^('<(_@`%"@X0-@
ME^<`,);E(#`+Y000H.$%`*#AMO__ZP``4.,.```*6#"?Y200&^4#,)?G`""3
MY0$Q@N`$`)/E``!0XP$!@N?Q__\*!#"#X@0@D^4``%+C!""#Y/O__QKK___J
M("`;Y0`PEN4#`%+A`0``&AS02^+PJ)WH;?[_ZSP!```,=0$`-`$``$0QG^4-
MP*#A0"&?Y0,PC^#PWRWI!+!,XJS03>("0)/G`&!0X@`PE.4P,`OE`@``"@`P
MUN4``%/C!0``&C`@&^4`,)3E`P!2X3X``!HHT$OB\*^=Z&;^_^L`<%#B]O__
M"NR`G^7,,$OBZ)"?Y=0P"^4(@(_@X#"?Y0F0C^`#,(_@T#`+Y0<`H.'+_O_K
M``!0XR4```H-4(#B"!"@X04`H.'"_O_K``!0X_7__PH)$*#A!0"@X;W^_^L`
M`%#C\/__"@8`H.&P_O_K`*"@X04`H.&M_O_K``"*X`(`@.*P_O_K!3"@X08@
MH.'0$!OE`*"@X9S^_^O4$!OE"@"@X1O^_^L!`'#C"@"@X0,```K$,!OE#SH#
MX@$)4^,(```*"O[_ZPH`H.'__?_KU?__Z@<`H.$I_O_K!@"@X>C]_^N^___J
MKO__Z_7__^H7_O_K='0!`#P!``!L*P``:"L``&@K```-P*#A'#*?Y?#?+>D$
ML$SB%)*?Y1303>(`(-'E"9"/X`%`H.$]`%+C`%"@X0-@F><!0($"!`"@X0`P
MEN4P,`OE>?[_ZS002^(`<*#A!0"@X2[__^L`@%#B/@``"G+^_^L'`%#A-0``
M*L`QG^4T@!OE`Y"9YPB!H.$`,-7E/0!3X@$`H!,``%/C``"@`P``4.,%```*
M!0"@X0$P\.4``%/C/0!3$_O__QH%`$#@`G"'X@"@F>4``(?@8?[_ZP``4.,(
M`(KG5```"@`@F>4!$(7B-``;Y0`PU>4``9+G/0!3XP``4Q,`,,#E!@``"@$@
M@.(!,-'D`@"@X3T`4^,``%,3`3#"Y/G__QH`(*#A/3"@XP`PP.4!,-3D``!3
MXP$PXN7[__\:``"@XS`@&^4`,);E`P!2X3H``!HHT$OB\*^=Z`$P2.(!(-3D
M``!2XP$@X^7[__\:\O__ZM`PG^4#D)GG`""9Y0"@DN4``%KC"!"@`PD```H(
M,*#A`A"@X0```.H*,*#A!`"QY0&@@^(``%#C^O__&@,P@^(#$:#AD#"?Y0,P
MC^```)/E``!0XQ,``!H!`*#A/"`+Y0$0H.,`$(/E'_[_ZP`P4.(X,`OE$@``
M"CP@&^4*@:#A`A"@X0@@H.%@_?_K.#`;Y0`PB>4!((KB`!"@XP(1@^<TH`OE
MG?__Z@(`H.%4_?_K``!0XP`PH.$``(GE"H&@$?/__QH``.#CP/__ZHC]_^L\
M`0``!',!`#0!``"(<P$`@#"?Y0W`H.%\()_E`S"/X##8+>D$L$SBH-!-X@)`
MD^>T$$OB`%"@X0`PE.48,`OE8OW_ZP``4.,``*`3!```&JPP&^4/.@/B"@I3
MXP$)4Q,%```*&"`;Y0`PE.4#`%+A!P``&A302^(PJ)WH!0"@X0(0H..T_?_K
M$`]OX:`"H.'S___J8OW_Z]QP`0`\`0```#&?Y0W`H.'PWRWI!+!,XO3`G^4#
M,(_@`=M-XD-.2^(,T$WB#("3YP!0H.'<`)_E`6"@X?\C`.,$$*#A`#"8Y0``
MC^`P,`OE7/W_ZP"@4.(B``#:N`"?Y0%P2N(``(_@POW_ZRPP2^(',(/@!#13
MY0``4^,`D)#E$0``"@IPH.$'(*#A+Q"@XP0`H.&0_?_K``!0XPP```H',*#A
M!""@X040H.$)`*#AE?W_ZS`@&^4`,)CE`P!2X1$``!HHT$OB\*^=Z```5^/L
M__\:!B"@X040H.$)`*#A+/W_Z_+__^HP`)_E``"/X*#]_^L&(*#A!1"@X0"0
MD.4)`*#A(_W_Z^G__^H>_?_K1'`!`#P!``"$)P``*'$!`*!P`0#D,)_E#<"@
MX3#8+>D$L$SBV,"?Y0,PC^`!VTWB04Y+X@C03>(,4)/G"$!$X@`0H.$!*Z#C
M!`"@X0`PE>48,`OEV/S_ZP0`0.`!`$#B`#"$X`0`4^$*``":`"#4YR\`4N,'
M```:`!"@XP(``.H`(-/E+P!2XP(``!H!$$/D!`!3X?G__QHO$*#C!`"@X='\
M_^L``%#C"@``"@0`4.$`,*"#`##`A00`H.%+_?_K&"`;Y0`PE>4#`%+A"@``
M&A302^(PJ)WH&"`;Y0`PE>4#`%+A!```&A@`G^4``(_@%-!+XC!HG>@\_?_J
MWOS_ZRAO`0`\`0``P"4``$P@G^4-P*#A2#"?Y4@`G^4"((_@\-@MZ02P3.(`
M`(_@`V"2YTS]_^LP4)_E,$"?Y050C^`&(*#A!$"/X`4PH.$$$*#A``"0Y1S0
M2^+P:)WH]OS_ZC1N`0`@`0``5&\!`%PE``",)0``7`"?Y0W`H.$PV"WI``"/
MX`C03>($L$SB2$"?Y33]_^L$0(_@`%"@X0``D.79___K-""?Y0``E>4`$*#C
M+#"?Y0(@E.<$$(WE`S"/X"`0G^4`,(WE`1"/X/7\_^L4T$OB,*B=Z/QN`0"\
M;0$`;`$````H```D)0````"@XQ[_+^%(,I_E#<"@X40BG^4#,(_@\-\MZ1/=
M3>($L$SB#-!-X@(PD^<`0*#A$WU+XB0"G^4,<$?BX#0+Y0%@H.$`,)/E!!!'
MX@``C^`P,`OEOOW_ZR\0H.,`4*#A!`"@X5S\_^L``%#C"P``"NP!G^4$$*#A
M``"/X'#^_^L$`*#AX#0;Y3`@&^4`,)/E`P!2X6X``!HHT$OB\*^=Z`8`H.'6
M_/_KN!&?Y0$0C^!__/_K`)!0XNS__PJH,9_EJ*&?Y0,PC^#<-`OEH#&?Y0J@
MC^"<@9_E`S"/X.0T"^64,9_E"("/X`,PC^#8-`OE`##9Y0``4^,\```*``!5
MXP0```H)$*#A!0"@X>#\_^L``%#C+0``"@D`H.'3_/_K`1!`X@!@H.$!,(G@
M`P!9X0T``"H!$-GG+P!1XPH``!H``*#C`@``Z@`@T^4O`%+C`@``&@$`0^0#
M`%GA^?__&@D`H.'!_/_K`&"@X00`H.&^_/_K!@"`X`$`@.(!"U#CNO__*M@T
M&^5#;DOB`2N@XP`0H.,(0(WE!@"@X00PC>4*,*#A`)"-Y5G\_^L'$*#A!@"@
MX2;\_^L``%#C`P``&L0T&^4/.@/B`@E3XQ$```H($*#A``"@XS7\_^L`D%#B
MHO__"@`PV>4``%/CPO__&@``5>,3```*W!0;Y04`H.&C_/_K``!0XP%@H!/<
ME!L5[O__"M3__^H!$*#C!@"@X6C\_^L``%#CZ/__&DP`G^4&$*#A``"/X`'^
M_^L&`*#A;_S_ZX[__^H!8*#CY)0;Y<7__^H-_/_K4&T!`#P!``#,)```K"0`
M`(@D``#\(P``:"0``.@C``!0)```&"<``/`B```-P*#A+Q"@XS#8+>D$L$SB
M`$"@X=C[_^L``%#C`4"`$@0`H.$4T$OB,&B=Z%/\_^H-P*#A,-@MZ0!`4.($
ML$SB"@``"@`PU.4``%/C!P``"F7\_^L!`(#B`0M0XP@``(H$`*#A%-!+XC!H
MG>C,_O_J%`"?Y0``C^`4T$OB,&B=Z#[\_^H``*#C,*B=Z,@A``!L(9_E#<"@
MX6@QG^4"((_@9`&?Y7#8+>D$L$SB#-!-X@-0DN<``(_@4&&?Y21`2^(`,)7E
M!F"/X"`P"^4Y_?_K/`&?Y0``C^`V_?_K-`&?Y0``C^`S_?_K+`&?Y0``C^`P
M_?_K)`&?Y0``C^`M_?_K'`&?Y0``C^`J_?_K%`&?Y0``C^`G_?_K#`&?Y0``
MC^`D_?_K!@"@X2+]_^O\`)_E``"/X!_]_^OT`)_E``"/X!S]_^OL`)_E!!"@
MX0``C^#>_/_K`!!0X@$```H&`*#AEOW_Z]``G^4$$*#A``"/X-;\_^L`$%#B
M`@``"KP`G^4``(_@C?W_Z[0`G^4$$*#A``"/X,W\_^L`$%#B#0``"J``G^4`
M`(_@A/W_ZY@0G^68`)_E`1"/X```C^!__?_K("`;Y0`PE>4#`%+A"P``&AC0
M2^)PJ)WH=`"?Y000H.$``(_@N?S_ZP`04.+N__\*8`"?Y0``C^!P_?_KZO__
MZH'[_^M$:@$`/`$```@B``!((@``]"$``/`A``#P(0``["$``/`A``"$(0``
MY"$``.@A``!D(0``V"$``,PA``#,(0``R"$``/`@``#((0``+"$``(@A```,
M(0``;#"?Y0W`H.%H()_E`S"/X&0`G^40V"WI!+!,X@S03>("0)/G'!!+X@``
MC^``,)3E&#`+Y8[\_^L``%#C!0``"@``T.4``%#C,`!0$P$`H`,``*`3`0`@
MXA@@&^4`,)3E`P!2X0$``!H0T$OB$*B=Z$[[_^MX:`$`/`$``'P@```-P*#A
M#P`MZ40PG^4`V"WI%+!,XCS`G^4#,(_@"-!-XC0`G^4(X(OB#,"3YQ3@"^4.
M(*#A!!";Y0#`G.40P`OE``"3Y[``@.*/^__K_P"@X_?Z_^OL9P$`/`$``&0!
M``#$$9_E#<"@X?#?+>D$L$SBN,&?Y0$0C^`!V$WB`N"@XR303>(,$)'G`<A+
MXD3`3.(`(*#C`#"@XP"`H.$`$(SE`'"@XP#@C>4`$)'E,!`+Y83[_^L!0%#B
M`%#!XL5OH.$&`*#AQA^@X09@).`!,"?@`"!6X')@_^8`(";@`"!2X`$PP^`"
M0%3@`U#%X```5.,`,-7B!&"@X05PH.%!``"Z+%&?Y0$H2^(!2$OB)#&?Y4`@
M0N((D*#C!5"/X`&00.,X0$3B`S"/X``P@N4`$*#C!B"@X0<PH.$`$(WE"`"@
MX6#[_^L`,.#C`"#@XP,`4>$"`%`!+@``"@D@H.$$$*#A"`"@X17[_^L!`'#C
M&@``"@<`4.,;``":"`!`X@"@E.`#```Z%P``Z@&@2N($`%KA%```.@@@H.,%
M$*#A"@"@X3S[_^L``%#C]O__&@1`2N`$`);@Q!^GX`$X2^(P(!OE1#!#X@`P
MD^4`,)/E`P!2X10``!HHT$OB\*^=Z%@`G^4``(_@A?__ZP%H5N(`(*#C`'#'
MXO\O3^,`,.#C`P!7X0(`5@'(__\:``#@XP`0X./H___J`3A+XD`P0^(``)/E
M=O__Z\O__^K`^O_KD&<!`#P!``!D:`$`@!\``-0>```-P*#A`-@MZ02P3.(I
M^__K(OO_ZP``4.,``)`5`*B=Z,@PG^4-P*#AQ""?Y0,PC^#PV2WI!+!,XA30
M3>("0)/G`&"@X:Q0G^4L$$OB`#"4Y050C^`%`*#A*#`+Y=3[_^L`<%#B`@``
M"@`PU^4``%/C"0``&H``G^4&$*#A``"/X(?\_^LH(!OE`#"4Y0,`4N$5```:
M(-!+XO"IG>@&`*#A"OO_ZP"`H.$'`*#A!_O_ZP``B.`"`(#B"OO_ZSPPG^4\
M$)_E!B"@X0!PC>4#,(_@`1"/X`"`H.'S^O_K"!"@X04`H.%N_/_KY?__ZG_Z
M_^N890$`/`$``#`>```$'@``R!P``,@<```-P*#A(#:?Y?#?+>D$L$SB&,:?
MY44?2^)+WTWB#,"/X#@!"^4,((SB0!$+Y0'@H.'\A9_E!P"2Z`B`C^`8<(SB
M2V]+X@.0F.<PP(SB$DY+XA->2^('`(;H#P"WZ#R1"^4`D)GE#P"NZ`,`E^@P
MD`OE`P".Z`<`G.@'`(3H!1"@X;`%G^4``(_@A_O_ZP!P4.("```*`##7Y0``
M4^.+```:E/__ZP"@4.*-```*"@"@X<3Z_^N``*#A`0"`XL?Z_^L`(-KE``!2
MXP"0H.$)```*9'6?Y0E`H.$'<(_@!`"@X0<0H.&N^O_K`2#ZY0)`A.(``%+C
M^/__&A0!&^4``%#C$@``"D!!&^4"``#J!`"TY0``4.,-```*!1"@X6+[_^L`
M<%#B^/__"@`PU^4``%/C]?__"J;\_^L``%#C\O__"@<`H.&$^O_K`*!0XA<`
M`!H`<*#C+#!+X@<Q@^``01/E``!4XPH```H`,-3E``!3XP<```H$`*#A`7"'
MXI3\_^L``%#C!```&@=!EN<``%3C]/__&@"@H.,$``#J!`"@X6WZ_^L``%#C
MZ?__"@"@H.$*`*#A@_K_ZP!`H.$)`*#A@/K_ZP``A.`!2X#B!$"$X@0`H.&!
M^O_K8!2?Y6`TG^4*(*#A`1"/X``0C>54%)_E`S"/X`20C>4!$(_@`'"@X6?Z
M_^L'`*#A!QV@XS?Z_^L!`'#C&@``"OP02^('`*#AV_G_ZP!@4.(#```:]#`;
MY0\Z`^(!"5/C0P``"@@$G^4',*#A."$;Y0!`H./\$Y_E``"8YP`@DN4!$(_@
ML`"`XM?Y_^L\,1OE!`"@X3`@&^4`,)/E`P!2X>H``!HHT$OB\*^=Z!#Z_^L`
M,)#E$0!3X^#__PHX,1OE`$"3Y0KZ_^NDPY_E!S"@X:03G^4$(*#A`$"@XPS@
MF.<!$(_@`,"0Y;``CN(`P(WEO?G_Z^3__^H0___K!P"@X1_Z_^L`0*#AW___
MZ@IPH.$'D83@!)!)X@0`N>4``%#C#```"@40H.$!<(?BZ/K_ZP`P4.+W__\*
M`##3Y0``4^/T__\*#OK_ZP``4./O__\*`*"@X5[__^H-`*#C'*.?Y2CZ_^M3
M(*#C"J"/X`"0H.%@___JX)`;Y1;Z_^L``%GAM___&O0P&^53,.CG!PU3X[/_
M_QH$`*#A&OK_ZP40H.$`0*#AV`*?Y0``C^#(^O_K`!!0XE(```HX,1OE``"3
MY?#\_^L`4%#B30``"@4`H.$&^O_K`P!0XP"`H.$,``":H!*?Y00`0.(``(7@
M`1"/X`?Z_^L``%#C!0``&@0`6.."```*"("%X`4P6.4O`%/C?@``"A'^_^L`
MH%#B!```&@4`H.$*$*#AX?G_ZP!04.(V```:4!*?Y5`"G^4!$(_@3)*?Y0``
MC^!(@I_E7/O_ZT12G^6^^?_K0!*?Y4`RG^4'(*#A`1"/X`00C>4T$I_E`S"/
MX`F0C^`(@(_@`1"/X`50C^```(WE!`"@X<[Y_^L-``#JEOG_ZP`PD.41`%/C
M#@``&JKY_^L!8(;B"%"-Y0DPH.$$8(WE!R"@X0@0H.$``(WE!`"@X;_Y_^L'
M':#C!`"@X8_Y_^L!`'#C[/__"@<`H.$H^?_KO`&?Y000H.$``(_@,OO_ZP0`
MH.&/_O_K8?__ZC@Q&^4`4)/EKO__Z@K^_^L``%#C`("@X0`PT>(!D*#APO__
MN@H@4.)D,$OB`*"-Y04`H.$X,0OE`##!XICY_^LX$1OE!B"@XP4`H.%2^?_K
M!@!0XP(```I,`9_E``"/X-W]_^M$,9_E9"`;Y0,PC^`,$(/B##"3Y0,`4N&K
M__\:.#$;Y;0@T^&T,-'A`P!2X:;__QHR(%CB7&!+X@`PR>(`$*#C!0"@X0`0
MC>5\^?_K!0"@X080H.$H(*#C-OG_ZR@`4.,"```*Y`"?Y0``C^#!_?_KW!"?
MY0<@H.'8,)_E!`"@X0$0C^`$$(WES!"?Y0,PC^``8(WE`,"@XP$0C^`TP$OE
M;_G_Z[/__^HX,1OE!("3Y0``6.-\__\*"`"@X7'Y_^L#`%#C>/__FI`0G^4$
M`$#B``"(X`$0C^!S^?_K``!0XPA0H`%P___J[OC_ZSP!```89@$`A&0!`.P;
M``"X'```P!L``&@=``"@&P``9`$``)`;``#T&@``2!H``+P:``"`&@``1!D`
M`,`8```,&P``,!H``-`:``#L&@``$!L``!P:``"H%P``'!D``+P6``"L&```
M:!D``(P9``"(&```.!@``#PBG^4-P*#A.#*?Y0(@C^`T`I_E\-DMZ02P3.(,
MT$WB`V"2YP``C^`L4$OB`#"6Y2@P"^4]^?_K!1"@X0`PD.4(`I_E9"#3Y0``
MC^!E(,/E[/G_ZP``4.,"```*`##0Y3$`4^->```*UOS_Z^`!G^7@09_E``"/
MX"SY_^L$0(_@`S"@XP$@H.,$$*#A`'"@X0``D.7(^/_K`$!0X@H```H,,)3E
ML"&?Y0``E^4`$)/E`B"/X*7X_^L,,)3E`!"3Y0@PD>4!!1/C40``&HP!G^6,
M09_E``"/X!3Y_^L$0(_@`S"@XP$@H.,$$*#A`'"@X0``D.6P^/_K`$!0X@D`
M``I@`9_E!1"@X0``C^"]^?_K`"!0XC8```H,,)3E``"7Y0`0D^6)^/_K/`&?
MY3R!G^4``(_@_?C_ZS01G^4!(*#C"("/X`$0C^``<)#E`$"@X0<`H.&D^/_K
M&"&?Y0(@C^``$*#A!P"@X7CX_^L`,)3E!`&?Y0`@H.,%$*#A``"/X&4@P^6?
M^?_K``!0XP@`B.4+```*!QV@XZOX_^L!,(#B$0!0XP$`4Q,=``"**"`;Y0`P
MEN4#`%+A&```&B#02^+PJ9WH!`"8Y>/]_^L``%#C"`"(Y>[__QJ@`)_E``"/
MX(3X_^L!,-#E``!3XY[__PJ<___JB#"?Y0P@E.4#,(_@!#"3Y0``DN4`$)/E
MZ_K_Z\3__^H``)?EB/C_ZZK__^I'^/_K7""?Y0`PH.%8`)_E`B"/X```C^`&
M`)+I`!"1Y6SX_^L,7@$`/`$``"!?`0#4%0``T%X!`'07``!$%P``<%X!`"`7
M``#0%```%%X!`.!>`0"\%@``O!8``%`4``!$%@``)%X!`/!=`0`0%@``#<"@
MX3#8+>D$L$SB`$"@X;O\_^L``%#C``!4$S"HG0@`,-3E``!3XS"HG0@$`*#A
M*/S_ZQO\_^L@$)_E!""@XP$0C^#P]__K``!0XS"HG1@$`*#A%-!+XC!HG>BI
M^?_J,!0``/0[G^4-P*#A\-\MZ0,PC^#HZY_E!+!,XE7?3>(73DOB'""`X@XP
MD^=<P(#B!!"@X7@Q"^4`,)/E,#`+Y00PDN0"`%SA,S^_Y@0P@>3Z__\:_$"$
MXEWO2^(.(*#A!!"^Y3@PDN4DP)+E!`!>X0P@DN4,,"/@`C`CX`$P(^#C/Z#A
M0#".Y?/__QIP<1OEF3D'XP00D.6".D7C$."0Y0-PA^`(4)#EH2L.XPQ`D.4.
M<(?@`&"0Y6&1H.$$X)#E!!`EX&S!&^79+D;C#A`!X&:!H.$)X"7@!!`AX&BA
M&^7F?8?@`7"'X`/`C.`($"G@!N`.X`<0`>`$P(S@`Z"*X`7@+N`,X([@!:"*
MX`G`(>#G[8[@"L",X&=QH.%DH1OE!Q`HX`X0`>#NS8S@`Z"*X`@0(>`)D(K@
M;N&@X0D0@>!@D1OE#J`GX.P=@>`,H`K@`Y")X&S!H.$(D(G@!X`JX%RA&^4)
M@(C@#)`NX`.@BN`!D`G@X8V(X`Z0*>!A$:#A!W"*X%BA&^4'<(G@`9`LX.A]
MA^`#H(K@")`)X`R0*>`.X(K@:(&@X0[@B>!4D1OE"*`AX`>@"N#G[8[@`Y")
MX&=QH.$,D(G@`<`JX%"A&^4)P(S@!Y`HX.[-C.`#H(K@#I`)X`$0BN`(H"G@
M;N&@X0$0BN!,H1OE#I`GX`R0">#L'8'@`Z"*X`>0*>`(@(K@;,&@X0B0B>!(
M@1OE#*`NX.&=B>`!H`K@`X"(X&$1H.$'@(C@#G`JX$2A&^4(@(?@`7`LX`.@
MBN`)<`?@Z8V(X`[@BN!ID:#A0*$;Y0QP)^`.<(?@">`AX`.@BN`(X`[@#,"*
MX`'@+N#H?8?@#.".X&B!H.$\P1OE"*`IX.?MCN`'H`K@`\",X&=QH.$!P(S@
M"1`JX#BA&^4,P('@!Q`HX`.@BN`.$`'@"9"*X`B@(>#NS8S@"9"*X&[AH.$T
MH1OE#A`GX.R=B>`#H(K@#!`!X`<0(>!LP:#A"("*X#"A&^4(@('@#!`NX`.@
MBN`)$`'@Z8V(X`X0(>!ID:#A!W"*X"RA&^4'<('@"1`LX.A]A^`#H(K@"!`!
MX`P0(>!H@:#A#N"*X"BA&^4.X('@"!`IX`.@BN`'$`'@#*"*X`D0(>#G[8[@
M"A"!X&=QH.$DH1OE!\`HX.X=@>`#,(K@#L`,X""A&^4(P"S@;N&@X0DP@^`#
M,(S@`I"*X`[`)^`<H1OEX3V#X`B0B>`!P"S@88&@X0@0+N`,P(G@`I"*X!BA
M&^7CS8S@!W")X`,0(>!C,:#A`I"*X`$0A^`4H1OE`W`HX.P=@>`.X(G@#'`G
MX&S!H.$"D(K@!W".X!"A&^4,X"/@X7V'X`B`B>`!X"[@81&@X0&0+.`.X(C@
M`H"*X`RA&^7G[8[@`X"(X`<P*>!G<:#A!Y`AX`,PB.`"@(K@"*$;Y>X]@^`,
M@(C@#L`IX&[AH.$.D"?@#,"(X`*`BN`$H1OEX\V,X`&`B.`#$"G@8S&@X0.0
M+N`!$(C@`H"*X`"A&^7L'8'@!X"(X`QP*>!LP:#A#)`CX`=PB.`"@(K@_*`;
MY>%]A^`.@(C@`>`IX&$1H.$!D"S@#N"(X`*`BN#XH!OEY^V.X`.`B.`',"G@
M9W&@X0>0(>`#,(C@`H"*X/2@&^7N/8/@#("(X`[`*>!NX:#A#I`GX`S`B.`"
M@(K@\*`;Y>/-C.`!@(C@`Q`IX&,QH.$#D"[@`1"(X`*`BN#LH!OE[!V!X`>`
MB.`,<"G@;,&@X0R0(^`'<(C@`H"*X.B@&^7A?8?@#H"(X`'@*>!A$:#A`9`L
MX`[@B.`"@(K@Y*`;Y>?MCN`#@(C@!S`IX&=QH.$'D"'@`S"(X`*`BN#@H!OE
M[CV#X`R`B.`.P"G@;N&@X0Z0)^`,P(C@`H"*X-R@&^7CS8S@`8"(X`,0*>!C
M,:#A`I"*X`$0B.#8H!OE`X`NX.P=@>`'<(G@#(`HX&S!H.$"D(K@"("'X-2@
M&^4,<"/@X8V(X`%P)^`.X(G@81&@X0?@CN`"H(K@T)`;Y0$@+.!H<:#AZ.V.
MX`.@BN`"@"C@W#P+XQL_2.,'(('A"*"*X`.0B>!N@:#A#B`"X`S`B>#N[8K@
MS)`;Y0>@`>`*H(+A"""'X0S`BN`.(`+@`Z")X`B0!^#NS8S@`1"*X&[AH.$)
M((+AR)`;Y0$@@N`.$(CA#J`(X`.0B>`,$`'@!W")X`H0@>'L+8+@!Q"!X&S!
MH.'$<!OE#)".X>(=@>`#H(?@`I`)X&)QH.$,(`[@`B")X<"0&^4(H(K@!X",
MX0H@@N`!@`C@`Z")X`>0#.#A+8+@"8"(X6$1H.$.X(K@O)`;Y0[@B.`!@(?A
M`:`'X`*`".`#D(G@XNV.X`R0B>!B(:#A"J"(X;B`&^4"P('A":"*X`[`#.`#
MD(C@;H&@X>[MBN`"H`'@!Y")X`K`C.$)P(S@M)`;Y0AP@N'NS8S@`Z")X`YP
M!^`(D`+@;N&@X0EPA^$!$(K@L)`;Y0$0A^`.<(CA#J`(X`QP!^`#D(G@[!V!
MX`*0B>!LP:#A"J"'X:QP&^4,(([A":"*X`$@`N`#D(?@87&@X>$=BN`,H`[@
M")")X`H@@N$)((+@J)`;Y0>`C.'A+8+@`Z")X`&`".`'D`S@81&@X0F`B.$.
MX(K@I)`;Y0[@B.`!@(?A`:`'X`*`".`#D(G@XNV.X`R0B>!B(:#A"J"(X:"`
M&^4"P('A":"*X`[`#.`#D(C@;H&@X>[MBN`"H`'@!Y")X`K`C.$)P(S@G)`;
MY0AP@N'NS8S@`Z")X`YP!^`(D`+@;N&@X0EPA^$!$(K@F)`;Y0$0A^`.<(CA
M#J`(X`QP!^`#D(G@[!V!X`*0B>!LP:#A"J"'X91P&^4,(([A":"*X`$@`N`#
MD(?@87&@X>$=BN`,H`[@")")X`H@@N$)((+@D)`;Y0>`C.'A+8+@`Z")X`&`
M".`'D`S@81&@X0F`B.$.X(K@C)`;Y0[@B.`!@(?A`:`'X`*`".`#D(G@"J"(
MX8B`&^7B[8[@#)")X&(AH.$)H(K@`L"!X0.0B.`.P`S@;H&@X0=PB>#N[8K@
MA)`;Y0*@`>`*H(SA",""X0=PBN`#,(G@#L`,X`B0`N"`H!OE[GV'X`$P@^!N
MX:#A"<",X=81#.-B&DSC`\",X`&0BN`.,"C@?*`;Y>?-C.`"((G@!S`CX&=Q
MH.$!D(K@`S""X'B@&^4'("[@[#V#X`B0B>`,@"+@;"&@X0C`B>`"D"?@`8"*
MX'2@&^7CS8S@#H"(X`/@*>!C,:#A`9"*X`[@B.!PH!OE`X`BX.SMCN`'D(G@
M#(`HX&QQH.$(P(G@!Y`CX`&`BN!LH!OE[LV,X`*`B.`.("G@;N&@X0(@B.`.
MD"?@`8"*X&B@&^7L+8+@`X"(X`PP*>!LP:#A`S"(X`R0+N`!@(K@9*`;Y>(]
M@^`'@(C@`G`IX&(AH.$'<(C@`I`LX`&`BN!@H!OEXWV'X`Z`B.`#X"G@8S&@
MX0[@B.`#D"+@`8"*X%R@&^7G[8[@#("(X`?`*>!G<:#A#,"(X`>0(^`!@(K@
M6*`;Y>[-C.`"@(C@#B`IX&[AH.$"((C@#I`GX`&`BN!4H!OE["V"X`.`B.`,
M,"G@;,&@X0,PB.`,D"[@`8"*X%"@&^7B/8/@!X"(X`)P*>!B(:#A!W"(X`*0
M+.`!@(K@3*`;Y>-]A^`.@(C@`^`IX&,QH.$.X(C@`Y`BX`&`BN!(H!OEY^V.
MX`R`B.`'P"G@9W&@X0S`B.`'D"/@`8"*X$2@&^7NS8S@`H"(X`X@*>!NX:#A
M`B"(X`Z0)^`!@(K@0*`;Y>PM@N`#@(C@##`IX&S!H.$#,(C@#)`NX`&`BN`\
MH!OEXCV#X`>`B.`"<"G@8B&@X0=PB.`"D"S@`8"*X#B@&^4.@(C@`^`IX.-]
MA^`.X(C@`8"*X&,QH.$,P(C@-(`;Y0.0(N#G[8[@!Y`IX`$0B.!G<:#A!X`C
MX`G`C.`&$('@[LV,X`(@@>`.@"C@"(""X!`0D.7L+8C@`""`Y7@A&^4#,('@
M,)`;Y6[AA>`$$)#E!W"$X``@DN4,P('@"."`Y0(`6>$0,(#E!,"`Y0QP@.4!
M```:*-!+XO"OG>@4]?_K&%L!`#P!```-P*#A8`"@XP#8+>D$L$SBB_7_ZP$3
M`N.)*PKC11=&X\TO3N,`$(#E_AP-XP0@@.6Z&$GC=B0%XP@0@.4R($'C\!$.
MXPP@@.72$TSC`""@XQ`0@.44((#E&""`Y5P@@.4`J)WH#<"@X?#;+>D"@*#A
M%#"0Y8(AH.$`4*#A`4"@X0(PD^`8()#E7`"0Y0`0H.,!$*`C``!1XP$@@A(`
M`%#CJ"Z"X`2P3.(4,(7E')"%XA@@A>4/```*0&!@X@00H.$&`%CA``")X`A@
MH+$&(*#AU_3_ZUPPE>4#,(;@7#"%Y4``4^/PJYT8!0"@X0:`2.`&0(3@P/S_
MZS\`6.,8``#:0(!(XBACH.$!<(;B!W.$X$`@A.($$*#A"3"@X0!`D>40$('B
M#.`1Y1`P@^((P!'E!``1Y0(`4>$00`/E#.`#Y0C``^4$``/E\___&@4`H.$!
M0*#AJ/S_ZP<`5.'K__\:!H-(X`00H.$)`*#A"""@X:_T_^M<@(7E\*N=Z`W`
MH.'PVRWI'("!XA1@D>4!0*#A&'"1Y0!0H.'6,>7G?Q#@XP$@@^($L$SB.`!2
MXP,0R.<"`(C@20``VD`@8N(`$*#CN?3_ZP0`H.&+_/_K"`"@X3@@H.,`$*#C
ML_3_ZU=PQ.4G+*#A6V#$Y2<XH.%4(,3E!`"@X28LH.%5,,3E)W2@X28XH.%6
M<,3E)F2@X5DPQ.58(,3E6F#$Y7?\_^L#,-3E!`"@X0`PQ>6R,-3A`3#%Y0`P
ME.4C-*#A`C#%Y0`PE.4#,,7E!S#4Y00PQ>6V,-3A!3#%Y00PE.4C-*#A!C#%
MY00PE.4',,7E"S#4Y0@PQ>6Z,-3A"3#%Y0@PE.4C-*#A"C#%Y0@PE.4+,,7E
M#S#4Y0PPQ>6^,-3A#3#%Y0PPE.4C-*#A#C#%Y0PPE.4/,,7E$S#4Y1`PQ>6R
M,=3A$3#%Y1`PE.4C-*#A$C#%Y1`PE.43,,7E)-!+XO!KG>@Y]/_J."!BX@`0
MH.-O]/_KNO__ZF@PG^4-P*#A,-@MZ0,PC^!<0)_E!+!,XB0@T^4$0(_@``!2
MXS"HG1A(()_E`1"@XR00P^4",)3G``!3XP(```HT,)_E`P"?YPWT_^LL,)_E
M`S"4YP``4^,PJ)T((`"?Y0``C^`4T$OB,&B=Z';T_^H$3@$`%$P!`!@!``!@
M30$`4`$``'!%```\$)_E/#"?Y0$0C^`#,(_@`"#1Y0``4N,>_R\1*""?Y0$`
MH.,``,'E`C"3YP``4^,>_R\!%`"?Y000@>(``(_@:/3_ZH!-`0"<2P$`5`$`
M``Q%```-P*#A-#*?Y?#8+>D$L$SB+&*?Y2#03>(H``OE!F"/X"P0"^4P(`OE
M*`!+X@-0EN<L$$OB,"!+X@`PE>4@,`OE-?3_Z_PAG^7\,9_E`B"6YP,`EN<"
M$*#A._3_Z^PQG^4#,);G`$#3Y0``5.-.```*W`&?Y0``C^![]/_K*'`;Y000
MA^(#<(?B!P%QXP`@D.5&,]+E`S"#XT8SPN57``"*`0&@X1#T_^NH(9_E+$`;
MY0(@C^"@$9_E`1"/X``PH.$``)3E!#""Y9`AG^7P`,/A*``;Y0(0EN<!`%#C
M@"&?Y0`0D>4"((_@$`"@TPP@@^4($(/E"```V@00A.(,P(/B`$&$X`3@D>0$
M`%'A!."LY?O__QH#`(#B``&@X41!G^4`$*#C`!"-Y0<@H.$`$(/G!$"/X#`1
MG^4,`)3E`1"/X!;T_^L``%#C*0``"AQ!G^4D8$OB!$"/X`P`E.7$\__K!A"@
MX0!PH.$$`9_E``"/X//T_^L``%#C&```"O0PG^4#,(_@#`"3Y:KS_^OQ\__K
M("`;Y0`PE>4'`*#A`P!2X2,``!H<T$OB\*B=Z+GS_^O$,)_E`S"/X```4.,,
M`(/E'```"H_S_^NP`)_E``"/X"7T_^L`,)#E[D7#Y:/__^H(`)3E``!0XPD`
M``IV^__KX?__Z@P`E.4#]/_KTO__ZGPPG^5\`)_E`Q"6YP``C^#'\__K<`"?
MY080H.$``(_@Q_3_ZP``4.,(`(3ETO__"NW__^J1\__K`0"@XU#S_^L\`0``
M.$L!`$P!``!<`0``*`$```Q,`0"L3`$`#`4``$@!``#D!```+$P!`#3<__\(
M3`$`_`$``-Q+`0"H2P$`M$H!`$0!``#X`@``[````'!S7W-T<FEN9W,@;6ES
M<VEN9PH`<&5R;'AS:2YC`````$-!0TA%```N````+BX``"5S+R5S````+W!R
M;V,O8W5R<')O8R]E>&4```!$>6YA3&]A9&5R.CIB;V]T7T1Y;F%,;V%D97(`
M26YT97)N86QS.CI005(Z.D)/3U0`````4$%27U1%35``````4$%27U!23T=.
M04U%`````#H````E<R5S)7,``%!%4DPU3$E"`````%!%4DQ,24(`4$523#5/
M4%0`````4$523$E/``!005)?24Y)5$E!3$E:140`4$%27U-005=.140`4$%2
M7T-,14%.````4$%27T1%0E5'````4$%27T-!0TA%````4$%27T=,3T)!3%]$
M14)51P````!005)?1TQ/0D%,7U1-4$1)4@```%!!4E]435!$25(``%!!4E]'
M3$]"04Q?5$5-4`!005)?1TQ/0D%,7T-,14%.`````#$```!L<V5E:R!F86EL
M960`````<F5A9"!F86EL960`3$1?3$E"4D%265]0051(`%-94U1%30``)3`R
M>``````E<R5S)7,E<P````!P87(M`````"5S.B!C<F5A=&EO;B!O9B!P<FEV
M871E('-U8F1I<F5C=&]R>2`E<R!F86EL960@*&5R<FYO/25I*0H``"5S.B!P
M<FEV871E('-U8F1I<F5C=&]R>2`E<R!I<R!U;G-A9F4@*'!L96%S92!R96UO
M=F4@:70@86YD(')E=')Y('EO=7(@;W!E<F%T:6]N*0H``%!!5$@`````<&%R
M;``````N<&%R`````'-H;W)T(')E860``"5S)7-C86-H92TE<R5S```E<R5S
M=&5M<"TE=25S````)7,E<W1E;7`M)74M)74E<P`````8````<&5R;``````P
M````4$%2.CI086-K97(Z.E9%4E-)3TX`````,2XP-C,```!5;F%B;&4@=&\@
M8W)E871E(&-A8VAE(&1I<F5C=&]R>0`````E<SH@8W)E871I;VX@;V8@<')I
M=F%T92!C86-H92!S=6)D:7)E8W1O<GD@)7,@9F%I;&5D("AE<G)N;STE:2D*
M`````"UE```M+0``55-%4@````!54T523D%-10````!414U01$E2`%1-4``O
M=&UP+P```&5V86P@)V5X96,@+W5S<B]B:6XO<&5R;"`@+5,@)#`@)'LQ*R(D
M0")])PII9B`P.R`C(&YO="!R=6YN:6YG('5N9&5R('-O;64@<VAE;&P*<&%C
M:V%G92!?7W!A<E]P;#L*;7D@*"1005)?34%'24,L("1&24Q%7V]F9G-E=%]S
M:7IE+"`D8V%C:&5?;F%M95]S:7IE*3L*0D5'24X@>PHD4$%27TU!1TE#(#T@
M(EQN4$%2+G!M7&XB.PHD1DE,15]O9F9S971?<VEZ92`](#0[("`@(R!P86-K
M*").(BD*)&-A8VAE7VYA;65?<VEZ92`](#0P.PHD4$M:25!?34%'24,@/2`B
M4$M<,#`S7#`P-"(["GT*<W5B(&9I;F1?<&%R7VUA9VEC"GL*;7D@*"1F:"D@
M/2!`7SL*;7D@)&-H=6YK7W-I>F4@/2`V-"`J(#$P,C0["FUY("1B=68["FUY
M("1S:7IE(#T@+7,@)&9H.PIM>2`D<&]S(#T@*"1S:7IE+3$I("T@*"1S:7IE
M+3$I("4@)&-H=6YK7W-I>F4[("`@("`@(R!.3U1%.B`D<&]S(&ES(&$@;75L
M=&EP;&4@;V8@)&-H=6YK7W-I>F4*=VAI;&4@*"1P;W,@/CT@,"D@>PIS965K
M("1F:"P@)'!O<RP@,#L*<F5A9"`D9F@L("1B=68L("1C:'5N:U]S:7IE("L@
M;&5N9W1H*"1005)?34%'24,I.PII9B`H*&UY("1I(#T@<FEN9&5X*"1B=68L
M("1005)?34%'24,I*2`^/2`P*2!["G)E='5R;B`D<&]S("L@)&D["GT*)'!O
M<R`M/2`D8VAU;FM?<VEZ93L*?0IR971U<FX@+3$["GT*;7D@*"1P87)?=&5M
M<"P@)'!R;V=N86UE+"!`=&UP9FEL92P@)4UO9'5L94-A8VAE*3L*14Y$('L@
M:68@*"1%3E9[4$%27T-,14%.?2D@>PIR97%U:7)E($9I;&4Z.E1E;7`["G)E
M<75I<F4@1FEL93HZ0F%S96YA;64["G)E<75I<F4@1FEL93HZ4W!E8SL*;7D@
M)'1O<&1I<B`]($9I;&4Z.D)A<V5N86UE.CID:7)N86UE*"1P87)?=&5M<"D[
M"F]U=',H<7%;4F5M;W9I;F<@9FEL97,@:6X@(B1P87)?=&5M<")=*3L*1FEL
M93HZ1FEN9#HZ9FEN9&1E<'1H*'-U8B![("@@+60@*2`_(')M9&ER(#H@=6YL
M:6YK('TL("1P87)?=&5M<"D["G)M9&ER("1P87)?=&5M<#L*:68@*"UD("1P
M87)?=&5M<"`F)B`D7D\@;F4@)TU35VEN,S(G*2!["FUY("1T;7`@/2!N97<@
M1FEL93HZ5&5M<"@*5$5-4$Q!5$4@/3X@)W1M<%A86%A8)RP*1$E2(#T^($9I
M;&4Z.D)A<V5N86UE.CID:7)N86UE*"1T;W!D:7(I+`I3549&25@@/3X@)RYC
M;60G+`I53DQ)3DL@/3X@,"P**3L*;7D@)&9I;&5N86UE(#T@)'1M<"T^9FEL
M96YA;64["G!R:6YT("1T;7`@/#PB+BXN(CL*>#TQ.R!W:&EL92!;(%PD>"`M
M;'0@,3`@73L@9&\*<FT@+7)F("<D<&%R7W1E;7`G"FEF(%L@7"$@+60@)R1P
M87)?=&5M<"<@73L@=&AE;@IB<F5A:PIF:0IS;&5E<"`Q"G@]8&5X<'(@7"1X
M("L@,6`*9&]N90IR;2`G)&9I;&5N86UE)PHN+BX*8VQO<V4@)'1M<#L*8VAM
M;V0@,#<P,"P@)&9I;&5N86UE.PIM>2`D8VUD(#T@(B1F:6QE;F%M92`^+V1E
M=B]N=6QL(#(^)C$@)B(["G-Y<W1E;2@D8VUD*3L*;W5T<RAQ<5M3<&%W;F5D
M(&)A8VMG<F]U;F0@<')O8V5S<R!T;R!P97)F;W)M(&-L96%N=7`Z("1F:6QE
M;F%M95TI.PI]"GT@?0I"14=)3B!["DEN=&5R;F%L<SHZ4$%2.CI"3T]4*"D@
M:68@9&5F:6YE9"`F26YT97)N86QS.CI005(Z.D)/3U0["F5V86P@>PI?<&%R
M7VEN:71?96YV*"D["FUY("1Q=6EE="`]("$D14Y6>U!!4E]$14)51WT["FUY
M("5S>7,@/2`H"G!A=&A?<V5P("`@(#T^("@D7D\@/7X@+UY-4U=I;B\@/R`G
M.R<@.B`G.B<I+`I?97AE("`@("`@("`]/B`H)%Y/(#U^("]>*#\Z35-7:6Y\
M3U,R?&-Y9W=I;BDO(#\@)RYE>&4G(#H@)R<I+`I?9&5L:6T@("`@("`]/B`H
M)%Y/(#U^("]>35-7:6Y\3U,R+R`_("=<7"<@.B`G+R<I+`HI.PI?<V5T7W!R
M;V=N86UE*"D["F]U=',H<7%;7"1P<F]G;F%M92`]("(D<')O9VYA;64B72D[
M"E]S971?<&%R7W1E;7`H*3L*;W5T<RAQ<5M<)$5.5GM005)?5$5-4'T@/2`B
M)$5.5GM005)?5$5-4'TB72D["FUY("@D<W1A<G1?<&]S+"`D<W1A<G1?;V9?
M1DE,15]S96-T:6]N*3L*34%'24,Z('L*;&]C86P@)%-)1WM?7U=!4DY?7WT@
M/2!S=6(@>WT["G5N;&5S<R`H;W!E;B!?1D@L("<\.G)A=R<L("1P<F]G;F%M
M92D@>PIO=71S*'%Q6T-A;B=T(')E860@9G)O;2!F:6QE("(D<')O9VYA;64B
M72D[("`C(&1O;B=T('5S92`D(2!H97)E(&%S(&ET(')E<75I<F5S($5R<FYO
M+G!M"FQA<W0@34%'24,["GT*;7D@)&UA9VEC7W!O<R`](&9I;F1?<&%R7VUA
M9VEC*"I?1D@I.PII9B`H)&UA9VEC7W!O<R`\(#`I('L*;W5T<RAQ<5M#86XG
M="!F:6YD(&UA9VEC('-T<FEN9R`B)%!!4E]-04=)0R(@:6X@9FEL92`B)'!R
M;V=N86UE(ETI.PIL87-T($U!1TE#.PI]"F]U=',H(D9O=6YD(%!!4B!M86=I
M8R!A="!P;W-I=&EO;B`D;6%G:6-?<&]S(BD["G-E96L@7T9(+"`D;6%G:6-?
M<&]S("T@)$9)3$5?;V9F<V5T7W-I>F4@+2!L96YG=&@H(EPP0T%#2$4B*2P@
M,#L*<F5A9"!?1D@L("1B=68L(&QE;F=T:"@B7#!#04-(12(I.PII9B`H)&)U
M9B!N92`B7#!#04-(12(@)B8@)&)U9B!N92`B7#!#3$5!3B(I('L*;W5T<R@B
M3F\@8V%C:&4@;6%R:V5R(&9O=6YD(BD["FQA<W0@34%'24,["GT*;W5T<RAQ
M<5M#86-H92!M87)K97(@(B1B=68B(&9O=6YD72D["G-E96L@7T9(+"`D;6%G
M:6-?<&]S("T@)$9)3$5?;V9F<V5T7W-I>F4L(#`["G)E860@7T9(+"`D8G5F
M+"`D1DE,15]O9F9S971?<VEZ93L*;7D@)&]F9G-E="`]('5N<&%C:R@B3B(L
M("1B=68I.PIO=71S*")/9F9S970@9G)O;2!S=&%R="!O9B!&24Q%<R!I<R`D
M;V9F<V5T(BD["G-E96L@7T9(+"`D;6%G:6-?<&]S("T@)$9)3$5?;V9F<V5T
M7W-I>F4@+2`D;V9F<V5T+"`P.PHD<W1A<G1?;V9?1DE,15]S96-T:6]N(#T@
M=&5L;"!?1D@["FUY("5R97%U:7)E7VQI<W0["G)E860@7T9(+"`D8G5F+"`T
M.R`@("`@("`@("`@("`@("`@("`@("`@("`@(",@<F5A9"!T:&4@9FER<W0@
M(D9)3$4B"G=H:6QE("@D8G5F(&5Q(")&24Q%(BD@>PIR96%D(%]&2"P@)&)U
M9BP@-#L*<F5A9"!?1D@L("1B=68L('5N<&%C:R@B3B(L("1B=68I.PIM>2`D
M9G5L;&YA;64@/2`D8G5F.PIO=71S*'%Q6U5N<&%C:VEN9R!&24Q%("(D9G5L
M;&YA;64B+BXN72D["FUY("1C<F,@/2`H("1F=6QL;F%M92`]?B!S?%XH6V$M
M9EQD77LX?2DO?'P@*2`_("0Q(#H@=6YD968["FUY("@D8F%S96YA;64L("1E
M>'0I(#T@*"1B=68@/7X@;7PH/SHN*B\I/R@N*BDH7"XN*BE\*3L*<F5A9"!?
M1D@L("1B=68L(#0["G)E860@7T9(+"`D8G5F+"!U;G!A8VLH(DXB+"`D8G5F
M*3L*:68@*&1E9FEN960H)&5X="D@86YD("1E>'0@(7X@+UPN*#\Z<&U\<&Q\
M:7A\86PI)"]I*2!["FUY("1F:6QE;F%M92`](%]S879E7V%S*"(D8W)C)&5X
M="(L("1B=68L(#`W-34I.PHD4$%2.CI(96%V>3HZ1G5L;$-A8VAE>R1F=6QL
M;F%M97T@/2`D9FEL96YA;64["B1005(Z.DAE879Y.CI&=6QL0V%C:&5[)&9I
M;&5N86UE?2`]("1F=6QL;F%M93L*?0IE;'-E('L*)$UO9'5L94-A8VAE>R1F
M=6QL;F%M97T@/2!["F)U9B`]/B`D8G5F+`IC<F,@/3X@)&-R8RP*;F%M92`]
M/B`D9G5L;&YA;64L"GT["B1R97%U:7)E7VQI<W1[)&9U;&QN86UE?2LK.PI]
M"G)E860@7T9(+"`D8G5F+"`T.PI]"FQO8V%L($!)3D,@/2`H<W5B('L*;7D@
M*"1S96QF+"`D;6]D=6QE*2`]($!?.PIR971U<FX@:68@<F5F("1M;V1U;&4@
M;W(@(21M;V1U;&4["FUY("1I;F9O(#T@)$UO9'5L94-A8VAE>R1M;V1U;&5]
M.PIR971U<FX@=6YL97-S("1I;F9O.PID96QE=&4@)')E<75I<F5?;&ES='LD
M;6]D=6QE?3L*)$E.0WLD;6]D=6QE?2`]("(O;&]A9&5R+R1I;F9O+R1M;V1U
M;&4B.PII9B`H)$5.5GM005)?0TQ%04Y](&%N9"!D969I;F5D*"9)3SHZ1FEL
M93HZ;F5W*2D@>PIM>2`D9F@@/2!)3SHZ1FEL92T^;F5W7W1M<&9I;&4@;W(@
M9&EE(")#86XG="!C<F5A=&4@=&5M<"!F:6QE.B`D(2(["B1F:"T^8FEN;6]D
M92@I.PHD9F@M/G!R:6YT*"1I;F9O+3Y[8G5F?2D["B1F:"T^<V5E:R@P+"`P
M*3L*<F5T=7)N("1F:#L*?0IE;'-E('L*;7D@)&9I;&5N86UE(#T@7W-A=F5?
M87,H(B1I;F9O+3Y[8W)C?2YP;2(L("1I;F9O+3Y[8G5F?2D["B1I;F9O+3Y[
M9FEL97T@/2`D9FEL96YA;64["F]P96X@;7D@)&9H+"`G/#IR87<G+"`D9FEL
M96YA;64@;W(@9&EE('%Q6T-A;B=T(')E860@(B1F:6QE;F%M92(Z("0A73L*
M<F5T=7)N("1F:#L*?0ID:64@(D)O;W1S=')A<'!I;F<@9F%I;&5D.B!C86XG
M="!F:6YD(&UO9'5L92`D;6]D=6QE(2(["GTL($!)3D,I.PIR97%U:7)E(%A3
M3&]A9&5R.PIR97%U:7)E(%!!4CHZ2&5A=GD["G)E<75I<F4@0V%R<#HZ2&5A
M=GD["G)E<75I<F4@17AP;W)T97(Z.DAE879Y.PI005(Z.DAE879Y.CI?:6YI
M=%]D>6YA;&]A9&5R*"D["G)E<75I<F4@24\Z.D9I;&4["G=H:6QE("AM>2`D
M9FEL96YA;64@/2`H<V]R="!K97ES("5R97%U:7)E7VQI<W0I6S!=*2!["G5N
M;&5S<R`H)$E.0WLD9FEL96YA;65](&]R("1F:6QE;F%M92`]?B`O0E-$4$%.
M+RD@>PII9B`H)&9I;&5N86UE(#U^("]<+G!M8S\D+VDI('L*<F5Q=6ER92`D
M9FEL96YA;64["GT*96QS92!["F1O("1F:6QE;F%M92!U;FQE<W,@)&9I;&5N
M86UE(#U^("]S:71E8W5S=&]M:7IE7"YP;"0O.PI]"GT*9&5L971E("1R97%U
M:7)E7VQI<W1[)&9I;&5N86UE?3L*?0II9B`H)&)U9B!N92`D4$M:25!?34%'
M24,I('L*;W5T<RAQ<5M.;R!Z:7`@9F]U;F0@869T97(@1DE,12!S96-T:6]N
M(&EN(&9I;&4@(B1P<F]G;F%M92)=*3L*;&%S="!-04=)0R`["GT*)'-T87)T
M7W!O<R`]("AT96QL(%]&2"D@+2`T.R`@("`@("`@("`@("`@("`C('-T87)T
M(&]F('II<`I]"FUY($!P87)?87)G<SL*;7D@*"1O=70L("1B=6YD;&4L("1L
M;V=F:"P@)&-A8VAE7VYA;64I.PID96QE=&4@)$5.5GM005)?05!07U)%55-%
M?3L@(R!S86YI=&EZ92`H4D554T4@;6%Y(&)E(&$@<V5C=7)I='D@<')O8FQE
M;2D*:68@*"$D<W1A<G1?<&]S(&]R("@D05)'5ELP72!E<2`G+2UP87(M;W!T
M:6]N<R<@)B8@<VAI9G0I*2!["FUY("5D:7-T7V-M9"`]('%W*`IP("`@8FQI
M8E]T;U]P87(*:2`@(&EN<W1A;&Q?<&%R"G4@("!U;FEN<W1A;&Q?<&%R"G,@
M("!S:6=N7W!A<@IV("`@=F5R:69Y7W!A<@HI.PII9B`H0$%21U8@86YD("1!
M4D=66S!=(&5Q("<M+7)E=7-E)RD@>PIS:&EF="!`05)'5CL*)$5.5GM005)?
M05!07U)%55-%?2`]('-H:69T($!!4D=6.PI]"F5L<V4@>R`C(&YO<FUA;"!P
M87)L(&)E:&%V:6]U<@IM>2!`861D7W1O7VEN8SL*=VAI;&4@*$!!4D=6*2![
M"B1!4D=66S!=(#U^("]>+2A;04E-3T),8G%P:75S5'9=*2@N*BDO(&]R(&QA
M<W0["FEF("@D,2!E<2`G22<I('L*<'5S:"!`861D7W1O7VEN8RP@)#(["GT*
M96QS:68@*"0Q(&5Q("=-)RD@>PIE=F%L(")U<V4@)#(B.PI]"F5L<VEF("@D
M,2!E<2`G02<I('L*=6YS:&EF="!`<&%R7V%R9W,L("0R.PI]"F5L<VEF("@D
M,2!E<2`G3R<I('L*)&]U="`]("0R.PI]"F5L<VEF("@D,2!E<2`G8B<I('L*
M)&)U;F1L92`]("=S:71E)SL*?0IE;'-I9B`H)#$@97$@)T(G*2!["B1B=6YD
M;&4@/2`G86QL)SL*?0IE;'-I9B`H)#$@97$@)W$G*2!["B1Q=6EE="`](#$[
M"GT*96QS:68@*"0Q(&5Q("=,)RD@>PIO<&5N("1L;V=F:"P@(CX^(BP@)#(@
M;W(@9&EE('%Q6T-A;B=T(&]P96X@;&]G(&9I;&4@(B0R(CH@)"%=.PI]"F5L
M<VEF("@D,2!E<2`G5"<I('L*)&-A8VAE7VYA;64@/2`D,CL*?0IS:&EF="A`
M05)'5BD["FEF("AM>2`D8VUD(#T@)&1I<W1?8VUD>R0Q?2D@>PID96QE=&4@
M)$5.5GLG4$%27U1%35`G?3L*:6YI=%]I;F,H*3L*<F5Q=6ER92!005(Z.D1I
M<W0["B9[(E!!4CHZ1&ES=#HZ)&-M9")]*"D@=6YL97-S($!!4D=6.PHF>R)0
M05(Z.D1I<W0Z.B1C;60B?2@D7RD@9F]R($!!4D=6.PIE>&ET.PI]"GT*=6YS
M:&EF="!`24Y#+"!`861D7W1O7VEN8SL*?0I]"FEF("@D;W5T*2!["GL*<F5Q
M=6ER92!)3SHZ1FEL93L*<F5Q=6ER92!!<F-H:79E.CI::7`["G)E<75I<F4@
M1&EG97-T.CI32$$["GT*;7D@)'!A<B`]('-H:69T*$!!4D=6*3L*;7D@)'II
M<#L*:68@*&1E9FEN960@)'!A<BD@>PIO<&5N(&UY("1F:"P@)SPZ<F%W)RP@
M)'!A<B!O<B!D:64@<7%;0V%N)W0@9FEN9"!P87(@9FEL92`B)'!A<B(Z("0A
M73L*8FQE<W,H)&9H+"`G24\Z.D9I;&4G*3L*)'II<"`]($%R8VAI=F4Z.EII
M<"T^;F5W.PHH("1Z:7`M/G)E861&<F]M1FEL94AA;F1L92@D9F@L("1P87(I
M(#T]($%R8VAI=F4Z.EII<#HZ05I?3TLH*2`I"F]R(&1I92!Q<5M%<G)O<B!R
M96%D:6YG('II<"!A<F-H:79E("(D<&%R(ET["GT*;7D@)6UE=&%?<&%R(#T@
M9&\@>PII9B`H)'II<"!A;F0@;7D@)&UE=&$@/2`D>FEP+3YC;VYT96YT<R@G
M345402YY;6PG*2D@>PHD;65T82`]?B!S+RXJ7G!A<CHD+R]M<SL*)&UE=&$@
M/7X@<R]>7%,N*B\O;7,["B1M971A(#U^("]>("`H6UXZ72LI.B`H+BLI)"]M
M9SL*?0I].PII9B`H9&5F:6YE9"`D<&%R*2!["F]P96X@;7D@)'!H+"`G/#IR
M87<G+"`D<&%R(&]R(&1I92!Q<5M#86XG="!R96%D('!A<B!F:6QE("(D<&%R
M(CH@)"%=.PIM>2`D8G5F.PIR96%D("1P:"P@)&)U9BP@-#L*9&EE('%Q6R(D
M<&%R(B!I<R!N;W0@82!P87(@9FEL95T@=6YL97-S("1B=68@97$@)%!+6DE0
M7TU!1TE#.PIC;&]S92`D<&@["GT*0W)E871E4&%T:"@D;W5T*2`["FUY("1F
M:"`]($E/.CI&:6QE+3YN97<H"B1O=70L"DE/.CI&:6QE.CI/7T-214%4*"D@
M?"!)3SHZ1FEL93HZ3U]21%=2*"D@?"!)3SHZ1FEL93HZ3U]44E5.0R@I+`HP
M-S<W+`HI(&]R(&1I92!Q<5M#86XG="!C<F5A=&4@9FEL92`B)&]U="(Z("0A
M73L*)&9H+3YB:6YM;V1E*"D["G-E96L@7T9(+"`P+"`P.PIM>2`D;&]A9&5R
M.PII9B`H9&5F:6YE9"`D<W1A<G1?;V9?1DE,15]S96-T:6]N*2!["G)E860@
M7T9(+"`D;&]A9&5R+"`D<W1A<G1?;V9?1DE,15]S96-T:6]N.PI](&5L<V4@
M>PIL;V-A;"`D+R`]('5N9&5F.PHD;&]A9&5R(#T@/%]&2#X["GT*:68@*"$D
M14Y6>U!!4E]615)"051)37T@86YD("1L;V%D97(@/7X@+UXH/SHC(7Q<0')E
M;2DO*2!["G)E<75I<F4@4$%2.CI&:6QT97(Z.E!O9%-T<FEP.PI005(Z.D9I
M;'1E<CHZ4&]D4W1R:7`M/F%P<&QY*%PD;&]A9&5R+"`D,"D["GT*)&9H+3YP
M<FEN="@D;&]A9&5R*0IO<B!D:64@<7%;17)R;W(@=W)I=&EN9R!L;V%D97(@
M=&\@(B1O=70B.B`D(5T["FEF("@D8G5N9&QE*2!["G)E<75I<F4@4$%2.CI(
M96%V>3L*4$%2.CI(96%V>3HZ7VEN:71?9'EN86QO861E<B@I.PII;FET7VEN
M8R@I.PIR97%U:7)E7VUO9'5L97,H*3L*;7D@0&EN8R`](&=R97`@>R`A+T)3
M1%!!3B\@?0IG<F5P('L**"1B=6YD;&4@;F4@)W-I=&4G*2!O<@HH)%\@;F4@
M)$-O;F9I9SHZ0V]N9FEG>V%R8VAL:6)E>'!](&%N9`HD7R!N92`D0V]N9FEG
M.CI#;VYF:6=[<')I=FQI8F5X<'TI.PI]($!)3D,["G-\+RM\+WQG+"!S?"\D
M?'P@9F]R96%C:"!`:6YC.PIM>2`E9FEL97,["B1F:6QE<WLD7WTK*R!F;W(@
M0$1Y;F%,;V%D97(Z.F1L7W-H87)E9%]O8FIE8W1S+"!V86QU97,@)4E.0SL*
M;7D@)&QI8E]E>'0@/2`D0V]N9FEG.CI#;VYF:6=[;&EB7V5X='T[("`@("`@
M("`@(R!86%@@;&EB7V5X="!V<R!D;&5X="`_"FUY("5W<FET=&5N.PIF;W)E
M86-H(&UY("1K97D@*'-O<G0@:V5Y<R`E9FEL97,I('L*;7D@*"1F:6QE+"`D
M;F%M92D["FEF("AD969I;F5D*&UY("1F8R`]("1005(Z.DAE879Y.CI&=6QL
M0V%C:&5[)&ME>7TI*2!["B@D9FEL92P@)&YA;64I(#T@*"1K97DL("1F8RD[
M"GT*96QS92!["F9O<F5A8V@@;7D@)&1I<B`H0&EN8RD@>PII9B`H)&ME>2`]
M?B!M?%Y<421D:7)<12\H+BHI)'QI*2!["B@D9FEL92P@)&YA;64I(#T@*"1K
M97DL("0Q*3L*;&%S=#L*?0II9B`H)&ME>2`]?B!M?%XO;&]A9&5R+UM>+UTK
M+R@N*BDD?"D@>PII9B`H;7D@)')E9B`]("1-;V1U;&5#86-H97LD,7TI('L*
M*"1F:6QE+"`D;F%M92D@/2`H)')E9BP@)#$I.PIL87-T.PI]"FEF("@M9B`B
M)&1I<B\D,2(I('L**"1F:6QE+"`D;F%M92D@/2`H(B1D:7(O)#$B+"`D,2D[
M"FQA<W0["GT*?0I]"GT*;F5X="!U;FQE<W,@9&5F:6YE9"`D;F%M93L*;F5X
M="!I9B`D=W)I='1E;GLD;F%M97TK*SL*;F5X="!I9B`A<F5F*"1F:6QE*2!A
M;F0@)&9I;&4@/7X@+UPN7%$D;&EB7V5X=%Q%)"]I.PIO=71S*'-P<FEN=&8H
M<7%;4&%C:VEN9R!&24Q%("(E<R(N+BY=+"!R968@)&9I;&4@/R`D9FEL92T^
M>VYA;65](#H@)&9I;&4I*3L*;7D@)&-O;G1E;G0["FEF("AR968H)&9I;&4I
M*2!["B1C;VYT96YT(#T@)&9I;&4M/GMB=69].PI]"F5L<V4@>PIL;V-A;"`D
M+R`]('5N9&5F.PIO<&5N(&UY("1T:"P@)SPZ<F%W)RP@)&9I;&4@;W(@9&EE
M('%Q6T-A;B=T(')E860@(B1F:6QE(CH@)"%=.PHD8V]N=&5N="`](#PD=&@^
M.PIC;&]S92`D=&@["E!!4CHZ1FEL=&5R.CI0;V13=')I<"T^87!P;'DH7"1C
M;VYT96YT+"`B/&5M8F5D9&5D/B\D;F%M92(I"FEF("$D14Y6>U!!4E]615)"
M051)37T@86YD("1N86UE(#U^("]<+B@_.G!M?&EX?&%L*20O:3L*4$%2.CI&
M:6QT97(Z.E!A=&-H0V]N=&5N="T^;F5W+3YA<'!L>2A<)&-O;G1E;G0L("1F
M:6QE+"`D;F%M92D["GT*)&9H+3YP<FEN="@B1DE,12(L"G!A8VLH)TXG+"!L
M96YG=&@H)&YA;64I("L@.2DL"G-P<FEN=&8H(B4P.'@O)7,B+"!!<F-H:79E
M.CI::7`Z.F-O;7!U=&5#4D,S,B@D8V]N=&5N="DL("1N86UE*2P*<&%C:R@G
M3B<L(&QE;F=T:"@D8V]N=&5N="DI+`HD8V]N=&5N="D*;W(@9&EE('%Q6T5R
M<F]R('=R:71I;F<@96UB961D960@1DE,12!T;R`B)&]U="(Z("0A73L*;W5T
M<RAQ<5M7<FET=&5N(&%S("(D;F%M92)=*3L*?0I]"FEF("@D>FEP*2!["B1Z
M:7`M/G=R:71E5&]&:6QE2&%N9&QE*"1F:"D@/3T@07)C:&EV93HZ6FEP.CI!
M6E]/2PIO<B!D:64@<7%;17)R;W(@=W)I=&EN9R!Z:7`@<&%R="!O9B`B)&]U
M=")=.PI]"FEF("@D;65T85]P87)[8VQE86Y]*2!["B1F:"T^<')I;G0H(EPP
M0TQ%04XB*3L*?0IE;'-E('L*:68@*"%D969I;F5D("1C86-H95]N86UE*2![
M"FUY("1C='@@/2!$:6=E<W0Z.E-(02T^;F5W*#$I.PIS965K("1F:"P@,"P@
M,#L*)&-T>"T^861D9FEL92@D9F@I.PIS965K("1F:"P@,"P@,CL*)&-A8VAE
M7VYA;64@/2`D8W1X+3YH97AD:6=E<W0["GT*)&-A8VAE7VYA;64@/2!S=6)S
M='(@)&-A8VAE7VYA;64L(#`L("1C86-H95]N86UE7W-I>F4["B1C86-H95]N
M86UE("X](")<,"(@>"`H)&-A8VAE7VYA;65?<VEZ92`M(&QE;F=T:"`D8V%C
M:&5?;F%M92D["B1F:"T^<')I;G0H)&-A8VAE7VYA;64L"B)<,$-!0TA%(BD[
M"GT*;7D@)&]F9G-E="`]("1F:"T^=&5L;"`M(&QE;F=T:"@D;&]A9&5R*3L*
M)&9H+3YP<FEN="AP86-K*"=.)RP@)&]F9G-E="DL"B1005)?34%'24,I.PHD
M9F@M/F-L;W-E(&]R(&1I92!Q<5M%<G)O<B!W<FET:6YG('1R86EL97(@;V8@
M(B1O=70B.B`D(5T["F-H;6]D(#`W-34L("1O=70["F5X:70["GT*>PIL87-T
M('5N;&5S<R!D969I;F5D("1S=&%R=%]P;W,["E]F:7A?<')O9VYA;64H*3L*
M<F5Q=6ER92!005(["E!!4CHZ2&5A=GDZ.E]I;FET7V1Y;F%L;V%D97(H*3L*
M>PIR97%U:7)E($9I;&4Z.D9I;F0["G)E<75I<F4@07)C:&EV93HZ6FEP.PI]
M"FUY("1F:"`]($E/.CI&:6QE+3YN97<[("`@("`@("`@("`@("`@("`@("`@
M("`@("`@("`C($%R8VAI=F4Z.EII<"!O<&5R871E<R!O;B!A;B!)3SHZ2&%N
M9&QE"B1F:"T^9F1O<&5N*&9I;&5N;RA?1D@I+"`G<B<I(&]R(&1I92!Q<5MF
M9&]P96XH*2!F86EL960Z("0A73L*07)C:&EV93HZ6FEP.CIS971#:'5N:U-I
M>F4H+7,@7T9(*3L*;7D@)'II<"`]($%R8VAI=F4Z.EII<"T^;F5W.PHH)'II
M<"T^<F5A9$9R;VU&:6QE2&%N9&QE*"1F:"P@)'!R;V=N86UE*2`]/2!!<F-H
M:79E.CI::7`Z.D%:7T]+*"DI"F]R(&1I92!Q<5M%<G)O<B!R96%D:6YG('II
M<"!A<F-H:79E("(D<')O9VYA;64B73L*07)C:&EV93HZ6FEP.CIS971#:'5N
M:U-I>F4H-C0@*B`Q,#(T*3L*<'5S:"!`4$%2.CI,:6)#86-H92P@)'II<#L*
M)%!!4CHZ3&EB0V%C:&5[)'!R;V=N86UE?2`]("1Z:7`["F]U=',H(D5X=')A
M8W1I;F<@>FEP+BXN(BD["FEF("AD969I;F5D("1%3E9[4$%27U1%35!]*2![
M(",@<VAO=6QD(&)E('-E="!A="!T:&ES('!O:6YT(0IF;W)E86-H(&UY("1M
M96UB97(@*"`D>FEP+3YM96UB97)S("D@>PIN97AT(&EF("1M96UB97(M/FES
M1&ER96-T;W)Y.PIM>2`D;65M8F5R7VYA;64@/2`D;65M8F5R+3YF:6QE3F%M
M93L*;F5X="!U;FQE<W,@)&UE;6)E<E]N86UE(#U^(&U["EX*+S]S:&QI8B\*
M*#\Z)$-O;F9I9SHZ0V]N9FEG>V%R8VAN86UE?2\I/PHH6UXO72LI"B0*?7@[
M"FUY("1E>'1R86-T7VYA;64@/2`D,3L*;7D@)&1E<W1?;F%M92`]($9I;&4Z
M.E-P96,M/F-A=&9I;&4H)$5.5GM005)?5$5-4'TL("1E>'1R86-T7VYA;64I
M.PII9B`H+68@)&1E<W1?;F%M92`F)B`M<R!?(#T]("1M96UB97(M/G5N8V]M
M<')E<W-E9%-I>F4H*2D@>PIO=71S*'%Q6U-K:7!P:6YG("(D;65M8F5R7VYA
M;64B('-I;F-E(&ET(&%L<F5A9'D@97AI<W1S(&%T("(D9&5S=%]N86UE(ETI
M.PI](&5L<V4@>PIO=71S*'%Q6T5X=')A8W1I;F<@(B1M96UB97)?;F%M92(@
M=&\@(B1D97-T7VYA;64B+BXN72D["B@D;65M8F5R+3YE>'1R86-T5&]&:6QE
M3F%M960H)&1E<W1?;F%M92D@/3T@07)C:&EV93HZ6FEP.CI!6E]/2R@I*0IO
M<B!D:64@<7%;17)R;W(@97AT<F%C=&EN9R!Z:7`@;65M8F5R('1O("(D9&5S
M=%]N86UE(ET["F-H;6]D*#`U-34L("1D97-T7VYA;64I(&EF("1>3R!E<2`B
M:'!U>"(["GT*?0I]"GT*=6YL97-S("@D4$%2.CI,:6)#86-H97LD<')O9VYA
M;65]*2!["F1I92`\/"`B+B(@=6YL97-S($!!4D=6.PI5<V%G93H@)#`@6R`M
M06QI8BYP87(@72!;("U)9&ER(%T@6R`M36UO9'5L92!=(%L@<W)C+G!A<B!=
M(%L@<')O9W)A;2YP;"!="B0P(%L@+4)\+6(@72!;+4]O=71F:6QE72!S<F,N
M<&%R"BX*)$5.5GM005)?4%)/1TY!345](#T@)'!R;V=N86UE(#T@)#`@/2!S
M:&EF="A`05)'5BD["GT*<W5B($-R96%T95!A=&@@>PIM>2`H)&YA;64I(#T@
M0%\["G)E<75I<F4@1FEL93HZ0F%S96YA;64["FUY("@D8F%S96YA;64L("1P
M871H+"`D97AT*2`]($9I;&4Z.D)A<V5N86UE.CIF:6QE<&%R<V4H)&YA;64L
M("@G7"XN*B<I*3L*<F5Q=6ER92!&:6QE.CI0871H.PI&:6QE.CI0871H.CIM
M:W!A=&@H)'!A=&@I('5N;&5S<R@M92`D<&%T:"D[(",@;6MP871H(&1I97,@
M=VET:"!E<G)O<@I]"G-U8B!R97%U:7)E7VUO9'5L97,@>PIR97%U:7)E(&QI
M8CL*<F5Q=6ER92!$>6YA3&]A9&5R.PIR97%U:7)E(&EN=&5G97(["G)E<75I
M<F4@<W1R:6-T.PIR97%U:7)E('=A<FYI;F=S.PIR97%U:7)E('9A<G,["G)E
M<75I<F4@0V%R<#L*<F5Q=6ER92!#87)P.CI(96%V>3L*<F5Q=6ER92!%<G)N
M;SL*<F5Q=6ER92!%>'!O<G1E<CHZ2&5A=GD["G)E<75I<F4@17AP;W)T97([
M"G)E<75I<F4@1F-N=&P["G)E<75I<F4@1FEL93HZ5&5M<#L*<F5Q=6ER92!&
M:6QE.CI3<&5C.PIR97%U:7)E(%A33&]A9&5R.PIR97%U:7)E($-O;F9I9SL*
M<F5Q=6ER92!)3SHZ2&%N9&QE.PIR97%U:7)E($E/.CI&:6QE.PIR97%U:7)E
M($-O;7!R97-S.CI:;&EB.PIR97%U:7)E($%R8VAI=F4Z.EII<#L*<F5Q=6ER
M92!$:6=E<W0Z.E-(03L*<F5Q=6ER92!005(["G)E<75I<F4@4$%2.CI(96%V
M>3L*<F5Q=6ER92!005(Z.D1I<W0["G)E<75I<F4@4$%2.CI&:6QT97(Z.E!O
M9%-T<FEP.PIR97%U:7)E(%!!4CHZ1FEL=&5R.CI0871C:$-O;G1E;G0["G)E
M<75I<F4@871T<FEB=71E<SL*979A;"![(')E<75I<F4@0W=D('T["F5V86P@
M>R!R97%U:7)E(%=I;C,R('T["F5V86P@>R!R97%U:7)E(%-C86QA<CHZ571I
M;"!].PIE=F%L('L@<F5Q=6ER92!!<F-H:79E.CI5;GII<#HZ0G5R<W0@?3L*
M979A;"![(')E<75I<F4@5&EE.CI(87-H.CI.86UE9$-A<'1U<F4@?3L*979A
M;"![(')E<75I<F4@4&5R;$E/.R!R97%U:7)E(%!E<FQ)3SHZ<V-A;&%R('T[
M"F5V86P@>R!R97%U:7)E('5T9C@@?3L*?0IS=6(@7W-E=%]P87)?=&5M<"![
M"FEF("AD969I;F5D("1%3E9[4$%27U1%35!](&%N9"`D14Y6>U!!4E]414U0
M?2`]?B`O*"XK*2\I('L*)'!A<E]T96UP(#T@)#$["G)E='5R;CL*?0IF;W)E
M86-H(&UY("1P871H("@**&UA<"`D14Y6>R1??2P@<7<H(%!!4E]435!$25(@
M5$U01$E2(%1%35!$25(@5$5-4"!435`@*2DL"G%W*"!#.EQ<5$5-4"`O=&UP
M("X@*0HI('L*;F5X="!U;FQE<W,@9&5F:6YE9"`D<&%T:"!A;F0@+60@)'!A
M=&@@86YD("UW("1P871H.PIM>2`D=7-E<FYA;64["FUY("1P=W5I9#L*979A
M;"![*"1P=W5I9"D@/2!G971P=W5I9"@D/BD@:68@9&5F:6YE9"`D/CM].PII
M9B`H(&1E9FEN960H)E=I;C,R.CI,;V=I;DYA;64I("D@>PHD=7-E<FYA;64@
M/2`F5VEN,S(Z.DQO9VEN3F%M93L*?0IE;'-I9B`H9&5F:6YE9"`D<'=U:60I
M('L*)'5S97)N86UE(#T@)'!W=6ED.PI]"F5L<V4@>PHD=7-E<FYA;64@/2`D
M14Y6>U5315).04U%?2!\?"`D14Y6>U5315)]('Q\("=365-414TG.PI]"FUY
M("1S=&UP9&ER(#T@(B1P871H)'-Y<WM?9&5L:6U]<&%R+2(N=6YP86-K*")(
M*B(L("1U<V5R;F%M92D["FUK9&ER("1S=&UP9&ER+"`P-S4U.PIM>2`D8V%C
M:&5?9&ER.PII9B`H)$5.5GM005)?0TQ%04Y]*2!["B1C86-H95]D:7(@/2`B
M=&5M<"TD)"(["GT*96QS92!["F]P96X@;7D@)&9H+"`B/#IR87<B+"`D<')O
M9VYA;64@;W(@9&EE('%Q6T-A;B=T(')E860@(B1P<F]G;F%M92(Z("0A73L*
M:68@*"AM>2`D;6%G:6-?<&]S(#T@9FEN9%]P87)?;6%G:6,H)&9H*2D@/CT@
M,"D@>PIS965K("1F:"P@)&UA9VEC7W!O<PHM("1&24Q%7V]F9G-E=%]S:7IE
M"BT@;&5N9W1H*")<,$-!0TA%(BDL(#`["FUY("1B=68["G)E860@)&9H+"`D
M8G5F+"!L96YG=&@H(EPP0T%#2$4B*3L*:68@*"1B=68@97$@(EPP0T%#2$4B
M*2!["G-E96L@)&9H+"`D;6%G:6-?<&]S"BT@)$9)3$5?;V9F<V5T7W-I>F4*
M+2!L96YG=&@H(EPP0T%#2$4B*0HM("1C86-H95]N86UE7W-I>F4L(#`["G)E
M860@)&9H+"`D8G5F+"`D8V%C:&5?;F%M95]S:7IE.PHD8G5F(#U^(',O7#`O
M+V<["B1C86-H95]D:7(@/2`B8V%C:&4M)&)U9B(["GT*?0IC;&]S92`D9F@[
M"GT*:68@*"$D8V%C:&5?9&ER*2!["B1C86-H95]D:7(@/2`B=&5M<"TD)"([
M"B1%3E9[4$%27T-,14%.?2`](#$["GT*)'-T;7!D:7(@+CT@(B1S>7-[7V1E
M;&EM?21C86-H95]D:7(B.PIM:V1I<B`D<W1M<&1I<BP@,#<U-3L*)$5.5GM0
M05)?5$5-4'T@/2`D<W1M<&1I<CL*;&%S=#L*?0HD<&%R7W1E;7`@/2`D,2!I
M9B`D14Y6>U!!4E]414U0?2!A;F0@)$5.5GM005)?5$5-4'T@/7X@+R@N*RDO
M.PI]"G-U8B!?<V%V95]A<R!["FUY("@D;F%M92P@)&-O;G1E;G1S+"`D;6]D
M92D@/2!`7SL*;7D@)&9U;&QN86UE(#T@(B1P87)?=&5M<"\D;F%M92(["G5N
M;&5S<R`H+64@)&9U;&QN86UE*2!["FUY("1T96UP;F%M92`]("(D9G5L;&YA
M;64N)"0B.PIM>2`D9F@["BAO<&5N("1F:"P@)SXZ<F%W)RP@)'1E;7!N86UE
M*0HF)B`H<')I;G0@)&9H("1C;VYT96YT<RD*)B8@*&-L;W-E("1F:"D*;W(@
M9&EE('%Q6T5R<F]R('=R:71I;F<@(B1T96UP;F%M92(Z("0A73L*8VAM;V0@
M)&UO9&4L("1T96UP;F%M92!I9B!D969I;F5D("1M;V1E.PIR96YA;64H)'1E
M;7!N86UE+"`D9G5L;&YA;64I(&]R('5N;&EN:R@D=&5M<&YA;64I.PI]"G)E
M='5R;B`D9G5L;&YA;64["GT*<W5B(%]S971?<')O9VYA;64@>PII9B`H9&5F
M:6YE9"`D14Y6>U!!4E]04D]'3D%-17T@86YD("1%3E9[4$%27U!23T=.04U%
M?2`]?B`O*"XK*2\I('L*)'!R;V=N86UE(#T@)#$["GT*)'!R;V=N86UE('Q\
M/2`D,#L*:68@*"1%3E9[4$%27U1%35!](&%N9"!I;F1E>"@D<')O9VYA;64L
M("1%3E9[4$%27U1%35!]*2`^/2`P*2!["B1P<F]G;F%M92`]('-U8G-T<B@D
M<')O9VYA;64L(')I;F1E>"@D<')O9VYA;64L("1S>7-[7V1E;&EM?2D@*R`Q
M*3L*?0II9B`H(21%3E9[4$%27U!23T=.04U%?2!O<B!I;F1E>"@D<')O9VYA
M;64L("1S>7-[7V1E;&EM?2D@/CT@,"D@>PII9B`H;W!E;B!M>2`D9F@L("<\
M)RP@)'!R;V=N86UE*2!["G)E='5R;B!I9B`M<R`D9F@["GT*:68@*"US("(D
M<')O9VYA;64D<WES>U]E>&5](BD@>PHD<')O9VYA;64@+CT@)'-Y<WM?97AE
M?3L*<F5T=7)N.PI]"GT*9F]R96%C:"!M>2`D9&ER("AS<&QI="`O7%$D<WES
M>W!A=&A?<V5P?5Q%+RP@)$5.5GM0051(?2D@>PIN97AT(&EF(&5X:7-T<R`D
M14Y6>U!!4E]414U0?2!A;F0@)&1I<B!E<2`D14Y6>U!!4E]414U0?3L*)&1I
M<B`]?B!S+UQ1)'-Y<WM?9&5L:6U]7$4D+R\["B@H)'!R;V=N86UE(#T@(B1D
M:7(D<WES>U]D96QI;7TD<')O9VYA;64D<WES>U]E>&5](BDL(&QA<W0I"FEF
M("US("(D9&ER)'-Y<WM?9&5L:6U])'!R;V=N86UE)'-Y<WM?97AE?2(["B@H
M)'!R;V=N86UE(#T@(B1D:7(D<WES>U]D96QI;7TD<')O9VYA;64B*2P@;&%S
M="D*:68@+7,@(B1D:7(D<WES>U]D96QI;7TD<')O9VYA;64B.PI]"GT*<W5B
M(%]F:7A?<')O9VYA;64@>PHD,"`]("1P<F]G;F%M92!\?#T@)$5.5GM005)?
M4%)/1TY!345].PII9B`H:6YD97@H)'!R;V=N86UE+"`D<WES>U]D96QI;7TI
M(#P@,"D@>PHD<')O9VYA;64@/2`B+B1S>7-[7V1E;&EM?21P<F]G;F%M92([
M"GT*;7D@)'!W9"`]("AD969I;F5D("9#=V0Z.F=E=&-W9"D@/R!#=V0Z.F=E
M=&-W9"@I"CH@*"AD969I;F5D("97:6XS,CHZ1V5T0W=D*2`_(%=I;C,R.CI'
M971#=V0H*2`Z(&!P=V1@*3L*8VAO;7`H)'!W9"D["B1P<F]G;F%M92`]?B!S
M+UXH/SU<+EPN/UQ1)'-Y<WM?9&5L:6U]7$4I+R1P=V0D<WES>U]D96QI;7TO
M.PHD14Y6>U!!4E]04D]'3D%-17T@/2`D<')O9VYA;64["GT*<W5B(%]P87)?
M:6YI=%]E;G8@>PII9B`H("1%3E9[4$%27TE.251)04Q)6D5$?2LK(#T](#$@
M*2!["G)E='5R;CL*?2!E;'-E('L*)$5.5GM005)?24Y)5$E!3$E:141](#T@
M,CL*?0IF;W(@*'%W*"!34$%73D5$(%1%35`@0TQ%04X@1$5"54<@0T%#2$4@
M4%)/1TY!344@*2`I('L*9&5L971E("1%3E9[(E!!4E\D7R)].PI]"F9O<B`H
M<7<O(%1-4$1)4B!414U0($-,14%.($1%0E5'("\I('L*)$5.5GLB4$%27R1?
M(GT@/2`D14Y6>R)005)?1TQ/0D%,7R1?(GT@:68@97AI<W1S("1%3E9[(E!!
M4E]'3$]"04Q?)%\B?3L*?0II9B`H)$5.5GM005)?5$5-4'TI('L*9&5L971E
M("1%3E9[4$%27T-,14%.?3L*?0I]"G-U8B!O=71S('L*<F5T=7)N(&EF("1Q
M=6EE=#L*:68@*"1L;V=F:"D@>PIP<FEN="`D;&]G9F@@(D!?7&XB.PI]"F5L
M<V4@>PIP<FEN="`B0%]<;B(["GT*?0IS=6(@:6YI=%]I;F,@>PIR97%U:7)E
M($-O;F9I9SL*<'5S:"!`24Y#+"!G<F5P(&1E9FEN960L(&UA<"`D0V]N9FEG
M.CI#;VYF:6=[)%]]+"!Q=R@*87)C:&QI8F5X<"!P<FEV;&EB97AP('-I=&5A
M<F-H97AP('-I=&5L:6)E>'`*=F5N9&]R87)C:&5X<"!V96YD;W)L:6)E>'`*
M*3L*?0IP86-K86=E(&UA:6X["G)E<75I<F4@4$%2.PIU;G-H:69T($!)3D,L
M(%PF4$%2.CIF:6YD7W!A<CL*4$%2+3YI;7!O<G0H0'!A<E]A<F=S*3L*9&EE
M('%Q6W!A<BYP;#H@0V%N)W0@;W!E;B!P97)L('-C<FEP="`B)'!R;V=N86UE
M(CH@3F\@<W5C:"!F:6QE(&]R(&1I<F5C=&]R>5QN70IU;FQE<W,@+64@)'!R
M;V=N86UE.PID;R`D<')O9VYA;64["D-/4D4Z.F5X:70H)#$I(&EF("@D0"`]
M?B]>7U1+7T58251?7"@H7&0K*5PI+RD["F1I92`D0"!I9B`D0#L*?3L*)#HZ
M7U]%4E)/4B`]("1`(&EF("1`.PI]"D-/4D4Z.F5X:70H)#$I(&EF("@D.CI?
M7T524D]2(#U^+UY?5$M?15A)5%]<*"A<9"LI7"DO*3L*9&EE("0Z.E]?15)2
M3U(@:68@)#HZ7U]%4E)/4CL*,3L*7U]%3D1?7PH``1L#.P@!```@````0(__
M_R`!``"@C___1`$``"B0__]L`0``V)'__]0!``#`DO__``(``&B3__\P`@``
MR)3__V0"````E___G`(``)"7___(`@``K)C__P`#``"DF?__/`,```R:__]T
M`P``A)K__YP#``",FO__L`,```B=___D`P``.)W__Q0$``"8G?__6`0``&2?
M__^(!```Y)___[`$``!$H/__V`0``"2B__\0!0``1*+__S`%```LH___8`4`
M`-"I__^8!0``8*S__\P%``#$K/__+`8``,BX__]D!@``++G__X0&``!$NO__
ML`8``+B[__^D`0``0+S__Y`!``"4O/___`4``!```````````7I2``)\#@$;
M#`T`(````!@````8CO__8`````!$#0Q,A`:%!8L$C0..`D(,"P0`)````#P`
M``!4CO__B`````!$#0Q(A`B%!X8&AP6+!(T#C@)"#`L$`"````!D````M([_
M_[`!````1@T,1(<%BP2-`XX"0@P+!````!````"(````J+K__U0`````````
M+````)P````,NO__B`````!$#0Q$A`:%!8L$C0..`D0,"P1H#`T80L[-R\7$
M#@0`*````,P```#\C___Z`````!"#0Q$A`J%"88(AP>(!HD%BP2-`XX"1`P+
M!``L````^````+B0__^H`````$(-#$2$"(4'A@:'!8L$C0..`D(,"P0"0@H,
M#2!""P`P````*`$``#"1__]@`0```$0-#$:$"X4*A@F'"(@'B0:*!8L$C0..
M`D(,"P1<"@P-+$(+-````%P!``!<DO__.`(```!"#0Q$A`N%"H8)APB(!XD&
MB@6+!(T#C@)"#`L$`I8*#`TL0@L````H````E`$``%R4__^0`````$0-#$:$
M!H4%BP2-`XX"0@P+!&@*#`T80@L``#0```#``0``P)3__QP!````1`T,0H0+
MA0J&"8<(B`>)!HH%BP2-`XX"0@P+!`)8"@P-+$(+````.````/@!``"DE?__
M^`````!$#0Q"A`:%!8L$C0..`D(,"P0"6`H,#1A""TX*#`T80L[-R\7$#@1"
M"P``-````#0"``!@EO__:`````!$#0Q(A`B%!X8&AP6+!(T#C@)"#`L$6`P-
M($+.S<O'QL7$#@0````D````;`(``)"6__]X`````$0-#$2$!H4%BP2-`XX"
M1`P+!&0,#1@`$````)0"``#@EO__"``````````P````J`(``-26__]\`@``
M`$0-#$:$"X4*A@F'"(@'B0:*!8L$C0..`D0,"P1\"@P-+$(++````-P"```<
MF?__,`````!"#0Q$A`:%!8L$C0..`D(,"P1,#`T80L[-R\7$#@0`0`````P#
M```<F?__8`````!"#0Q"A`:%!8L$C0..`D0,"P14"@P-&$+.S<O%Q`X$0@M&
M"@P-&$+.S<O%Q`X$0@L````L````4`,``#B9___,`0```$0-#$B$!X4&A@6+
M!(T#C@)"#`L$`I0*#`T<0@L````D````@`,``-2:__^``````$0-#$B$!8L$
MC0..`D(,"P1H"@P-%$(+)````*@#```LF___8`````!"#0Q"@`2!`X("@P%$
MBPB-!XX&0@P+%#0```#0`P``9)O__^`!````1`T,0H0+A0J&"8<(B`>)!HH%
MBP2-`XX"0@P+!`*T"@P-+$(+````'`````@$```,G?__(`````!"#0Q"BP2-
M`XX"0@P+!``L````*`0```R=___H`````$0-#$:$"84(A@>'!H@%BP2-`XX"
M0@P+!'`*#`TD0@LT````6`0``,2=__^D!@```$(-#$2$"X4*A@F'"(@'B0:*
M!8L$C0..`D(,"P0#.`$*#`TL0@L``#````"0!```,*3__Y`"````1`T,2(0)
MA0B&!X<&B`6+!(T#C@)"#`L$`M(*#`TD0@L````L````Q`0``(RF__]D````
M`$(-#$*$!H4%BP2-`XX"0@P+!&8,#1A"SLW+Q<0.!``L````]`0``)"V__^0
M`@```$(-#$2$"(4'A@:'!8L$C0..`D(,"P0"S`H,#2!""P`T````)`4``)"F
M__\$#````$0-#$2$"X4*A@F'"(@'B0:*!8L$C0..`D0,"P0#[@4*#`TL0@L`
M`!P```!<!0``7++__V0`````0@T,1(L$C0..`D(,"P0`*````'P%``"@LO__
M&`$```!"#0Q"A`J%"88(AP>(!HD%BP2-`XX"7@P+!``\````J`4``(RS__]T
M`0```$(-#$*$"H4)A@B'!X@&B06+!(T#C@)2#`L$`I@*#`TH0L[-R\G(Q\;%
MQ`X$0@L``````+A#```P0P````````$````!`````0```,0"```!````S@(`
M``$```#<`@```0````(#```/````FP0``!D```!LC@$`&P````0````:````
M<(X!`!P````$````!````'@!```%````8`L```8```"P!```"@```.0$```+
M````$````!4``````````P```&B/`0`"````$`(``!0````1````%P```&P1
M```1````1!```!(````H`0``$P````@```#[__]O````"/K__V\9````````
M`````````````````````````````````````````````````'B.`0``````
M`````'P3``!\$P``?!,``'P3``!\$P``?!,``'P3``!\$P``?!,``'P3``!\
M$P``?!,``'P3``!\$P``?!,``'P3``!\$P``?!,``'P3``!\$P``?!,``'P3
M``!\$P``?!,``'P3``!\$P``?!,``'P3``!\$P``?!,``'P3``!\$P``?!,`
M`'P3``!\$P``?!,``'P3``!\$P``?!,``'P3``!\$P``?!,``'P3``!\$P``
M?!,``'P3``!\$P``?!,``'P3``!\$P``?!,``'P3``!\$P``?!,``'P3``!\
M$P``?!,``'P3``!\$P``?!,``'P3``!\$P``?!,``'P3``!\$P``?!,``'2.
M`0``````7!$```````!LC@$`````````````````@)$!`%P1````````<(X!
M```````<D0$`````````````````>(X!``````!$D0$``````-B0`0!(,0``
M0)$!`-R0`0`*4$%2+G!M"@``````2@``Q$8```````#41P``V$<``/1)```@
M1P``_$D```````#@20``Z$D````````(2@``1T-#.B`H3F5T0E-$(&YB-"`R
M,#(P,#@Q,"D@-RXU+C``1T-#.B`H;F(T(#(P,C`P.#$P*2`W+C4N,``D3F5T
M0E-$.B!C<G0P+E,L=B`Q+C0@,C`Q."\Q,2\R-B`Q-SHS-SHT-"!J;V5R9R!%
M>'`@)``D3F5T0E-$.B!C<G0P+6-O;6UO;BYC+'8@,2XR,R`R,#$X+S$R+S(X
M(#(P.C$R.C,U(&-H<FES=&]S($5X<"`D`"1.971"4T0Z(&-R=&DN4RQV(#$N
M,2`R,#$R+S`X+S$S(#`R.C0Y.C`T(&UA='0@17AP("0`)$YE=$)31#H@8W)T
M8F5G:6XN8RQV(#$N,3<@,C`Q."\Q,B\R."`Q.#HQ-SHQ,2!C:')I<W1O<R!%
M>'`@)``D3F5T0E-$.B!C<G1E;F0N4RQV(#$N,B`R,#$S+S$R+S(P(#$U.C,U
M.C0X(&IO97)G($5X<"`D`````/'/`@`````````````````(````"`````@`
M```(````&P````$``````````"HJ*B!.;R!,86)E;"!0<F]V:61E9"`J*BH`
M`/'/`@`````````````````(````"`````@````(````&P````$`````````
M`"HJ*B!.;R!,86)E;"!0<F]V:61E9"`J*BH`038```!A96%B:0`!+`````5#
M;W)T97@M03@`!@H'00@!"0(*`A($%`$5`1<#&`$:`AP!(@%$`0`N<VAS=')T
M86(`+FEN=&5R<``N;F]T92YN971B<V0N:61E;G0`+FYO=&4N;F5T8G-D+G!A
M>``N;F]T92YN971B<V0N;6%R8V@`+FAA<V@`+F1Y;G-Y;0`N9'EN<W1R`"YR
M96PN9'EN`"YR96PN<&QT`"YT97AT`"YR;V1A=&$`+F5H7V9R86UE7VAD<@`N
M96A?9G)A;64`+FEN:71?87)R87D`+F9I;FE?87)R87D`+FIC<@`N9'EN86UI
M8P`N9V]T`"YD871A`"YB<W,`+F-O;6UE;G0`+E-53E=?8W1F`"Y!4DTN871T
M<FEB=71E<P``````````````````````````````````````````````````
M``````L````!`````@```!0!```4`0``%P```````````````0`````````3
M````!P````(````L`0``+`$``!@```````````````0`````````)@````<`
M```"````1`$``$0!```4```````````````$`````````#<````'`````@``
M`%@!``!8`0``(```````````````!`````````!*````!0````(```!X`0``
M>`$``#@#```&``````````0````$````4`````L````"````L`0``+`$``"P
M!@``!P````,````$````$````%@````#`````@```&`+``!@"P``Y`0`````
M`````````0````````!@````"0````(```!$$```1!```"@!```&````````
M``0````(````:0````D```!"````;!$``&P1```0`@``!@```!,````$````
M"````&T````!````!@```'P3``!\$P``+`,`````````````!`````0```!R
M`````0````8```"H%@``J!8``/0O``````````````0`````````>`````$`
M```"````G$8``)Q&``#<0``````````````$`````````(`````!`````@``
M`'B'``!XAP``#`$`````````````!`````````".`````0````(```"$B```
MA(@``.@%``````````````0`````````F`````X````#````;(X!`&R.```$
M```````````````$````!````*0````/`````P```'".`0!PC@``!```````
M````````!`````0```"P`````0````,```!TC@$`=(X```0`````````````
M``0`````````M0````8````#````>(X!`'B.``#P````!P`````````$````
M"````+X````!`````P```&B/`0!HCP``<`$`````````````!`````0```##
M`````0````,```#8D`$`V)```$@```````````````0`````````R0````@`
M```#````0)$!`""1``!$``````````````!``````````,X````!````,```
M```````@D0``.P```````````````0````$````?`````0``````````````
M6Y$``"(!``````````````$`````````UP````$``````````````("2``"/
M```````````````$`````````.$````#``!P```````````/DP``-P``````
M`````````0`````````!`````P``````````````1I,``/$`````````````
)``$`````````
