/* riemann/event.h -- Riemann C client library
 * SPDX-FileCopyrightText: 2013-2024 Gergely Nagy
 * SPDX-FileContributor: Gergely Nagy
 *
 * SPDX-License-Identifier: EUPL-1.2
 */

#pragma once

#include <riemann/proto/riemann.pb-c.h>
#include <riemann/attribute.h>
#include <stdarg.h>

typedef Event riemann_event_t;

typedef enum
  {
    RIEMANN_EVENT_FIELD_NONE = 0,
    RIEMANN_EVENT_EMPTY = 0,
    RIEMANN_EVENT_FIELD_TIME,
    RIEMANN_EVENT_FIELD_STATE,
    RIEMANN_EVENT_FIELD_SERVICE,
    RIEMANN_EVENT_FIELD_HOST,
    RIEMANN_EVENT_FIELD_DESCRIPTION,
    RIEMANN_EVENT_FIELD_TAGS,
    RIEMANN_EVENT_FIELD_TTL,
    RIEMANN_EVENT_FIELD_ATTRIBUTES,
    RIEMANN_EVENT_FIELD_METRIC_S64,
    RIEMANN_EVENT_FIELD_METRIC_D,
    RIEMANN_EVENT_FIELD_METRIC_F,

    RIEMANN_EVENT_FIELD_STRING_ATTRIBUTES,
    RIEMANN_EVENT_FIELD_TIME_MICROS
  } riemann_event_field_t;

#ifdef __cplusplus
extern "C" {
#endif

riemann_event_t *riemann_event_new (void);
riemann_event_t *riemann_event_create (riemann_event_field_t field, ...);
riemann_event_t *riemann_event_create_va (riemann_event_field_t field,
                                          va_list aq);
riemann_event_t *riemann_event_create_full (int has_time, int64_t time,
                                            char *state,
                                            char *service,
                                            char *host,
                                            char *desc,
                                            size_t n_tags, char **tags,
                                            int has_ttl, float ttl,
                                            size_t n_attributes, riemann_attribute_t **attributes,
                                            int has_time_micros, int64_t time_micros,
                                            int has_metric_s64, int64_t metric_s64,
                                            int has_metric_d, double metric_d,
                                            int has_metric_f, float metric_f);
riemann_event_t *riemann_event_clone (const riemann_event_t *event);
void riemann_event_free (riemann_event_t *event);

int riemann_event_set (riemann_event_t *event, ...);
int riemann_event_set_va (riemann_event_t *event,
                          riemann_event_field_t first_field, va_list aq);
#define riemann_event_set_one(event, field, ...)                        \
  riemann_event_set (event, RIEMANN_EVENT_FIELD_##field, __VA_ARGS__,   \
                     RIEMANN_EVENT_FIELD_NONE)

int riemann_event_set_tags (riemann_event_t *event, ...);
int riemann_event_set_tags_va (riemann_event_t *event, va_list aq);
int riemann_event_set_tags_n (riemann_event_t *event, size_t n_tags, const char **tags);
int riemann_event_tag_add (riemann_event_t *event, const char *tag);

int riemann_event_set_attributes (riemann_event_t *event, ...);
int riemann_event_set_attributes_va (riemann_event_t *event, va_list aq);
int riemann_event_set_attributes_n (riemann_event_t *event, size_t n_attributes,
                                    riemann_attribute_t **attributes);
int riemann_event_set_string_attributes_n (riemann_event_t *event,
                                           size_t n_attributes,
                                           const char **keys,
                                           const char **values);
int riemann_event_attribute_add (riemann_event_t *event,
                                 riemann_attribute_t *attrib);
int riemann_event_string_attribute_add (riemann_event_t *event,
                                        const char *key,
                                        const char *value);

#ifdef __cplusplus
}
#endif
