;;;
;;; io.scm
;;;
;;; Copyright (C) 2009, 2010, ivan demakov.
;;;
;;; This code is free software; you can redistribute it and/or modify
;;; it under the terms of the GNU Lesser General Public License as published by
;;; the Free Software Foundation; either version 2.1 of the License, or (at your
;;; option) any later version.
;;;
;;; This code is distributed in the hope that it will be useful, but
;;; WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
;;; or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU Lesser General Public
;;; License for more details.
;;;
;;; You should have received a copy of the GNU Lesser General Public License
;;; along with this code; see the file COPYING.LESSER.  If not, write to
;;; the Free Software Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston,
;;; MA 02110-1301, USA.
;;;
;;;
;;; Author:        ivan demakov <ksion@users.sourceforge.net>
;;; Creation date: Tue Dec 15 19:56:35 2009
;;; Last Update:   Sun Feb 28 22:26:09 2010
;;;
;;;

(library (ksi io)
         (import (ksi core syntax) (ksi core list) (ksi core io))
         (export open-input-file open-output-file
                 open-input-string open-output-string get-output-string
                 close-input-port close-output-port
                 call-with-input-file call-with-output-file
                 call-with-input-string call-with-output-string
                 with-input-from-port with-output-to-port
                 with-error-to-port with-input-from-file
                 with-output-to-file with-error-to-file

                 ; from (ksi io core)
                 current-input-port current-output-port current-error-port
                 set-current-input-port set-current-output-port set-current-error-port
                 open-file open-string close-port
                 port? input-port? output-port?
                 flush-port port-string
                 read-char peek-char char-ready? port-ready? eof-object?
                 read write display newline write-char
                 format setlocale)

(define open-input-file
  (lambda (file-name)
    (open-file file-name "r")))

(define open-output-file
  (lambda (file-name)
    (open-file file-name "w")))

(define open-input-string
  (lambda (string)
    (open-string string "r")))

(define open-output-string
  (lambda ()
    (open-string (make-string 0) "w")))

(define get-output-string
  (lambda (output-port)
    (string-copy (port-string output-port))))

(define close-input-port
  (lambda (port)
    (close-port port)))

(define close-output-port
  (lambda (port)
    (close-port port)))

(define call-with-input-file
  (lambda (file-name proc)
    (let* ((file (open-input-file file-name))
	   (vals (proc file)))
      (close-port file)
      vals)))

(define call-with-output-file
  (lambda (file-name proc)
    (let* ((file (open-output-file file-name))
	   (vals (proc file)))
      (close-port file)
      vals)))

(define call-with-input-string
  (lambda (string proc)
    (let* ((file (open-input-string string))
	   (vals (proc file)))
      (close-port file)
      vals)))

(define call-with-output-string
  (lambda (string proc)
    (let* ((file (open-output-string string))
	   (vals (proc file)))
      (close-port file)
      vals)))

(define with-input-from-port
  (lambda (port thunk)
    (let ((swaports (lambda () (set! port (set-current-input-port port)))))
      (dynamic-wind swaports thunk swaports))))

(define with-output-to-port
  (lambda (port thunk)
    (let ((swaports (lambda () (set! port (set-current-output-port port)))))
      (dynamic-wind swaports thunk swaports))))

(define with-error-to-port
  (lambda (port thunk)
    (let ((swaports (lambda () (set! port (set-current-error-port port)))))
      (dynamic-wind swaports thunk swaports))))

(define with-input-from-file
  (lambda (file thunk)
    (let* ((port (open-input-file file))
	   (vals (with-input-from-port port thunk)))
      (close-port port)
      vals)))

(define with-output-to-file
  (lambda (file thunk)
    (let* ((port (open-output-file file))
	   (vals (with-output-to-port port thunk)))
      (close-port port)
      vals)))

(define with-error-to-file
  (lambda (file thunk)
    (let* ((port (open-output-file file))
	   (vals (with-error-to-port port thunk)))
      (close-port port)
      vals)))

)

;;; End of code