module Hashery

  # OrderedHash is a simple ordered hash implmentation, for users of
  # Ruby 1.8.7 or less.
  #
  # NOTE: As of Ruby 1.9+ this class is not needed, since
  # Ruby 1.9's standard Hash tracks inseration order.
  #
  # This implementation derives from the same class in
  # ActiveSupport library.
  #
  class OrderedHash < ::Hash

    def to_yaml_type
      "!tag:yaml.org,2002:omap"
    end

    def to_yaml(opts = {})
      YAML.quick_emit(self, opts) do |out|
        out.seq(taguri, to_yaml_style) do |seq|
          each do |k, v|
            seq.add(k => v)
          end
        end
      end
    end

    # Hash is ordered in Ruby 1.9!
    if RUBY_VERSION < '1.9'

      def initialize(*args, &block)
        super
        @keys = []
      end

      def self.[](*args)
        ordered_hash = new

        if (args.length == 1 && args.first.is_a?(Array))
          args.first.each do |key_value_pair|
            next unless (key_value_pair.is_a?(Array))
            ordered_hash[key_value_pair[0]] = key_value_pair[1]
          end

          return ordered_hash
        end

        unless (args.size % 2 == 0)
          raise ArgumentError.new("odd number of arguments for Hash")
        end

        args.each_with_index do |val, ind|
          next if (ind % 2 != 0)
          ordered_hash[val] = args[ind + 1]
        end

        ordered_hash
      end

      def initialize_copy(other)
        super(other)
        @keys = other.keys
      end

      def []=(key, value)
        @keys << key unless key?(key)
        super(key, value)
      end

      def delete(key)
        if has_key? key
          index = @keys.index(key)
          @keys.delete_at(index)
        end
        super(key)
      end

      def delete_if
        super
        sync_keys!
        self
      end

      def reject!
        super
        sync_keys!
        self
      end

      def reject(&block)
        dup.reject!(&block)
      end

      def keys
        @keys.dup
      end

      def values
        @keys.collect{ |key| self[key] }
      end

      def to_hash
        self
      end

      def to_a
        @keys.map{ |key| [ key, self[key] ] }
      end

      def each_key
        @keys.each{ |key| yield(key) }
      end

      def each_value
        @keys.each{ |key| yield(self[key]) }
      end

      def each
        @keys.each{ |key| yield(key, self[key]) }
      end

      alias_method :each_pair, :each

      def clear
        super
        @keys.clear
        self
      end

      def shift
        k = @keys.first
        v = delete(k)
        [k, v]
      end

      def merge!(other_hash)
        other_hash.each{ |k,v| self[k] = v }
        self
      end

      def merge(other_hash)
        dup.merge!(other_hash)
      end

      # When replacing with another hash, the initial order of our
      # keys must come from the other hash, ordered or not.
      def replace(other)
        super
        @keys = other.keys
        self
      end

      def inspect
        "#<OrderedHash #{super}>"
      end

    private

      def sync_keys!
        @keys.delete_if{ |k| !key?(k) }
      end

    end

  end

end

require 'yaml'

YAML.add_builtin_type("omap") do |type, val|
  OrderedHash[val.map(&:to_a).map(&:first)]
end

