package TDS::AccessLog::AnalysisUpdate;
# $Id: AnalysisUpdate.pm,v 1.11 2001/01/03 08:20:47 tom Exp $
################################################################

=head1 NAME

TDS::AccessLog::Analysis

=head1 SYNOPSIS

 use TDS::AccessLog::Analysis;

 $a = new TDS::AccessLog::Analysis;
 $a->Read;
 $a->ReadLogFile;
 print $a->AsHTML;

=head1 DESCRIPTION

if year, month are not specified,
use current one.

=cut

################################################################

use strict;
use vars qw(@ISA @EXPORT $SkeltonFile);
use Exporter;
use FileHandle;

use ObjectTemplate;
use DateTime::Date;
use DateTime::Time;
use DateTime::Format;
use CGI::QueryString;
use CGI::Tools;
use Template;

use TDS::System;
use TDS::ID_Map;
use TDS::Admin::Authorize;
use TDS::Admin::Skelton;
use TDS::AccessLog::DirInfo;
use TDS::AccessLog::Logging;

@ISA = qw(TDS::Admin::Skelton TDS::Admin::Authorize TDS::AccessLog::DirInfo);
@EXPORT = qw(attributes);

=head1 STATIC VARIABLES

 $SkeltonFile    

=cut

$SkeltonFile = "update_log_skelton.html";

attributes qw(year month part limit
	      total daily hours recent
	      first_date last_date
	      );

################################################################
sub initialize($)
{
    my $self = shift;

    $self->filename($SkeltonFile) unless $self->filename;

    # unless year, month specified, use current time
    if (!$self->year && !$self->month){
	my $date = new DateTime::Date;
	$date->SetTime(time(), $TDS::System::TZ);
	$self->year($date->year);
	$self->month($date->month);
#	$self->part(undef);
    }

    $self->daily({});
    $self->hours([]);
    $self->recent([]);

    $self->TDS::Admin::Authorize::initialize;
    $self->TDS::Skelton::initialize;
    $self->TDS::AccessLog::DirInfo::initialize;
}


=head2 $a->ReadLogFile

read logfile

=cut

sub ReadLogFile($)
{
    my $self = shift;

    # custom flag check
    unless ($TDS::AccessLog::Logging::UpdateLogFlag){
	die qq(<h2>Update Log not Activated</h2>
	       <p>if you want to activate update log,</p>
	       <blockquote>\$TDS::AccessLog::Logging::UpdateLogFlag = 1;</blockquote>
	       <p>in diary_dir/conf/customize.ph</p>);
    }
    # do read
    my $filename = $self->GetLogFilename($self->year, $self->month, "update");
    my $fh = new FileHandle;
    $fh->open($filename) || die "can't open logfile: $filename";
    my $total = 0;
    
    my $time = new DateTime::Time;
    my $first_date = undef;
    my $yesterday = $TDS::Status->start_time->Dup;
    $yesterday->Decrement("2D");
    my $yesterday_tm = $yesterday->GetTime($TDS::System::TZ);
    my $now_hour = $TDS::Status->start_time->hour;
    
    my $limit;
    my $limit_tm;
    
    if ($self->limit){
	    $limit = $TDS::Status->start_time->Dup;
	if ($self->limit eq '24h'){
	    $limit->Decrement("1D");
	    unless ($limit->min == 0 && $limit->sec == 0){
		$limit->Increment("1h");
		$limit->min(0), $limit->sec(0);
	    }
	} elsif ($self->limit eq 'month'){
	    $limit->day(1);
	    $limit->hour(0), $limit->min(0), $limit->sec(0);
	}
	$limit_tm = $limit->GetTime($TDS::System::TZ);
    }
    
    while (<$fh>){
	chomp;
	my $tm = str2time($_);
	
	next if $self->limit && $tm < $limit_tm;

	$time->SetTime($tm, $TDS::System::TZ);
	my $ymd = Expand("%y/%0m/%0d", $time->GetParams);
	#$self->daily->[$tm->day]++;
	my $hour = $time->hour;

	$self->daily->{$ymd}++;
	$self->hours->[$hour]++;
	if ($tm >= $yesterday_tm){
	    push(@{$self->recent}, $time->Dup);
	}
	$total++;
	
	unless (ref $first_date){
	    $first_date = new DateTime::Date(year=>$time->year,
					     month=>$time->month,
					     day=>$time->day);
	}
    }
    $fh->close;
    
    $self->total($total);
    $self->first_date($first_date);
    $self->last_date($time->Dup);
}

sub SetMacros($)
{
    my $self = shift;

    $self->SetMacro("TOTAL", $self->total);
    $self->SetMacro("AVERAGE", sub {
	unless (ref $self->last_date){
	    return 0;
	}
	my $diff = ($self->last_date->GetTime - $self->first_date->GetTime)/(60*60*24) + 1;
	return sprintf("%6.2f", $self->total/$diff);
    });
    $self->SetMacro("DAILY", sub {
	my $self = shift;
	my $rtn = qq(<table border="1">\n);
	
	my $dt = $self->first_date->Dup;

	while ($dt <= $self->last_date){
	    my $ymd = Expand("%y/%0m/%0d" ,$dt->GetParams);
	    my $times = $self->daily->{$ymd} || 0;
	    $rtn .= "<tr><td>$ymd</td><td>$times</td><td>" . bar($times) . "</td></tr>\n";
	    $dt += "1D";
	}
	return "$rtn</table>";
    });

    $self->SetMacro("HOURS", sub {
	my $rtn = qq(<table border="1">\n);
	
	for (0..23){
	    my $hour = $_;
	    my $times = $self->hours->[$_];
	    
	    $rtn .= "<tr><td>$hour</td><td>$times</td><td>" . bar($times) . "</td></tr>\n";
	}
	return "$rtn</table>";
    });
    $self->SetMacro("RECENT", sub {
#	return "not impemented";
	
	my $self = shift;
	my $rtn = qq(<table border="1">\n);
	
	for (@{$self->recent}){
	    my $params = $_->GetParams;
	    $params->{TZ} = $TDS::System::TZ;
	    my $ymd = Expand("%y/%0m/%0d %hh:%mm:%ss %TZ", $params);
	    $rtn .= "<tr><td>$ymd</td></tr>";
	}
	return "$rtn</table>";
    });
    $self->SUPER::SetMacros;
}
sub AsHTML($)
{
    my $self = shift;

    return $self->SUPER::AsHTML;
}
sub GetLogFilename($)
{
    my $self = shift;

    return sprintf("%s/update.log", GetLogDir());
}


################################################################
sub bar ($){
    my $cnt = shift;
    my ($rtn, $i);

    my $j = int($cnt/100);
    for (1..$j){
	$rtn .= bar_img(100);
    }
    $j = int(($cnt%100)/10);
    for (1..$j){
	$rtn .= bar_img(10);
    }
    $j = $cnt%10;
    for (1..$j){
	$rtn .= bar_img(1);
    }
    $rtn;
}
sub bar_img($)
{
    my $cnt = shift;
    my $alt;

    for (1..$cnt){ $alt .= "*"; }
#    my $url = TDS::IdentInfo->Get('url');
#    return qq(<img src="${url}image/bar$cnt.jpg" alt="$alt">);
    return qq(<img src="../image/bar$cnt.jpg" alt="$alt">);    
}
1;
