/*
 * AppleRecords - A DAAP client
 * Copyright (C) 2004  Chris Davies <c.davies@cdavies.org>
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
 *
 */

package org.cdavies.applerecords;

import org.cdavies.itunes.*;
import javazoom.jl.player.*;
import org.cdavies.applerecords.status.*;

import java.util.*;
import java.io.*;

/**
 * 
 * The thread that manages actually playing the songs fetched via DAAP
 * This class is in development right now, and is likely to be 
 * ripped up and rewritten, and probably renamed to incorporate
 * such functionality as downloading tracks.
 * <p>
 * About the best thing I can say about this class is that it works ;)
 * 
 * @author       Chris Davies (c.davies@cdavies.org)
 * @version      0.5 21/06/2004
 * @see	AppleRecordsControlPanel
 *
 */

public class PlayerThread extends Thread {
	
	private AppleRecordsStatusPanel _status;
	private Track _t;
	private Player _p;
	private AppleRecordsControlPanel _controls;
	
	private Timer _statusTimer;
	private PlayerStatusTimerTask _task;
	private File _downloadFile;
	
	private boolean _playMode;
	
	/**
	 *
	 * Constructs a new PlayerThread ready to be run. There will ever be one
	 * PlayerThread for any instance of AppleRecords.
	 *
	 * @param	status	This instance of AppleRecords' Status Panel
	 * @param	controls	This instance of AppleRecords' Control Panel
	 *
	 */
	
	public PlayerThread(AppleRecordsStatusPanel status, AppleRecordsControlPanel controls) {
		
		_status = status;
		_controls = controls;
		_playMode = true;
		
	}
	
	/**
	 *
	 * Runs the PlayerThread, this method does a number of things. Initially
	 * it waits until the Control Panel indicates that is ought to begin playing
	 * a song.
	 * <p>
	 * The song is played to the end, unless the player is interupted by calling
	 * the <code>stopTrack</code> method on this thread. During the time the
	 * track is playing a further thread, in the guise of timer task updates
	 * the status display with the current position in the track.
	 * <p>
	 * Once the track is complete, the timer task is destroyed and we begin
	 * the cycle once again.
	 *
	 * @see AppleRecordsControlPanel
	 * @see org.cdavies.applerecords.status.PlayerStatusTimerTask
	 *
	 */
	
	public void run() {
		
		do {
			
			try {
			
				while (!_controls.continuePlaying())
					sleep(150);
				
				_t = _controls.getNextTrack();
				_controls.selectPlayingTrack();
				
				if (_playMode) {
					
					_p = new Player(_t.getInputStream());
					
					_task = new PlayerStatusTimerTask(_status, this);
					_statusTimer = new Timer(true);
					_statusTimer.schedule(_task, 1000, 1000);
				
					_p.play();
				
					_task.cancel();
					
				}
				else {
					
					_controls.noNextTrack();
					_status.setText("Downloading track '" + _t.getTrackName() + "'...");
					
					InputStream _in = _t.getInputStream();
					byte _buf[] = new byte[1024];
					
					FileOutputStream _out = new FileOutputStream(_downloadFile);
					int _len;
					
					while (-1 != (_len = _in.read(_buf))) {
						
						_out.write(_buf, 0, _len);
						
					}
					
					_in.close();
					_out.close();
					
					_controls.stopped();
					
				}
				
				_status.setTimedText("Finished " + (_playMode ? "playing" : "downloading") + "'" + _t.getTrackName() + "'");
				
			}
			catch (Exception _e) {
			
				_e.printStackTrace();
				_status.setTimedText("Error during operation!");
			
			}
		
			
		
		} while (!(_controls.closeDownMode()));
		
		_controls.stopped();
		
	}
	
	/**
	 *
	 * Returns the position in the currently playing track.
	 *
	 * @return	The position in milliseconds that we are at in the currently playing track
	 *
	 */
	
	public int getPosition() {
		
		return _p.getPosition();
		
	}
	
	/**
	 *
	 * Stops the currently playing song, and cancels any timer tasks associated 
	 * with it.
	 *
	 */
	
	public void stopTrack() {
		
		if (_playMode) {
		
			_task.cancel();
			_p.close();
			
		}
		
		_controls.stopped();
		
	}
	
	/**
	 *
	 * Returns the currently playing track.
	 *
	 * @return The track currently playing, or null if no track has ever been played.
	 *
	 */
	
	public Track getTrack() {
		
		return _t;
		
	}
	
	public void setPlayMode() {
		
		_playMode = true;
		
	}
	
	public void setDownloadMode() {
		
		_playMode = false;
		
	}
	
	public void setDownloadFile(File downloadFile) {
		
		_downloadFile = downloadFile;
		
	}
	
}
