#   This file is part of PyFFTW.
#
#    Copyright (C) 2009 Jochen Schroeder
#    Email: jschrod@berlios.de
#
#    PyFFTW is free software: you can redistribute it and/or modify
#    it under the terms of the GNU General Public License as published by
#    the Free Software Foundation, either version 3 of the License, or
#    (at your option) any later version.
#
#    PyFFTW is distributed in the hope that it will be useful,
#    but WITHOUT ANY WARRANTY; without even the implied warranty of
#    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
#    GNU General Public License for more details.
#
#    You should have received a copy of the GNU General Public License
#    along with PyFFTW.  If not, see <http://www.gnu.org/licenses/>.

import ctypes
from ctypes import pythonapi, util, py_object
from numpy import ctypeslib, typeDict
from platform import system as psystem
from os.path import splitext, join, isfile, dirname, abspath, basename
from os.path import join as joinpath
from os import name as osname
from os import environ
from warnings import warn

try:
    fftw_path = environ['FFTW_PATH']
    libfullpath = joinpath(abspath(fftw_path),r'libfftw3f.so')
    if not isfile(libfullpath):
        raise IOError
except KeyError:
    libfullpath = r'/usr/pkg/lib/libfftw3f.so'
except IOError:
    warn('could not find %s in FFTW_PATH using installtime path'
             %'libfftw3f.so')
    libfullpath = r'/usr/pkg/lib/libfftw3f.so'

if not isfile(libfullpath) and (osname=='nt' or psystem=='Windows'):
    if isfile(joinpath(dirname(__file__), libfullpath)):
        libfullpath = joinpath(dirname(__file__), libfullpath)

# must use ctypes.RTLD_GLOBAL for threading support
ctypes._dlopen(libfullpath, ctypes.RTLD_GLOBAL)
lib = ctypes.cdll.LoadLibrary(libfullpath)
#check if library is actually loaded there doesn't seem to be a better way to
#do this in ctypes
if not hasattr(lib, 'fftwf_plan_dft_1d'):
    raise OSError('Could not load libfftw3f.so')

if osname == 'nt' or psystem() == 'Windows':
    lib_threads = lib
else:
    libbase, dot, ext = basename(libfullpath).partition('.')
    libdir = dirname(libfullpath)
    lib_threads = joinpath(libdir, libbase + '_threads.'+ ext)
    try:
        lib_threads = ctypes.cdll.LoadLibrary(lib_threads)
    except OSError, e:
        warn("Could not load threading library %s, threading support is disabled"
            %lib_threads)
        lib_threads = None


_typelist =    [('fftwf_plan_dft_1d', (typeDict['singlecomplex'], typeDict['singlecomplex'], 1)),
                       ('fftwf_plan_dft_2d', (typeDict['singlecomplex'], typeDict['singlecomplex'], 2)),
                       ('fftwf_plan_dft_3d', (typeDict['singlecomplex'], typeDict['singlecomplex'], 3)),
                       ('fftwf_plan_dft', (typeDict['singlecomplex'], typeDict['singlecomplex'])),
                       ('fftwf_plan_dft_c2r_1d', (typeDict['singlecomplex'], typeDict['single'], 1)),
                       ('fftwf_plan_dft_c2r_2d', (typeDict['singlecomplex'], typeDict['single'], 2)),
                       ('fftwf_plan_dft_c2r_3d', (typeDict['singlecomplex'], typeDict['single'], 3)),
                       ('fftwf_plan_dft_c2r', (typeDict['singlecomplex'], typeDict['single'])),
                       ('fftwf_plan_dft_r2c_1d', (typeDict['single'], typeDict['singlecomplex'], 1)),
                       ('fftwf_plan_dft_r2c_2d', (typeDict['single'], typeDict['singlecomplex'], 2)),
                       ('fftwf_plan_dft_r2c_3d', (typeDict['single'], typeDict['singlecomplex'], 3)),
                       ('fftwf_plan_dft_r2c', (typeDict['single'], typeDict['singlecomplex'])),
                       ('fftwf_plan_r2r_1d', (typeDict['single'], typeDict['single'], 1)),
                       ('fftwf_plan_r2r_2d', (typeDict['single'], typeDict['single'], 2)),
                       ('fftwf_plan_r2r_3d', (typeDict['single'], typeDict['single'], 3)),
                       ('fftwf_plan_r2r', (typeDict['single'], typeDict['single']))]

_adv_typelist = [('fftwf_plan_many_dft', (typeDict['singlecomplex'],
                                              typeDict['singlecomplex'])),
                  ('fftwf_plan_many_dft_c2r', (typeDict['singlecomplex'],
                                                   typeDict['single'])),
                  ('fftwf_plan_many_dft_r2c', (typeDict['single'],
                                                   typeDict['singlecomplex'])),
                  ('fftwf_plan_many_r2r', (typeDict['single'],
                                                   typeDict['single']))]


def set_argtypes(val, types):
    if types[0] == typeDict['singlecomplex'] and types[1] == typeDict['singlecomplex']:
        set_argtypes_c2c(val,types)
    elif types[0] == typeDict['singlecomplex'] or types[1] == typeDict['singlecomplex']:
        set_argtypes_c2r(val,types)
    else:
        set_argtypes_r2r(val,types)

def set_argtypes_c2c(val,types):
    if len(types) >2:
        val.argtypes = [ctypes.c_int for i in range(types[2])] +\
                       [ctypeslib.ndpointer(dtype=types[0],ndim=types[2], \
                                            flags='contiguous, writeable, '\
                                                  'aligned'),
                        ctypeslib.ndpointer(dtype=types[1], ndim=types[2],\
                                            flags='contiguous, writeable, '\
                                                  'aligned'),
                        ctypes.c_int, ctypes.c_uint]
    else:
        val.argtypes = [ctypes.c_int, ctypeslib.ndpointer(dtype=int, ndim=1,\
                                                          flags='contiguous, '\
                                                                'aligned'),
                        ctypeslib.ndpointer(dtype=types[0], flags='contiguous,'\
                                                                 ' writeable, '\
                                                                  'aligned'),
                        ctypeslib.ndpointer(dtype=types[1],flags='contiguous, '\
                                                                 'writeable,'\
                                                                 'aligned'),
                        ctypes.c_int, ctypes.c_uint]

def set_argtypes_c2r(val,types):
    if len(types) >2:
        val.argtypes = [ctypes.c_int for i in range(types[2])] +\
                       [ctypeslib.ndpointer(dtype=types[0],ndim=types[2], \
                                            flags='contiguous, writeable, '\
                                                  'aligned'),
                        ctypeslib.ndpointer(dtype=types[1], ndim=types[2],\
                                            flags='contiguous, writeable, '\
                                                  'aligned'),
                        ctypes.c_uint]
    else:
        val.argtypes = [ctypes.c_int, ctypeslib.ndpointer(dtype=int, ndim=1,\
                                                          flags='contiguous, '\
                                                                'aligned'),
                        ctypeslib.ndpointer(dtype=types[0], flags='contiguous,'\
                                                                 ' writeable, '\
                                                                  'aligned'),
                        ctypeslib.ndpointer(dtype=types[1],flags='contiguous, '\
                                                                 'writeable,'\
                                                                 'aligned'),
                        ctypes.c_uint]

def set_argtypes_r2r(val, types):
    if len(types) > 2:
        val.argtypes = [ctypes.c_int for i in range(types[2])] +\
                       [ctypeslib.ndpointer(dtype=types[0], ndim=types[2],
                                            flags='contiguous, writeable, '\
                                                  'aligned'),
                        ctypeslib.ndpointer(dtype=types[1], ndim=types[2],
                                            flags='contiguous, writeable, '\
                                                  'aligned')] +\
                        [ctypes.c_int for i in range(types[2])] +\
                        [ctypes.c_uint]
    else:
        val.argtypes = [ctypes.c_int, ctypeslib.ndpointer(dtype=int, ndim=1,
                                                          flags='contiguous, '\
                                                                'aligned'),
                        ctypeslib.ndpointer(dtype=types[0], flags='contiguous,'\
                                                                  'writeable, '\
                                                                  'aligned'),
                        ctypeslib.ndpointer(dtype=types[1], flags='contiguous,'\
                                                                  'writeable, '\
                                                                  'aligned'),
                        ctypeslib.ndpointer(dtype=int, ndim=1,
                                            flags='contiguous, aligned'),
                        ctypes.c_uint]

def set_argtypes_adv(val, types):
    if types[0] == typeDict['singlecomplex'] and types[1] == typeDict['singlecomplex']:
        val.argtypes = [ctypes.c_int, ctypeslib.ndpointer(dtype=int, ndim=1,
                                                          flags='contiguous, '\
                                                                'aligned'),
                        ctypes.c_int,
                        ctypeslib.ndpointer(dtype=types[0], flags='contiguous,'\
                                                                  'aligned,'\
                                                                  'writeable'),
                        ctypeslib.ndpointer(dtype=int, ndim=1,
                                            flags='contiguous,aligned'),
                        ctypes.c_int, ctypes.c_int,
                        ctypeslib.ndpointer(dtype=types[1], flags='contiguous,'\
                                                                  'aligned,'\
                                                                  'writeable'),
                        ctypeslib.ndpointer(dtype=int, ndim=1,
                                            flags='contiguous,aligned'),
                        ctypes.c_int, ctypes.c_int,
                        ctypes.c_int, ctypes.c_uint]
    elif types[0] == typeDict['singlecomplex'] or types[1]==typeDict['singlecomplex']:
        val.argtypes = [ctypes.c_int, ctypeslib.ndpointer(dtype=int, ndim=1,
                                                          flags='contiguous, '\
                                                                'aligned'),
                        ctypes.c_int,
                        ctypeslib.ndpointer(dtype=types[0], flags='contiguous,'\
                                                                  'aligned,'\
                                                                  'writeable'),
                        ctypeslib.ndpointer(dtype=int, ndim=1,
                                            flags='contiguous,aligned'),
                        ctypes.c_int, ctypes.c_int,
                        ctypeslib.ndpointer(dtype=types[1], flags='contiguous,'\
                                                                  'aligned,'\
                                                                  'writeable'),
                        ctypeslib.ndpointer(dtype=int, ndim=1,
                                            flags='contiguous,aligned'),
                        ctypes.c_int, ctypes.c_int,
                        ctypes.c_uint]
    elif types[0] == typeDict['single'] and types[1]==typeDict['single']:
        val.argtypes = [ctypes.c_int, ctypeslib.ndpointer(dtype=int, ndim=1,
                                                          flags='contiguous, '\
                                                                'aligned'),
                        ctypes.c_int,
                        ctypeslib.ndpointer(dtype=types[0], flags='contiguous,'\
                                                                  'aligned,'\
                                                                  'writeable'),
                        ctypeslib.ndpointer(dtype=int, ndim=1,
                                            flags='contiguous,aligned'),
                        ctypes.c_int, ctypes.c_int,
                        ctypeslib.ndpointer(dtype=types[1], flags='contiguous,'\
                                                                  'aligned,'\
                                                                  'writeable'),
                        ctypeslib.ndpointer(dtype=int, ndim=1,
                                            flags='contiguous, aligned'),
                        ctypes.c_int, ctypes.c_int,
                        ctypeslib.ndpointer(dtype=int, ndim=1,
                                            flags='contiguous, aligned'),
                        ctypes.c_uint]



# set the return and argument types on the plan functions
for name, types in _typelist:
    val = getattr(lib, name)
    val.restype = ctypes.c_void_p
    set_argtypes(val,types)

##do the same for advanced plans
for name, types in _adv_typelist:
    val = getattr(lib, name)
    val.restype = ctypes.c_void_p
    set_argtypes_adv(val,types)

#malloc and free
lib.fftwf_malloc.restype = ctypes.c_void_p
lib.fftwf_malloc.argtypes = [ctypes.c_int]
lib.fftwf_free.restype = None
lib.fftwf_free.argtypes = [ctypes.c_void_p]

#create a buffer from memory (necessary for array allocation)
PyBuffer_FromReadWriteMemory = pythonapi.PyBuffer_FromReadWriteMemory
PyBuffer_FromReadWriteMemory.restype = py_object
PyBuffer_FromReadWriteMemory.argtypes = [ctypes.c_void_p, ctypes.c_int]

#executing arrays
lib.fftwf_execute.restype = None
lib.fftwf_execute.argtypes = [ctypes.c_void_p]

#guru execution
lib.fftwf_execute_dft.restype = None
lib.fftwf_execute_dft.argtypes = [ctypes.c_void_p,
                        ctypeslib.ndpointer(flags='aligned, contiguous, '\
                                                        'writeable'),\
                        ctypeslib.ndpointer(flags='aligned, contiguous, '\
                                                        'writeable')]

#destroy plans
lib.fftwf_destroy_plan.restype = None
lib.fftwf_destroy_plan.argtypes = [ctypes.c_void_p]

#enable threading for plans
if lib_threads is not None:
    lib_threads.fftwf_init_threads.restype = ctypes.c_int
    lib_threads.fftwf_init_threads.argtypes = []
    lib_threads.fftwf_plan_with_nthreads.restype = None
    lib_threads.fftwf_plan_with_nthreads.argtypes = [ctypes.c_int]
    lib_threads.fftwf_cleanup_threads.restype = None
    lib_threads.fftwf_cleanup_threads.argtypes = []

    s = lib_threads.fftwf_init_threads()
    if not s:
        sys.stderr.write('fftwf_init_threads call failed, disabling threads support\n')
        lib_threads = None

#wisdom

# create c-file object from python
PyFile_AsFile = pythonapi.PyFile_AsFile
PyFile_AsFile.argtypes = [ctypes.py_object]
PyFile_AsFile.restype = ctypes.c_void_p

#export to file
lib.fftwf_export_wisdom_to_file.argtypes = [ctypes.c_void_p]
lib.fftwf_export_wisdom_to_file.restype = None

#export to string
lib.fftwf_export_wisdom_to_string.argtypes = None
lib.fftwf_export_wisdom_to_string.restype = ctypes.c_char_p

#import from file
lib.fftwf_import_wisdom_from_file.argtypes = [ctypes.c_void_p]
lib.fftwf_import_wisdom_from_file.restype = ctypes.c_int

#import from string
lib.fftwf_import_wisdom_from_string.argtypes = [ctypes.c_char_p]
lib.fftwf_import_wisdom_from_string.restype = ctypes.c_int

#import system wisdom
lib.fftwf_import_system_wisdom.restype = ctypes.c_int
lib.fftwf_import_system_wisdom.argtypes = None

#forget wisdom
lib.fftwf_forget_wisdom.restype = None
lib.fftwf_forget_wisdom.argtype = None
