/** @file scim_types.h
 *  @brief defines some basic data types.
 */

/*
 * Smart Common Input Method
 * 
 * Copyright (c) 2002-2004 James Su <suzhe@tsinghua.org.cn>
 *
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this program; if not, write to the
 * Free Software Foundation, Inc., 59 Temple Place, Suite 330,
 * Boston, MA  02111-1307  USA
 *
 * $Id: scim_types.h.in,v 1.2.4.1 2007/04/10 09:18:27 suzhe Exp $
 */


#ifndef __SCIM_TYPES_H
#define __SCIM_TYPES_H

#include <stdint.h>
#include <string>

namespace scim {

typedef uint16_t uint16;
typedef uint32_t uint32;
typedef uint64_t uint64;

#ifdef __STDC_ISO_10646__
    typedef wchar_t ucs4_t;
#else
    typedef uint32 ucs4_t;
#endif

} // namespace scim

#ifndef __STDC_ISO_10646__
#include <cstring>

namespace std
{

template<>
struct char_traits<scim::ucs4_t>
{

typedef scim::ucs4_t char_type;
typedef scim::ucs4_t int_type;
typedef streampos pos_type;
typedef streamoff off_type;
typedef mbstate_t state_type;

static void
assign(char_type& __c1, const char_type& __c2)
{
    __c1 = __c2;
}

static bool 
eq(const char_type& __c1, const char_type& __c2)
{
    return __c1 == __c2;
}

static bool 
lt(const char_type& __c1, const char_type& __c2)
{
    return __c1 < __c2;
}

static char_type*
assign(char_type* __s, size_t __n, char_type __a)
{
    while (__n-- > 0) 
        assign(__s[__n], __a);
    return __s;
}

static char_type*
copy(char_type* __s1, const char_type* __s2, size_t __n)
{
    return static_cast<char_type *>
        (memcpy(__s1, __s2, __n * sizeof(char_type)));
}

static char_type*
move(char_type* __s1, const char_type* __s2, size_t __n)
{
    return static_cast<char_type *>
        (memmove(__s1, __s2, __n * sizeof(char_type)));
}

static size_t
length(const char_type* __s)
{
    const char_type *__p;

    __p = __s;
    while (eq(*__p, static_cast<int_type>(0)))
        ++__p;
    return static_cast<size_t>(__p - __s);
}

static int
compare(const char_type* __s1, const char_type* __s2, size_t __n)
{
    size_t __i;

    for (__i = 0; __i < __n; ++__i)
        if (!eq(__s1[__i], __s2[__i]))
            return lt(__s1[__i], __s2[__i]) ? -1 : 1;
    return 0;
}

static const char_type*
find(const char_type* __s, size_t __n, const char_type& __a)
{
    while (__n-- > 0) {
        if (eq(*__s, __a))
            return __s;
        ++__s;
    }
    return 0;
}

static bool
eq_int_type(const int_type& __c1, const int_type& __c2)
{
    return __c1 == __c2;
}

static char_type
to_char_type(const int_type& __c)
{
    return static_cast<char_type>(__c);
}

static int_type
to_int_type(const char_type& __c)
{
    return static_cast<int_type>(__c);
}

static int_type
eof()
{
    return static_cast<int_type>(-1);
}

static int_type
not_eof(const int_type& __c)
{
    return eq_int_type(__c, eof())
	? static_cast<int_type>(0) : __c;
}

}; // struct char_traits<scim::ucs4_t>

} // namespace std

#endif //!__STDC_ISO_10646__

namespace scim {

typedef std::basic_string<char> String;
typedef std::basic_string<ucs4_t> WideString;

} // namespace scim

#endif //__SCIM_TYPES_H

/*
vi:ts=4:nowrap:ai:expandtab
*/
