#!/usr/bin/ruby -w

require 'optparse'
require 'ostruct'
require 'epcb/stopwatch'
require 'epcb/csv/genall'

$KCODE="s"
MY_NAME = "epcb-csv"
MY_VERSION = "1.0.0"

def print_usage_and_exit()
  ARGV.options { |opts| $stderr.puts(opts) }
  exit(EPCB::CODE_ARGUMENT_ERROR)
end

def print_backtrace(e)
  $stderr.puts("\n#{e.inspect}")
  e.backtrace.each do |s|
    $stderr.puts("    #{s}")
  end
end

stopwatch = Stopwatch.new
stopwatch.start
subcommands = [
  EPCB::Subcommand.new(EPCB::SUBC_CSV, EPCB::CSVGenerator),
  EPCB::Subcommand.new(EPCB::SUBC_EXCEL, EPCB::ExcelGenerator),
  EPCB::Subcommand.new(EPCB::SUBC_ALL, EPCB::CSVAndExcelGenerator),
]

$options = OpenStruct.new
$options.source_version = nil
$options.template = nil
$options.outfile = nil
$options.leave = false
$options.master = false
$options.bench = false
$options.verbose = false

ARGV.options do |opts|
  opts.banner = <<BANNER
Usage: #{MY_NAME}.exe subcommand src_file [options]
#{MY_NAME}.exe is a CSV data file generator and ExcelPettyCashBook generator from a CSV. Available subcommands are following.
    - '#{EPCB::SUBC_CSV}'   generates a CSV data file
    - '#{EPCB::SUBC_EXCEL}' generates ExcelPettyCashBook (an excel file)
    - '#{EPCB::SUBC_ALL}'   means both 'gencsv' and 'genexcel' are specified
BANNER

  opts.separator("")
  opts.separator("subcommand '#{EPCB::SUBC_CSV}' specific options are:")
  # --source-version
  opts.on("-S", "--source-version VERSION", String,
          "specify a source file version.") do |s|
    $options.source_version = s
  end

  opts.separator("subcommand '#{EPCB::SUBC_EXCEL}' specific options are:")
  # --template
  opts.on("-t", "--template FILE", String,
          "specify a template file.") do |t|
    $options.template = t
  end

  opts.separator("subcommand '#{EPCB::SUBC_ALL}' specific options are:")
  # --leave
  opts.on("-l", "--leave",
          "leave a CSV data file.") do |l|
    $options.leave = l
  end

  opts.separator("options are:")
  # --outfile
  opts.on("-o", "--outfile FILE", String,
          "specify a file name to be output.") do |o|
    $options.outfile = o
  end
  # --master
  opts.on("-m", "--master",
          "output/input master sheet data.") do |m|
    $options.master = m
  end
  # --bench
  opts.on("-b", "--bench",
          "report an elapsed time.") do |b|
    $options.bench = b
  end
  # --verbose
  opts.on_tail("-v", "--verbose",
               "verbose output.") do |v|
    $options.verbose = v
  end
  # --version
  opts.on_tail("-V", "--version",
               "output version information and exit.") do
    puts("#{MY_NAME} #{MY_VERSION}")
    exit(0)
  end
  
  # --help
  opts.on_tail("-h", "--help", "print this message and exit.") do
    puts(opts)
    exit(0)
  end

  # IvV̉
  opts.parse!
end or exit(EPCB::CODE_ARGUMENT_ERROR)

program = nil
begin
  subcommands.each do |subc|
    if subc.command == ARGV[0]
      program = subc.klass.new
    end
  end
  # subcommandw肳ĂȂꍇ
  if program.nil?
    if ARGV[0] != nil
      EPCB::eoutln("Unknown subcommand \"#{ARGV[0]}\".")
    else
      EPCB::eoutln("Must specify a subcommand.")
    end
    $stderr.puts("")
    print_usage_and_exit
  end

  if ARGV[1].nil?
    EPCB::eoutln("Must specify src_file.")
    exit(EPCB::CODE_ARGUMENT_ERROR)
  end

  program.setup(ARGV[1], $options.outfile)
  program.execute
  EPCB::voutln("The process successfully finished.")
rescue EPCB::FileNotFoundError => e
  EPCB::eoutln(e.message)
  print_backtrace(e)
  exit(EPCB::CODE_FILE_NOT_FOUND)
rescue IOError => e
  EPCB::eoutln(e.message)
  print_backtrace(e)
  exit(EPCB::CODE_IO_ERROR)
rescue EPCB::OptionUnsatisfiedError, EPCB::UnknownVersionError => e
  EPCB::eoutln(e.message)
  exit(EPCB::CODE_ARGUMENT_ERROR)
rescue SystemExit
  exit($!.status)
rescue Exception => e
  $stderr.puts("Oops! Unexpected error. (#{e.message})")
  print_backtrace(e)
  exit(EPCB::CODE_UNEXPECTED_ERROR)
ensure
  program.destroy unless program.nil?
end

# vԂ\
if $options.bench
  stopwatch.suspend
  puts "Elapsed #{format("%0.3f", stopwatch.time)} (sec)"
end
