package net.sf.amateras.air.as.syntax.analyze;

import java.io.File;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

import net.sf.amateras.air.as.syntax.ISyntaxClass;
import net.sf.amateras.air.as.syntax.PreviousWord;
import net.sf.amateras.air.as.syntax.proposal.DefaultCompletionProposal;
import net.sf.amateras.air.as.syntax.proposal.ImportCompletionProposal;

import org.eclipse.core.resources.IContainer;
import org.eclipse.core.resources.IFile;
import org.eclipse.core.resources.IFolder;
import org.eclipse.core.resources.IProject;
import org.eclipse.core.resources.IResource;
import org.eclipse.core.resources.IResourceVisitor;
import org.eclipse.core.runtime.CoreException;
import org.eclipse.core.runtime.Path;
import org.eclipse.jface.text.contentassist.ContextInformation;
import org.eclipse.jface.text.contentassist.ICompletionProposal;

/**
 * 
 * @author hideko ogawa
 */
public abstract class AbstractClassAnalizer {
	protected static final String CLASS_SEPARATOR = ".";
	protected static final String AS_EXTENSION = ".as";
	protected static final String DIRECTORY_SEPARATOR = "/";
	private static final String EMPTY = "";

	public void addSyntaxClass(int documentOffset, PreviousWord previousWord,
			List<ICompletionProposal> completionProposals, IProject project) {

		if (!isEnabled(previousWord)) {
			return;
		}

		Map<String, ICompletionProposal> packages = new HashMap<String, ICompletionProposal>();
		Object[] sourcePaths = getParentPath(project);
		for (Object sourcePath : sourcePaths) {

			if (previousWord.getType() == PreviousWord.TYPE_IMPORT) {
				addImportSyntax(documentOffset, previousWord.getPreviousWord(), completionProposals, packages,
						sourcePath);

			} else if (sourcePath instanceof File) {
				addClassSyntax(documentOffset, previousWord.getPreviousWord(), completionProposals, (File) sourcePath,
						(File) sourcePath);

			} else if (sourcePath instanceof IResource) {
				addClassSyntax(documentOffset, previousWord.getPreviousWord(), completionProposals,
						(IResource) sourcePath, (IResource) sourcePath);
			}
		}

		for (String key : packages.keySet()) {
			completionProposals.add(packages.get(key));
		}
	}

	protected abstract Object[] getParentPath(IProject project);

	protected boolean isEnabled(PreviousWord previousWord) {
		if (previousWord.getPreviousWord().length() == 0) {
			return false;
		}

		if (previousWord.getType() != PreviousWord.TYPE_ALL && previousWord.getType() != PreviousWord.TYPE_CLASS
				&& previousWord.getType() != PreviousWord.TYPE_CLASS_NEW
				&& previousWord.getType() != PreviousWord.TYPE_IMPORT) {
			return false;
		}
		return true;
	}

	protected String[] splitImportPackageAndClass(String word) {
		String subDirPath = word;
		String className = EMPTY;
		if (!(subDirPath.endsWith(CLASS_SEPARATOR))) {
			int p = subDirPath.lastIndexOf(CLASS_SEPARATOR);
			if (p > 0) {
				subDirPath = subDirPath.substring(0, p);
				className = word.substring(p + 1);
			} else {
				subDirPath = EMPTY;
				className = word;
			}
		} else {
			subDirPath = subDirPath.substring(0, subDirPath.length() - 1);
		}

		return new String[] { subDirPath, className };
	}

	protected void addImportSyntax(final int documentOffset, final String previousWord,
			final List<ICompletionProposal> completionProposals, final Map<String, ICompletionProposal> packages,
			final Object sourceDirPath) {

		String[] splitPackages = splitImportPackageAndClass(previousWord);
		final String packageName = splitPackages[0];
		final String className = splitPackages[1];

		String subDirPath = packageName.replaceAll("\\.", DIRECTORY_SEPARATOR);

		if (sourceDirPath instanceof File) {
			File packageDir = new File((File) sourceDirPath, subDirPath);
			if (!packageDir.exists() || packageDir.list() == null || packageDir.list().length == 0) {
				return;
			}

			for (String fileName : packageDir.list()) {
				File file = new File(packageDir, fileName);
				if (fileName.toLowerCase().startsWith(className.toLowerCase())) {
					if (file.isDirectory()) {
						packages.put(packageName + CLASS_SEPARATOR + fileName, createPackageCompletionProposal(
								documentOffset, previousWord, packageName, fileName, ISyntaxClass.TYPE_PACKAGE));

					} else if (fileName.endsWith(AS_EXTENSION)) {
						completionProposals.add(createPackageCompletionProposal(documentOffset, previousWord,
								packageName, fileName.replaceAll(AS_EXTENSION, ""), ISyntaxClass.TYPE_CLASS));
					}
				}
			}
		} else if (sourceDirPath instanceof IContainer) {
			try {
				IContainer folder = (IContainer) sourceDirPath;
				if (packageName.length() > 0) {
					folder = ((IContainer) sourceDirPath).getFolder(new Path(packageName.replaceAll("\\.",
							DIRECTORY_SEPARATOR)));
				}
				if (!folder.exists()) {
					return;
				}
				IResource[] members = folder.members();
				for (IResource member : members) {
					if (member.getName().toLowerCase().startsWith(className.toLowerCase())) {
						if (member instanceof IFolder) {
							packages.put(packageName + CLASS_SEPARATOR + member.getName(),
									createPackageCompletionProposal(documentOffset, previousWord, packageName, member
											.getName(), ISyntaxClass.TYPE_PACKAGE));
						} else if (member instanceof IFile && member.getFileExtension().equals("as")) {
							completionProposals
									.add(createPackageCompletionProposal(documentOffset, previousWord, packageName,
											member.getName().replaceAll(AS_EXTENSION, ""), ISyntaxClass.TYPE_CLASS));
						}
					}
				}

			} catch (CoreException e) {
				// TODO Auto-generated catch block
				e.printStackTrace();
			}
		}

	}

	private DefaultCompletionProposal createPackageCompletionProposal(int documentOffset, String previousWord,
			String packageName, String subPackageName, int type) {
		String path = subPackageName;
		if (packageName != null && packageName.length() > 0) {
			path = packageName + CLASS_SEPARATOR + path;
		}
		DefaultCompletionProposal proposal = new DefaultCompletionProposal(type, path);
		proposal.setReplacementLength(previousWord.length());
		proposal.setReplacementOffset(documentOffset - previousWord.length());
		proposal.setCursorPosition(path.length());
		proposal.setDisplayString(subPackageName + " - " + packageName);
		proposal.setAdditionalProposalInfo(path);
		proposal.setContextInformation(new ContextInformation(subPackageName, path));
		return proposal;
	}

	private void addClassSyntax(final int documentOffset, final String previousWord,
			final List<ICompletionProposal> completionProposals, final IResource dir, final IResource sourceDirPath) {

		try {
			sourceDirPath.accept(new IResourceVisitor() {
				public boolean visit(IResource resource) {
					if (resource instanceof IFile && ((IFile) resource).getFileExtension().equals("as")
							&& ((IFile) resource).getName().toLowerCase().startsWith(previousWord.toLowerCase())) {
						String[] parseClassAndPackageName = parseClassAndPackageName(((IFile) resource)
								.getProjectRelativePath().toString(), sourceDirPath.getProjectRelativePath().toString());
						String className = parseClassAndPackageName[0];
						String packageName = parseClassAndPackageName[1];

						DefaultCompletionProposal proposal = createClassCompletionProposal(documentOffset,
								previousWord, className, packageName);

						completionProposals.add(proposal);
					}
					return true;
				}
			});
		} catch (CoreException e) {
			// TODO Auto-generated catch block
			e.printStackTrace();
		}
	}

	protected void addClassSyntax(int documentOffset, String previousWord,
			List<ICompletionProposal> completionProposals, File dir, File sourceDirPath) {

		if (dir == null || !dir.exists() || dir.list() == null) {
			return;
		}
		for (String child : dir.list()) {
			File file = new File(dir, child);
			if (file.isDirectory()) {
				addClassSyntax(documentOffset, previousWord, completionProposals, file, sourceDirPath);
			} else if (child.toLowerCase().startsWith(previousWord.toLowerCase())
					&& child.toLowerCase().endsWith(AS_EXTENSION)) {
				String[] parseClassAndPackageName = parseClassAndPackageName(file.getAbsolutePath(), sourceDirPath
						.getAbsolutePath());
				String className = parseClassAndPackageName[0];
				String packageName = parseClassAndPackageName[1];

				DefaultCompletionProposal proposal = createClassCompletionProposal(documentOffset, previousWord,
						className, packageName);
				completionProposals.add(proposal);
			}
		}
	}

	private String[] parseClassAndPackageName(String path, String sourceDirPath) {
		String sd = sourceDirPath;
		if (!sd.endsWith(DIRECTORY_SEPARATOR)) {
			sd = sd + DIRECTORY_SEPARATOR;
		}
		String work = path.replaceAll("^" + sd, EMPTY);
		work = work.replace(AS_EXTENSION, EMPTY);
		work = work.replace(DIRECTORY_SEPARATOR, CLASS_SEPARATOR);
		int p = work.lastIndexOf(CLASS_SEPARATOR);
		String className = work;
		String packageName = EMPTY;
		if (p >= 0) {
			className = work.substring(work.lastIndexOf(CLASS_SEPARATOR) + 1);
			packageName = work.substring(0, work.lastIndexOf(CLASS_SEPARATOR));
		}
		return new String[] { className, packageName };
	}

	private DefaultCompletionProposal createClassCompletionProposal(int documentOffset, String previousWord,
			String className, String packageName) {
		DefaultCompletionProposal proposal = new ImportCompletionProposal(ISyntaxClass.TYPE_CLASS, className,
				packageName);
		proposal.setReplacementLength(previousWord.length());
		proposal.setReplacementOffset(documentOffset - previousWord.length());
		proposal.setCursorPosition(className.length());
		proposal.setDisplayString(className + " - " + packageName);
		proposal.setAdditionalProposalInfo(packageName + CLASS_SEPARATOR + className);
		proposal.setContextInformation(new ContextInformation(className, packageName + CLASS_SEPARATOR + className));
		return proposal;
	}

}
