package net.sf.amateras.air.debug;

import java.io.File;
import java.io.IOException;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Vector;

import net.sf.amateras.air.AIRPlugin;
import net.sf.amateras.air.debug.debugger.FdbRunner;
import net.sf.amateras.air.debug.debugger.command.AddBreakpointCommand;
import net.sf.amateras.air.debug.debugger.command.AddExpressionCommand;
import net.sf.amateras.air.debug.debugger.command.ContinueCommand;
import net.sf.amateras.air.debug.debugger.command.IDebuggerCommand;
import net.sf.amateras.air.debug.debugger.command.QuitCommand;
import net.sf.amateras.air.debug.debugger.command.RemoveBreakpointCommand;
import net.sf.amateras.air.debug.debugger.command.RemoveExpressionCommand;
import net.sf.amateras.air.debug.debugger.command.RunCommand;
import net.sf.amateras.air.debug.event.IAirEventListener;
import net.sf.amateras.air.debug.matcher.StackframeMatchResult;
import net.sf.amateras.air.debug.thread.AirThread;
import net.sf.amateras.air.debug.watch.AirWatchExpression;
import net.sf.amateras.air.util.ProcessUtil;
import net.sf.amateras.air.util.UIUtil;

import org.eclipse.core.resources.IMarkerDelta;
import org.eclipse.core.resources.IResource;
import org.eclipse.core.runtime.CoreException;
import org.eclipse.debug.core.DebugEvent;
import org.eclipse.debug.core.DebugException;
import org.eclipse.debug.core.DebugPlugin;
import org.eclipse.debug.core.IBreakpointManagerListener;
import org.eclipse.debug.core.IExpressionListener;
import org.eclipse.debug.core.ILaunch;
import org.eclipse.debug.core.model.IBreakpoint;
import org.eclipse.debug.core.model.IDebugTarget;
import org.eclipse.debug.core.model.IExpression;
import org.eclipse.debug.core.model.IMemoryBlock;
import org.eclipse.debug.core.model.IProcess;
import org.eclipse.debug.core.model.IThread;
import org.eclipse.debug.core.model.IWatchExpressionResult;
import org.eclipse.jface.preference.IPreferenceStore;

/**
 * DebugTarget of AIR GEAR.
 * 
 * @author hideko ogawa
 */
public class AirDebugTarget extends AirDebugElement implements IDebugTarget, IBreakpointManagerListener,
		IExpressionListener, IAirEventListener {

	/** executing launch */
	private ILaunch launch;

	/** target file name */
	private String targetExecutingFileName;

	/** terminated state */
	private boolean fTerminated = false;

	/** FDB command runner */
	private FdbRunner fdb;

	// threads
	private IThread[] fThreads;

	private AirThread airThread;

	private List<IAirEventListener> airEventListeners = new Vector<IAirEventListener>();

	private Map<String, IWatchExpressionResult> expressionResult = new HashMap<String, IWatchExpressionResult>();

	/**
	 * constructor
	 * @param launch
	 * @param targetExecutingFileName
	 */
	public AirDebugTarget(ILaunch launch, String targetExecutingFileName) {
		super(null);
		this.launch = launch;
		this.targetExecutingFileName = targetExecutingFileName;

		airThread = new AirThread(this);
		fThreads = new IThread[] { airThread };

		this.fdb = FdbRunner.newInstance();
		fdb.addEventListener(this);
		fdb.addEventListener(airThread);

		getBreakpointManager().addBreakpointListener(this);
		getBreakpointManager().addBreakpointManagerListener(this);
		getExpressionManager().addExpressionListener(this);

		fireCreationEvent();
	}

	public static boolean isAlreadyInstance() {
		if (FdbRunner.getInstance() == null) {
			return false;
		}
		IProcess process = FdbRunner.getInstance().getIProcess();
		if (process.canTerminate()) {
			return true;
		} else {
			FdbRunner.getInstance().dispose();
			return false;
		}
	}

	public String getName() {
		return "AirGear Debug";
	}

	public IProcess getProcess() {
		if (fdb == null) {
			return null;
		} else {
			return fdb.getIProcess();
		}
	}

	public IThread[] getThreads() {
		return fThreads;
	}

	public boolean hasThreads() {
		return true;
	}

	@Override
	public IDebugTarget getDebugTarget() {
		return this;
	}

	@Override
	public ILaunch getLaunch() {
		return launch;
	}

	// *************************************
	// Breakpoint support
	// *************************************
	/**
	 * is supports breakpoint
	 * @param breakpoint
	 * @return boolean 
	 */
	public boolean supportsBreakpoint(IBreakpoint breakpoint) {
		if (breakpoint.getModelIdentifier().equals(AirLineBreakPoint.DEBUG_MODEL_IDENTIFIER)) {
			return true;
		}
		return false;
	}

	/**
	 * when breakpoint is added
	 * @param breakpoint 
	 */
	public void breakpointAdded(IBreakpoint breakpoint) {
		if (supportsBreakpoint(breakpoint)) {
			try {
				if (breakpoint.isEnabled()) {
					AirLineBreakPoint breakPoint = (AirLineBreakPoint) breakpoint;
					addBreakpointToFdb(breakPoint);
				}
			} catch (CoreException e) {
				AIRPlugin.logException(e);
			}
		}
	}

	/**
	 * when changed of breakpoint
	 * @param breakpoint
	 * @param delta
	 */
	public void breakpointChanged(IBreakpoint breakpoint, IMarkerDelta delta) {
		if (supportsBreakpoint(breakpoint)) {
			try {
				if (breakpoint.isEnabled() && getBreakpointManager().isEnabled()) {
					breakpointAdded(breakpoint);
				} else {
					breakpointRemoved(breakpoint, null);
				}
			} catch (CoreException e) {
				AIRPlugin.logException(e);
			}
		}
	}

	/**
	 * when breakpoint is removed
	 * @param breakpoint
	 * @param delta
	 */
	public void breakpointRemoved(IBreakpoint breakpoint, IMarkerDelta delta) {
		if (supportsBreakpoint(breakpoint)) {
			removeBreakPointToFdb((AirLineBreakPoint) breakpoint);
		}
	}

	/**
	 * add breakpoint to fdb.
	 * @param breakpoint
	 */
	private void addBreakpointToFdb(AirLineBreakPoint breakpoint) {
		try {
			int lineNumber = breakpoint.getLineNumber();
			IResource resource = breakpoint.getMarker().getResource();
			fdb.addDebuggerCommand(new AddBreakpointCommand(resource.getName(), lineNumber));
		} catch (CoreException e) {
			e.printStackTrace();
		}
	}

	/**
	 * add breakpoint to fdb.
	 * @param breakpoint
	 */
	private void removeBreakPointToFdb(AirLineBreakPoint breakpoint) {
		try {
			int lineNumber = breakpoint.getLineNumber();
			IResource resource = breakpoint.getMarker().getResource();
			fdb.addDebuggerCommand(new RemoveBreakpointCommand(resource.getName(), lineNumber));
		} catch (CoreException e) {
			e.printStackTrace();
		}
	}

	/**
	 * when breakpoint manager enable is changed
	 * @param enabled
	 */
	public void breakpointManagerEnablementChanged(boolean enabled) {
		System.out.println("breakpointManagerEnablementChanged:" + enabled);
		// FIXME breakpointManagerEnablementChanged!
		//suspendCheck();
	}

	public IMemoryBlock getMemoryBlock(long startAddress, long length) {
		return null;
	}

	public boolean supportsStorageRetrieval() {
		return true;
	}

	// **************************************
	// Expression
	// **************************************
	public IWatchExpressionResult getExpressionResult(String expressionText) {
		return expressionResult.get(expressionText);
	}

	public void addExpressionResult(String expressionText, IWatchExpressionResult result) {
		expressionResult.put(expressionText, result);
	}

	public void expressionAdded(IExpression expression) {
		String expressionText = expression.getExpressionText();
		fdb.addDebuggerCommand(new AddExpressionCommand(expressionText));
	}

	public void expressionRemoved(IExpression expression) {
		if (expression instanceof AirWatchExpression && ((AirWatchExpression) expression).getId() != null) {
			fdb.addDebuggerCommand(new RemoveExpressionCommand(((AirWatchExpression) expression).getId()));
		}
	}

	public void expressionChanged(IExpression expression) {

	}

	// *************************************
	// terminate
	// *************************************
	public boolean canTerminate() {
		if (fTerminated) {
			return false;
		} else {
			if (getProcess() == null) {
				return false;
			} else {
				return getProcess().canTerminate();
			}
		}
	}

	public boolean isTerminated() {
		return fTerminated;
	}

	public void terminate() throws DebugException {
		getBreakpointManager().removeBreakpointListener(this);
		getBreakpointManager().removeBreakpointManagerListener(this);
		getExpressionManager().removeExpressionListener(this);
		fireTerminateEvent();

		fdb.addDebuggerCommand(new QuitCommand());

		fTerminated = true;
		airThread = null;
		fThreads = new IThread[0];

		if (fdb != null) {
			fdb.dispose();
			fdb = null;
		}
		if (launch != null) {
			launch.terminate();
			launch = null;
		}
		fireEvent(new DebugEvent(this, DebugEvent.TERMINATE));

	}

	// *************************************
	// resume
	// *************************************
	public boolean canResume() {
		return isSuspended();
	}

	public void resume() {
		fdb.addDebuggerCommand(new ContinueCommand());
	}

	// *************************************
	// suspend
	// *************************************
	public boolean canSuspend() {
		return false;
	}

	public boolean isSuspended() {
		if (fdb == null) {
			return false;
		} else if (airThread == null) {
			return false;
		} else {
			return fdb.isSuspended();
		}
	}

	public void suspend() {
		//Nothing todo.
	}

	// *************************************
	// Disconnect
	// *************************************

	public void disconnect() throws DebugException {
		terminate();
	}

	public boolean canDisconnect() {
		return canTerminate();
	}

	public boolean isDisconnected() {
		return isTerminated();
	}

	protected void shutdown() {
		try {
			terminate();
		} catch (DebugException e) {
			AIRPlugin.logException(e);
		}
	}

	public void addEventListener(IAirEventListener listener) {
		if (!airEventListeners.contains(listener)) {
			airEventListeners.add(listener);
		}
	}

	public void removeEventListener(IAirEventListener listener) {
		airEventListeners.remove(listener);
	}

	// *******************************************
	// Air Event
	// *******************************************
	public void waitingStartPlayer(String event) {
		fdb.addDebuggerCommand(new RunCommand());
		fireCreationEvent();
	}

	public void waitingConnectPlayer(String event) {
		try {
			startAirApplication();
		} catch (IOException e) {
			AIRPlugin.logException(e);
		}
	}

	public void waitingContinue(String event) {
		fdb.addDebuggerCommand(new ContinueCommand());
		fireResumeEvent(DebugEvent.RESUME);
	}

	/**
	 * 
	 * @param event
	 */
	public void waitingAddBreakpoint(String event) {
		for (IBreakpoint b : getBreakpointManager().getBreakpoints()) {
			if (b instanceof AirLineBreakPoint) {
				addBreakpointToFdb((AirLineBreakPoint) b);
			}
		}
		List<IExpression> newList = new ArrayList<IExpression>();
		for (IExpression expression : getExpressionManager().getExpressions()) {
			AirWatchExpression watch = new AirWatchExpression(expression.getExpressionText());
			newList.add(watch);
			getExpressionManager().removeExpression(expression);
		}
		getExpressionManager().addExpressions(newList.toArray(new IExpression[newList.size()]));

		if (getBreakpointManager().getBreakpoints().length == 0) {
			UIUtil.openMessageDialog(AIRPlugin.getResourceString("ADD_BREAKPOINT_AND_CONTINUE"));
			fireSuspendEvent(DebugEvent.CLIENT_REQUEST);
		} else {
			waitingContinue(event);
		}
	}

	public void suspended(String event, String fileName, int lineNo, int debugType) {
		fireEvent(new DebugEvent(this, DebugEvent.SUSPEND, debugType));
	}

	public void resumed(String event, int debugType) {
		fireEvent(new DebugEvent(this, DebugEvent.RESUME, debugType));
	}

	public void playerTerminated() {
		try {
			fdb.clearProcess();
			terminate();
		} catch (DebugException e) {
			AIRPlugin.logException(e);
		}
	}

	public void resultOfStackFrames(List<StackframeMatchResult> results) {

	}

	@Override
	public void addDebuggerCommand(IDebuggerCommand command) {
		fdb.addDebuggerCommand(command);
	}

	/**
	 * Air Application start!
	 * @throws IOException
	 */
	public IProcess startAirApplication() throws IOException {
		IPreferenceStore store = AIRPlugin.getDefault().getPreferenceStore();
		String adlpath = store.getString(AIRPlugin.PREF_ADL_COMMAND);
		String sdkPath = ProcessUtil.getAirSdkPath(adlpath);

		if (sdkPath == null) {
			return null;
		}

		File executeFile = new File(sdkPath, adlpath);
		String[] command = new String[] { targetExecutingFileName };
		Process process = ProcessUtil.createProcess(executeFile, command, null);
		IProcess iProcess = DebugPlugin.newProcess(launch, process, "\"" + executeFile.getName() + " "
				+ new File(command[0]).getName());
		return iProcess;
	}
}
