package net.sf.amateras.air.debug.debugger;

import java.io.IOException;
import java.util.List;
import java.util.Vector;
import java.util.concurrent.ConcurrentLinkedQueue;

import net.sf.amateras.air.debug.debugger.command.ContinueCommand;
import net.sf.amateras.air.debug.debugger.command.IDebuggerCommand;
import net.sf.amateras.air.debug.event.IAirEventListener;
import net.sf.amateras.air.debug.matcher.BreakpointMatchResult;
import net.sf.amateras.air.debug.matcher.FdbConsoleTextMatcher;
import net.sf.amateras.air.debug.matcher.StackframeMatchResult;

import org.eclipse.debug.core.DebugEvent;
import org.eclipse.debug.core.IStreamListener;
import org.eclipse.debug.core.model.IProcess;
import org.eclipse.debug.core.model.IStreamMonitor;

/**
 * fdb runner for air debug.
 * @author ogawahideko
 *
 */
public class FdbRunner implements IStreamListener {
	private static FdbGateway fdb;
	private static FdbRunner myInstance;

	private ConcurrentLinkedQueue<IDebuggerCommand> commandQueue = new ConcurrentLinkedQueue<IDebuggerCommand>();

	private boolean isSuspended;

	private List<IAirEventListener> fEventListeners = new Vector<IAirEventListener>();

	private FdbRunner() {
		// no instance from other.
		isSuspended = false;
	}

	public static FdbRunner newInstance() {
		if (myInstance == null) {
			myInstance = new FdbRunner();
			myInstance.start();
		} else {
			myInstance.fEventListeners.clear();
		}
		return myInstance;
	}

	public static FdbRunner getInstance() {
		return myInstance;
	}

	/**
	 * 
	 * @param command
	 */
	public void addDebuggerCommand(IDebuggerCommand command) {
		commandQueue.add(command);
		fireQueueChanged();
	}

	private synchronized void fireQueueChanged() {
		//System.out.println("FdbRunner#fireQueueChanged:" + isSuspended);
		if (isSuspended) {
			while (!commandQueue.isEmpty()) {
				IDebuggerCommand command = commandQueue.poll();
				fdb.sendRequest(command.getSendMessage());
				if (command.isResumeCommand()) {
					isSuspended(false);
					fireAirEventOfResumed(null, DebugEvent.RESUME);
				}
			}
		}
	}

	/**
	 * Start fdb.
	 * @param commands
	 * @throws IOException CoreException
	 */
	public boolean start() {
		if (fdb == null) {
			fdb = new FdbGateway();
			try {
				fdb.start();
				fdb.addOutputStreamListener(this);
			} catch (IOException e) {
				e.printStackTrace();
				fdb = null;
				return false;
			}
			return true;
		} else {
			return false;
		}

	}

	public boolean isStarted() {
		return fdb != null;
	}

	/**
	 * Stream append event from StreamBuffer.
	 * @param text
	 * @param monitor
	 */
	public synchronized void streamAppended(final String text, IStreamMonitor monitor) {
		if (FdbConsoleTextMatcher.isPlayerTerminated(text)) {
			commandQueue.clear();
			fireAirEventOfPlayerTerminated();
			isSuspended(false);
			return;
		}

		if (FdbConsoleTextMatcher.isWaitingStartPlayer(text)) {
			isSuspended(true);
			fireAirEventofWaitingStartPlayer(text);
		} else if (FdbConsoleTextMatcher.isWaitingConnectPlayer(text)) {
			isSuspended(true);
			fireAirEventofWaitingConnectPlayer(text);
		} else if (FdbConsoleTextMatcher.isWaitingContinue(text)) {
			isSuspended(true);
			fireAirEventofWaitingContinue(text);
		} else if (FdbConsoleTextMatcher.isWaitingAddBreakpoint(text)) {
			isSuspended(true);
			fireAirEventofWaitingAddBreakpoint(text);
		} else {
			boolean isStopped = false;
			BreakpointMatchResult result = FdbConsoleTextMatcher.breakpointMatch(text);
			if (result.isMatched) {
				isStopped = true;
				fireAirEventOfSuspended(text, result.fileName, result.lineNo, DebugEvent.BREAKPOINT);
			}

			result = FdbConsoleTextMatcher.stepMatch(text);
			if (result.isMatched) {
				isStopped = true;
				fireAirEventOfSuspended(text, result.fileName, result.lineNo, DebugEvent.STEP_OVER);
			}

			List<StackframeMatchResult> stackFrameResults = FdbConsoleTextMatcher.matchStackFramesPattern(text);
			if (stackFrameResults.size() != 0) {
				//isStopped = true;
				fireAirEventOfStackFrames(stackFrameResults);
			}

			if (isStopped) {
				isSuspended(true);
			} else if (stackFrameResults.size() == 0) {
				fireAirEventOfResumed(text, DebugEvent.RESUME);
			}

		}

		//		if (text.endsWith("(fdb) ")) {
		//			isSuspended(true);
		//		} else {
		//			isSuspended(false);
		//		}

	}

	/**
	 * IProcess of FDB
	 * @return
	 */
	public IProcess getIProcess() {
		if (fdb == null) {
			return null;
		} else {
			return fdb.getIProcess();
		}
	}

	public void clearProcess() {
		fdb.clearProcess();
	}

	/**
	 * dispose
	 */
	public void dispose() {
		if (isSuspended()) {
			addDebuggerCommand(new ContinueCommand());
		}
		if (fdb != null) {
			fdb.removeOutputStreamListener(this);
			fdb.stopShell();
			fdb = null;
		}
		myInstance = null;
	}

	public boolean isSuspended() {
		return isSuspended;
	}

	private void isSuspended(boolean isSuspended) {
		this.isSuspended = isSuspended;
		if (isSuspended) {
			fireQueueChanged();
		}
	}

	// *********************************
	// Event
	// *********************************
	public void addEventListener(IAirEventListener listener) {
		if (!fEventListeners.contains(listener)) {
			fEventListeners.add(listener);
		}
	}

	public void removeEventListener(IAirEventListener listener) {
		fEventListeners.remove(listener);
	}

	private void fireAirEventofWaitingStartPlayer(String event) {
		for (IAirEventListener listener : fEventListeners) {
			listener.waitingStartPlayer(event);
		}

	}

	private void fireAirEventofWaitingConnectPlayer(String event) {
		for (IAirEventListener listener : fEventListeners) {
			listener.waitingConnectPlayer(event);
		}
	}

	private void fireAirEventofWaitingContinue(String event) {
		for (IAirEventListener listener : fEventListeners) {
			listener.waitingContinue(event);
		}
	}

	private void fireAirEventofWaitingAddBreakpoint(String event) {
		for (IAirEventListener listener : fEventListeners) {
			listener.waitingAddBreakpoint(event);
		}
	}

	private void fireAirEventOfSuspended(String event, String fileName, int lineNo, int debugEvent) {
		for (IAirEventListener listener : fEventListeners) {
			listener.suspended(event, fileName, lineNo, debugEvent);
		}
	}

	private void fireAirEventOfResumed(String event, int debugEvent) {
		for (IAirEventListener listener : fEventListeners) {
			listener.resumed(event, debugEvent);
		}
	}

	private void fireAirEventOfPlayerTerminated() {
		for (IAirEventListener listener : fEventListeners) {
			listener.playerTerminated();
		}
	}

	private void fireAirEventOfStackFrames(List<StackframeMatchResult> results) {
		for (IAirEventListener listener : fEventListeners) {
			listener.resultOfStackFrames(results);
		}
	}
}
