package net.sf.amateras.air.debug.ruler;

import java.util.Iterator;
import java.util.List;

import net.sf.amateras.air.AIRPlugin;
import net.sf.amateras.air.debug.AirLineBreakPoint;

import org.eclipse.core.resources.IMarker;
import org.eclipse.core.resources.IResource;
import org.eclipse.core.resources.IWorkspaceRoot;
import org.eclipse.core.resources.IWorkspaceRunnable;
import org.eclipse.core.runtime.CoreException;
import org.eclipse.core.runtime.IProgressMonitor;
import org.eclipse.debug.core.DebugPlugin;
import org.eclipse.debug.core.IBreakpointManager;
import org.eclipse.debug.core.model.IBreakpoint;
import org.eclipse.jface.action.Action;
import org.eclipse.jface.text.BadLocationException;
import org.eclipse.jface.text.IDocument;
import org.eclipse.jface.text.source.IVerticalRulerInfo;
import org.eclipse.ui.IEditorInput;
import org.eclipse.ui.texteditor.ITextEditor;
import org.eclipse.ui.texteditor.IUpdate;

/**
 * Setting/removing breakpoints in the ruler
 * 
 */

public class BreakpointRulerAction extends AbstractBreakpointRulerAction {

	private List<IMarker> fMarkers;

	private String fAddLabel;

	private String fRemoveLabel;

	public BreakpointRulerAction(ITextEditor editor, IVerticalRulerInfo ruler) {
		setInfo(ruler);
		setTextEditor(editor);
		setText("Breakpoint &Properties...");
		fAddLabel = "Add Breakpoint";
		fRemoveLabel = "Remove Breakpoint";
	}

	/**
	 * @see IUpdate#update()
	 */
	public void update() {
		fMarkers = getMarkersFromCurrentFile();
		setText(fMarkers.isEmpty() ? fAddLabel : fRemoveLabel);
	}

	/**
	 * @see Action#run()
	 */
	@Override
	public void run() {
		if (fMarkers.isEmpty()) {
			addMarker();
		} else {
			removeMarkers(fMarkers);
		}
	}

	protected List<IMarker> getMarkersFromCurrentFile() {
		return getMarkersFromEditorResource(getResourceForDebugMarkers(), getDocument(), getTextEditor()
				.getEditorInput(), getInfo(), true);
	}

	/**
	 * This is the function that actually adds the marker to the Eclipse
	 * structure.
	 */
	protected void addMarker() {
		try {
			IDocument document = getDocument();
			int rulerLine = getInfo().getLineOfLastMouseButtonActivity();

			final int lineNumber = rulerLine + 1;
			if (lineNumber < 0) {
				return;
			}

			document.getLineInformation(lineNumber - 1);
			final IResource resource = getResourceForDebugMarkers();
			final IEditorInput editorInput = getTextEditor().getEditorInput();

			if (editorInput == null && resource instanceof IWorkspaceRoot) {
				return;
			}

			//System.out.println("[add debug marker]" + resource + ":" + lineNumber);

			IWorkspaceRunnable runnable = new IWorkspaceRunnable() {
				public void run(IProgressMonitor monitor) throws CoreException {
					AirLineBreakPoint breakpoint = new AirLineBreakPoint();
					breakpoint.addBreakPointManager(resource, lineNumber);
				}
			};

			resource.getWorkspace().run(runnable, null);
		} catch (BadLocationException e) {
			e.printStackTrace();
		} catch (CoreException e) {
			e.printStackTrace();
		}
	}

	/**
	 * @param markers the markers that will be removed in this function (they may be in any editor, not only in the current one)
	 */
	protected void removeMarkers(List<IMarker> markers) {
		IBreakpointManager breakpointManager = DebugPlugin.getDefault().getBreakpointManager();
		try {
			Iterator<IMarker> e = markers.iterator();
			while (e.hasNext()) {
				IBreakpoint breakpoint = breakpointManager.getBreakpoint(e.next());
				breakpointManager.removeBreakpoint(breakpoint, true);
			}
		} catch (CoreException e) {
			AIRPlugin.logException(e);
		}
	}
}
