package net.sf.amateras.air.mxml.descriptor.editor;

import java.text.MessageFormat;

import org.eclipse.jface.viewers.DialogCellEditor;
import org.eclipse.swt.SWT;
import org.eclipse.swt.events.FocusAdapter;
import org.eclipse.swt.events.FocusEvent;
import org.eclipse.swt.events.FocusListener;
import org.eclipse.swt.events.KeyAdapter;
import org.eclipse.swt.events.KeyEvent;
import org.eclipse.swt.events.ModifyEvent;
import org.eclipse.swt.events.ModifyListener;
import org.eclipse.swt.events.MouseAdapter;
import org.eclipse.swt.events.MouseEvent;
import org.eclipse.swt.events.SelectionAdapter;
import org.eclipse.swt.events.SelectionEvent;
import org.eclipse.swt.events.TraverseEvent;
import org.eclipse.swt.events.TraverseListener;
import org.eclipse.swt.graphics.Color;
import org.eclipse.swt.graphics.Point;
import org.eclipse.swt.graphics.Rectangle;
import org.eclipse.swt.widgets.Button;
import org.eclipse.swt.widgets.Composite;
import org.eclipse.swt.widgets.Control;
import org.eclipse.swt.widgets.Label;
import org.eclipse.swt.widgets.Layout;
import org.eclipse.swt.widgets.Text;

public abstract class AbstractTextDialogCellEditor extends DialogCellEditor {

	private static final int GAP = 6;

	private Composite composite;

	protected Label imageLabel;

	protected Text text;

	private ModifyListener modifyListener;

	private FocusListener buttonFocusListener;

	private Button button;

	private boolean isSelection = false;

	private boolean isDeleteable = false;

	private boolean isSelectable = false;

	protected abstract Object getDefaultValue();

	public AbstractTextDialogCellEditor(Composite parent) {
		super(parent, SWT.NONE);
		doSetValue(getDefaultValue());
	}

	protected Layout getLayout() {
		return new Layout() {
			@Override
			public Point computeSize(Composite editor, int wHint, int hHint, boolean force) {
				if (wHint != SWT.DEFAULT && hHint != SWT.DEFAULT) {
					return new Point(wHint, hHint);
				}
				Point colorSize = imageLabel.computeSize(SWT.DEFAULT, SWT.DEFAULT, force);
				Point rgbSize = text.computeSize(SWT.DEFAULT, SWT.DEFAULT, force);
				return new Point(colorSize.x + GAP + rgbSize.x, Math.max(colorSize.y, rgbSize.y));
			}

			@Override
			public void layout(Composite editor, boolean force) {
				Rectangle bounds = editor.getClientArea();
				Point colorSize = imageLabel.computeSize(SWT.DEFAULT, SWT.DEFAULT, force);
				Point rgbSize = text.computeSize(SWT.DEFAULT, SWT.DEFAULT, force);
				int ty = (bounds.height - rgbSize.y) / 2;
				if (ty < 0) {
					ty = 0;
				}
				imageLabel.setBounds(-1, 0, colorSize.x, colorSize.y);
				text.setBounds(colorSize.x + GAP - 1, ty, bounds.width - colorSize.x - GAP, bounds.height);
			}
		};
	}

	@Override
	protected Control createContents(Composite cell) {
		Color bg = cell.getBackground();
		composite = new Composite(cell, getStyle());
		composite.setBackground(bg);
		composite.setLayout(getLayout());
		imageLabel = new Label(composite, SWT.LEFT);
		imageLabel.setBackground(bg);

		text = createTextControl(composite);
		text.setBackground(bg);
		text.setFont(cell.getFont());

		return composite;
	}

	// ***************************************************
	@Override
	protected Button createButton(Composite parent) {
		button = super.createButton(parent);
		return button;
	}

	private Text createTextControl(Composite parent) {
		Text text = new Text(parent, getStyle());
		text.setFont(parent.getFont());
		text.setBackground(parent.getBackground());
		text.addSelectionListener(new SelectionAdapter() {
			@Override
			public void widgetDefaultSelected(SelectionEvent e) {
				handleDefaultSelection(e);
			}
		});
		text.addKeyListener(new KeyAdapter() {
			@Override
			public void keyPressed(KeyEvent e) {
				keyReleaseOccured(e);
				if ((getControl() == null) || getControl().isDisposed()) {
					return;
				}
				checkSelection(); // see explanation below
				checkDeleteable();
				checkSelectable();
			}
		});
		text.addTraverseListener(new TraverseListener() {
			public void keyTraversed(TraverseEvent e) {
				if (e.detail == SWT.TRAVERSE_ESCAPE || e.detail == SWT.TRAVERSE_RETURN) {
					e.doit = false;
				}
			}
		});
		text.addMouseListener(new MouseAdapter() {
			@Override
			public void mouseUp(MouseEvent e) {
				checkSelection();
				checkDeleteable();
				checkSelectable();
			}
		});
		//		text.addFocusListener(new FocusAdapter() {
		//			@Override
		//			public void focusLost(FocusEvent e) {
		//				AbstractTextDialogCellEditor.this.focusLost();
		//			}
		//		});
		text.setFont(parent.getFont());
		text.setBackground(parent.getBackground());
		text.setText("");
		text.addModifyListener(getModifyListener());
		return text;
	}

	private void checkSelection() {
		boolean oldIsSelection = isSelection;
		isSelection = text.getSelectionCount() > 0;
		if (oldIsSelection != isSelection) {
			fireEnablementChanged(COPY);
			fireEnablementChanged(CUT);
		}
	}

	private void checkDeleteable() {
		boolean oldIsDeleteable = isDeleteable;
		isDeleteable = isDeleteEnabled();
		if (oldIsDeleteable != isDeleteable) {
			fireEnablementChanged(DELETE);
		}
	}

	private void checkSelectable() {
		boolean oldIsSelectable = isSelectable;
		isSelectable = isSelectAllEnabled();
		if (oldIsSelectable != isSelectable) {
			fireEnablementChanged(SELECT_ALL);
		}
	}

	protected void handleDefaultSelection(SelectionEvent event) {
		fireApplyEditorValue();
		deactivate();
	}

	private ModifyListener getModifyListener() {
		if (modifyListener == null) {
			modifyListener = new ModifyListener() {
				public void modifyText(ModifyEvent e) {
					editOccured(e);
				}
			};
		}
		return modifyListener;
	}

	protected void editOccured(ModifyEvent e) {
		String value = text.getText();
		if (value == null) {
			value = "";
		}
		Object typedValue = value;
		boolean oldValidState = isValueValid();
		boolean newValidState = isCorrect(typedValue);
		//if (newValidState) {
		//	Assert.isTrue(false, "Validator isn't limiting the cell editor's type range");//$NON-NLS-1$
		//}
		if (!newValidState) {
			setErrorMessage(MessageFormat.format(getErrorMessage(), new Object[] { value }));
		}
		valueChanged(oldValidState, newValidState);
	}

	@Override
	protected void fireApplyEditorValue() {
		doSetValue(text.getText());
		super.fireApplyEditorValue();
	}

	@Override
	protected void doSetFocus() {
		super.doSetFocus();
		text.setFocus();
		text.addFocusListener(getButtonFocusListener());
	}

	private FocusListener getButtonFocusListener() {
		if (buttonFocusListener == null) {
			buttonFocusListener = new FocusAdapter() {
				@Override
				public void focusLost(FocusEvent e) {
					if (!button.isFocusControl() && text.isEnabled()) {
						AbstractTextDialogCellEditor.this.focusLost();
					}
				}
			};
		}
		return buttonFocusListener;
	}
}
