package net.sf.amateras.air.mxml.editparts.policy;

import java.util.List;

import net.sf.amateras.air.mxml.editparts.command.AddCommand;
import net.sf.amateras.air.mxml.editparts.command.ChangeConstraintCommand;
import net.sf.amateras.air.mxml.editparts.command.CreateCommand;
import net.sf.amateras.air.mxml.editparts.command.MoveChildCommand;
import net.sf.amateras.air.mxml.editparts.command.OrphanChildrenCommand;
import net.sf.amateras.air.mxml.models.FlexRectangle;
import net.sf.amateras.air.mxml.models.IComponentModel;
import net.sf.amateras.air.mxml.models.IContainerModel;

import org.eclipse.draw2d.FlowLayout;
import org.eclipse.draw2d.IFigure;
import org.eclipse.draw2d.LayoutManager;
import org.eclipse.draw2d.ToolbarLayout;
import org.eclipse.draw2d.geometry.Point;
import org.eclipse.draw2d.geometry.PrecisionRectangle;
import org.eclipse.draw2d.geometry.Rectangle;
import org.eclipse.gef.EditPart;
import org.eclipse.gef.EditPolicy;
import org.eclipse.gef.GraphicalEditPart;
import org.eclipse.gef.Request;
import org.eclipse.gef.commands.Command;
import org.eclipse.gef.commands.CompoundCommand;
import org.eclipse.gef.editpolicies.FlowLayoutEditPolicy;
import org.eclipse.gef.editpolicies.ResizableEditPolicy;
import org.eclipse.gef.requests.AlignmentRequest;
import org.eclipse.gef.requests.ChangeBoundsRequest;
import org.eclipse.gef.requests.CreateRequest;
import org.eclipse.gef.requests.GroupRequest;

/**
 * EditPolicy for FlowLayout or ToolbarLayout.
 * @author ogawahideko
 *
 */
public class ResizeableFlowLayoutEditPolicy extends FlowLayoutEditPolicy {

	@Override
	public boolean isHorizontal() {
		IFigure figure = ((GraphicalEditPart) getHost()).getContentPane();
		LayoutManager layout = figure.getLayoutManager();
		if (layout instanceof FlowLayout) {
			return ((FlowLayout) figure.getLayoutManager()).isHorizontal();
		}
		if (layout instanceof ToolbarLayout) {
			return ((ToolbarLayout) figure.getLayoutManager()).isHorizontal();
		}
		return false;
	}

	@Override
	protected Command createAddCommand(EditPart child, EditPart after) {
		IContainerModel root = (IContainerModel) getHost().getModel();
		IComponentModel childModel = (IComponentModel) child.getModel();
		IComponentModel afterModel = after == null ? null : (IComponentModel) after.getModel();
		return new AddCommand(root, childModel, afterModel, childModel.getConstraint());
	}

	@Override
	protected Command createMoveChildCommand(EditPart child, EditPart after) {
		return new MoveChildCommand(getHost(), child, after);
	}

	@Override
	protected Command getCreateCommand(CreateRequest request) {
		IContainerModel root = (IContainerModel) getHost().getModel();
		return new CreateCommand(root, request, null);
	}

	//	@Override
	//	protected Command getCloneCommand(ChangeBoundsRequest request) {
	//		return new CloneCommand(getHost(), request);
	//	}

	@Override
	@SuppressWarnings("unchecked")
	protected Command getOrphanChildrenCommand(Request request) {
		GroupRequest gRequest = (GroupRequest) request;
		return new OrphanChildrenCommand(gRequest.getEditParts());
	}

	@Override
	protected EditPolicy createChildEditPolicy(EditPart child) {
		return new ResizableEditPolicy();
	}

	@Override
	public Command getCommand(Request request) {
		if (REQ_RESIZE_CHILDREN.equals(request.getType())) {
			return getResizeChildrenCommand((ChangeBoundsRequest) request);
		}
		if (REQ_ALIGN_CHILDREN.equals(request.getType())) {
			return getAlignChildrenCommand((AlignmentRequest) request);
		}

		return super.getCommand(request);
	}

	protected Command getAlignChildrenCommand(AlignmentRequest request) {
		return getResizeChildrenCommand(request);
	}

	protected Object getConstraintFor(ChangeBoundsRequest request, GraphicalEditPart child) {
		Rectangle rect = new PrecisionRectangle(child.getFigure().getBounds());
		child.getFigure().translateToAbsolute(rect);
		rect = request.getTransformedRectangle(rect);
		child.getFigure().translateToRelative(rect);
		rect.translate(getLayoutOrigin().getNegated());
		return getConstraintFor(rect);
	}

	public Object getConstraintFor(Rectangle r) {
		return new Rectangle(r);
	}

	protected Point getLayoutOrigin() {
		return getLayoutContainer().getClientArea().getLocation();
	}

	protected Command getResizeChildrenCommand(ChangeBoundsRequest request) {
		CompoundCommand resize = new CompoundCommand();
		Command c;
		GraphicalEditPart child;

		@SuppressWarnings("unchecked")
		List<EditPart> children = request.getEditParts();

		for (int i = 0; i < children.size(); i++) {
			child = (GraphicalEditPart) children.get(i);
			c = createChangeConstraintCommand(request, child, translateToModelConstraint(getConstraintFor(request,
					child)));
			resize.add(c);
		}
		return resize.unwrap();
	}

	protected Object translateToModelConstraint(Object figureConstraint) {
		return figureConstraint;
	}

	protected Command createChangeConstraintCommand(ChangeBoundsRequest request, EditPart child, Object constraint) {
		return createChangeConstraintCommand(child, constraint);
	}

	protected Command createChangeConstraintCommand(EditPart child, Object constraint) {
		Rectangle rectangle = (Rectangle) constraint;
		snapToGrid(rectangle);

		ChangeConstraintCommand command = new ChangeConstraintCommand();
		command.setModel((IComponentModel) child.getModel());
		FlexRectangle fr = new FlexRectangle();
		fr.setWidth(rectangle.width);
		fr.setHeight(rectangle.height);
		command.setConstraint(fr);
		return command;
	}

	private void snapToGrid(Rectangle rectangle) {
		rectangle.x = roundValue(rectangle.x);
		rectangle.y = roundValue(rectangle.y);
		rectangle.width = roundValue(rectangle.width);
		rectangle.height = roundValue(rectangle.height);
	}

	private int roundValue(int value) {
		int div = value % 10;
		if (div >= 5) {
			return value + (10 - div);
		} else {
			return value - div;
		}
	}

}
