package net.sf.amateras.air.mxml.models;

import java.beans.PropertyChangeListener;
import java.beans.PropertyChangeSupport;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.Iterator;
import java.util.LinkedHashMap;
import java.util.List;
import java.util.Map;

import net.sf.amateras.air.mxml.descriptor.ColorChoosePropertyDescriptor;
import net.sf.amateras.air.mxml.descriptor.IEditorValueDescriptor;
import net.sf.amateras.air.mxml.descriptor.ImagePropertyDescriptor;
import net.sf.amateras.air.mxml.descriptor.StringListPropertyDescriptor;
import net.sf.amateras.air.mxml.descriptor.StringPropertyDescriptor;
import net.sf.amateras.air.mxml.descriptor.WidthHeightPropertyDescriptor;
import net.sf.amateras.air.util.ColorUtil;
import net.sf.amateras.air.util.XMLUtil;

import org.eclipse.swt.graphics.RGB;
import org.eclipse.ui.views.properties.IPropertyDescriptor;
import org.eclipse.ui.views.properties.IPropertySource;
import org.w3c.dom.Element;

/**
 * This is a base class for Flex component models.
 * 
 * @author Naoki Takezoe
 */
public abstract class AbstractModel implements IModel, IPropertySource {
	private static final String UNDER_SCORE = "_";
	public static final String NULL_PROPERTY = "";
	private Map<Object, ModelProperty> properties;

	private PropertyChangeSupport listeners = new PropertyChangeSupport(this);
	private Map<Object, Object> propertyAttributes = new LinkedHashMap<Object, Object>();
	private Map<String, Object> additionalAttributes = new HashMap<String, Object>();

	private List<Element> additionalChildElements = new ArrayList<Element>();

	public AbstractModel() {
		properties = PropertyManager.getProperties(getClass().getName());
		if (properties == null) {
			properties = new HashMap<Object, ModelProperty>();
			installModelProperty();
			PropertyManager.putProperties(getClass().getName(), properties);
		}
	}

	protected void installModelProperty() {
		addStringModelProperty(ID, CATEGORY_COMMON, null);
	}

	/**
	 * @see IModel#setAdditionalAttributes(Map)
	 */
	public void setAdditionalAttributes(Map<String, Object> additionalAttributes) {
		this.additionalAttributes = additionalAttributes;
	}

	/**
	 * @see IModel#getAdditionalAttributes()
	 */
	public Map<String, Object> getAdditionalAttributes() {
		return this.additionalAttributes;
	}

	/**
	 * @see IModel#addAdditionalChildElement(Element)
	 */
	public void addAdditionalChildElement(Element childNode) {
		additionalChildElements.add(childNode);
	}

	/**
	 * @see IModel#getAdditionalChildElements()
	 */
	public List<Element> getAdditionalChildElements() {
		return additionalChildElements;
	}

	public void addPropertyChangeListener(PropertyChangeListener listener) {
		listeners.addPropertyChangeListener(listener);
	}

	public void firePropertyChange(String propName, Object oldValue, Object newValue) {
		listeners.firePropertyChange(propName, oldValue, newValue);
	}

	public void removePropertyChangeListener(PropertyChangeListener listener) {
		listeners.removePropertyChangeListener(listener);
	}

	protected ModelProperty addModelProperty(Object id, ModelProperty property) {
		properties.put(id, property);
		return property;
	}

	public ModelProperty getModelProperty(Object id) {
		if (properties == null) {
			return null;
		}
		return properties.get(id);
	}

	public Map<Object, ModelProperty> getModelProperties() {
		return properties;
	}

	/**
	 * @see IPropertySource#getPropertyDescriptors()
	 */
	public IPropertyDescriptor[] getPropertyDescriptors() {
		List<IPropertyDescriptor> descriptors = new ArrayList<IPropertyDescriptor>();
		for (Iterator<ModelProperty> ite = this.properties.values().iterator(); ite.hasNext();) {
			ModelProperty property = ite.next();
			IPropertyDescriptor descriptor = property.getDescriptor();
			if (descriptor != null) {
				descriptors.add(descriptor);
			}
		}
		return descriptors.toArray(new IPropertyDescriptor[descriptors.size()]);
	}

	/** @see IPropertySource#getPropertyValue(Object) */
	public Object getPropertyValue(Object id) {
		if (getModelProperty(id) == null) {
			//System.err.println("No ModelProperty id=" + id);
			return null;
		}
		Object prop = propertyAttributes.get(id);
		if (prop == null) {
			IPropertyDescriptor descriptor = getModelProperty(id).getDescriptor();
			if (descriptor instanceof ColorChoosePropertyDescriptor) {
				return ((ColorChoosePropertyDescriptor) descriptor).getDefaultValue();
			} else {
				return NULL_PROPERTY;
			}
		} else {
			return prop;
		}
	}

	/**
	 * @see IPropertySource#isPropertySet(Object)
	 */
	public boolean isPropertySet(Object id) {
		return properties.containsKey(id);
	}

	/**
	 * @see IPropertySource#resetPropertyValue(Object)
	 */
	public void resetPropertyValue(Object id) {
		propertyAttributes.remove(id);
	}

	/** @see IPropertySource#setPropertyValue(Object, Object) */
	public void setPropertyValue(Object id, Object value) {
		setPropertyValue(id, value, true);
	}

	public void setPropertyValue(Object id, Object value, boolean isFirePropertyChange) {
		if (value == null || value.equals(NULL_PROPERTY) || value.toString().length() == 0) {
			resetPropertyValue(id);
			return;
		}
		Object oldValue = propertyAttributes.get(id);
		propertyAttributes.put(id, value);
		if (isFirePropertyChange) {
			firePropertyChange(id.toString(), oldValue, value);
		}
	}

	@SuppressWarnings("unchecked")
	private Object getRealPropertyValue(Object id) {
		if (getModelProperty(id) == null) {
			//System.err.println("No ModelProperty id=" + id);
			return null;
		}
		IPropertyDescriptor descriptor = getModelProperty(id).getDescriptor();
		if (descriptor instanceof IEditorValueDescriptor) {
			try {
				Object value = ((IEditorValueDescriptor<Object>) descriptor).getEditorValue(propertyAttributes.get(id));
				return value;
			} catch (Exception ex) {
				System.err.println("UnMatch DataType. model=" + this + ",id=" + id + ", value="
						+ propertyAttributes.get(id).getClass());
				ex.printStackTrace();
			}
		}
		return propertyAttributes.get(id);
	}

	@SuppressWarnings("unchecked")
	public boolean isDefaultValue(Object id, Object value) {
		if (value == null) {
			return true;
		} else {
			IPropertyDescriptor descriptor = getModelProperty(id).getDescriptor();
			if (descriptor instanceof IEditorValueDescriptor) {
				Object defaultValue = ((IEditorValueDescriptor) descriptor).getDefaultValue();
				if (value.equals(defaultValue)) {
					return true;
				}
			}
			return false;
		}
	}

	public Object getEditableValue() {
		return this;
	}

	private boolean isLayoutKey(Object id) {
		return id.equals(UNDER_SCORE + X) || id.equals(UNDER_SCORE + Y) || id.equals(UNDER_SCORE + WIDTH)
				|| id.equals(UNDER_SCORE + HEIGHT) ? true : false;
	}

	///////////////////////////////////////////////////////
	// To XML

	protected String getAttributesXML() {
		StringBuilder sb = new StringBuilder();
		StringBuilder layoutInfos = new StringBuilder();

		// properties
		Iterator<Object> iterator = getPropertyAttributes().keySet().iterator();
		while (iterator.hasNext()) {
			String key = (String) iterator.next();
			if (!isOutputAtribute(key.substring(1))) {
				continue;
			}
			if (isLayoutKey(key)) {
				addAttributeXml(layoutInfos, key);
			} else {
				addAttributeXml(sb, key);
			}
		}

		sb.append(layoutInfos.toString());
		return sb.toString();
	}

	protected boolean isOutputAtribute(String propertyName) {
		return true;
	}

	private void addAttributeXml(StringBuilder sb, String key) {
		Object value = getPropertyAttributes().get(key);
		if (value != null) {
			if (value instanceof RGB) {
				sb.append(" ").append(key.substring(1)).append("=\"");
				sb.append(ColorUtil.toHex((RGB) value)).append("\"");
			} else {
				if (value.toString().length() != 0) {
					sb.append(" ").append(key.substring(1)).append("=\"");
					sb.append(XMLUtil.escape(value.toString())).append("\"");
				}
			}
		}
	}

	protected String getAdditionalAttributesXML() {
		StringBuffer sb = new StringBuffer();
		Iterator<Map.Entry<String, Object>> ite = this.additionalAttributes.entrySet().iterator();
		while (ite.hasNext()) {
			Map.Entry<String, Object> entry = ite.next();
			sb.append(" ");
			sb.append(entry.getKey());
			sb.append("=\"");
			sb.append(XMLUtil.escape((String) entry.getValue()));
			sb.append("\"");
		}
		return sb.toString();
	}

	protected String getAdditionalChildElementsXML() {
		StringBuilder sb = new StringBuilder();
		if (additionalChildElements.size() != 0) {
			Iterator<Element> ite = additionalChildElements.iterator();
			while (ite.hasNext()) {
				sb.append(XMLUtil.nodeToXML(ite.next()));
			}
			sb.append("\n");
		}
		return sb.toString();
	}

	public abstract String toMXML();

	///////////////////////////////////////////////////////
	// Property Methods

	protected ModelProperty addStringModelProperty(String propertyName, String category, String defaultValue) {
		return addModelProperty(UNDER_SCORE + propertyName, new ModelProperty(propertyName,
				new StringPropertyDescriptor(UNDER_SCORE + propertyName, propertyName, defaultValue), category));
	}

	protected ModelProperty addStringModelProperty(String propertyName, String category) {
		return addStringModelProperty(propertyName, category, null);
	}

	private static final String[] LIST_BOOLEAN = { "true", "false" };

	protected ModelProperty addBooleanModelProperty(String propertyName, String category, boolean defaultValue) {
		return addListModelProperty(propertyName, category, LIST_BOOLEAN, String.valueOf(defaultValue));
	}

	protected ModelProperty addBooleanModelProperty(String propertyName, String category) {
		return addListModelProperty(propertyName, category, LIST_BOOLEAN, "");
	}

	protected ModelProperty addNumberModelProperty(String propertyName, String category, Integer defaultValue) {
		return addModelProperty(UNDER_SCORE + propertyName, new ModelProperty(propertyName,
				new StringPropertyDescriptor(UNDER_SCORE + propertyName, propertyName, defaultValue == null ? ""
						: String.valueOf(defaultValue)), category));
	}

	protected ModelProperty addNumberModelProperty(String propertyName, String category) {
		return addNumberModelProperty(propertyName, category, null);
	}

	protected ModelProperty addIntModelProperty(String propertyName, String category, int defaultValue) {
		return addNumberModelProperty(propertyName, category, defaultValue);
	}

	protected ModelProperty addIntModelProperty(String propertyName, String category) {
		return addNumberModelProperty(propertyName, category, null);
	}

	protected ModelProperty addWidthHeightModelProperty(String propertyName, String category, String defaultValue) {
		return addModelProperty(UNDER_SCORE + propertyName, new ModelProperty(propertyName,
				new WidthHeightPropertyDescriptor(UNDER_SCORE + propertyName, propertyName, String
						.valueOf(defaultValue)), category));
	}

	protected ModelProperty addDoubleModelProperty(String propertyName, String category, Double defaultValue) {
		if (defaultValue == null) {
			return addModelProperty(UNDER_SCORE + propertyName, new ModelProperty(propertyName,
					new StringPropertyDescriptor(UNDER_SCORE + propertyName, propertyName, null), category));

		} else {
			return addModelProperty(UNDER_SCORE + propertyName,
					new ModelProperty(propertyName, new StringPropertyDescriptor(UNDER_SCORE + propertyName,
							propertyName, String.valueOf(defaultValue)), category));
		}
	}

	protected ModelProperty addColorModelProperty(String propertyName, String category, RGB defaultValue) {
		return addModelProperty(UNDER_SCORE + propertyName, new ModelProperty(propertyName,
				new ColorChoosePropertyDescriptor(UNDER_SCORE + propertyName, propertyName, defaultValue), category));
	}

	protected ModelProperty addColorModelProperty(String propertyName, String category) {
		return addColorModelProperty(propertyName, category, null);
	}

	protected ModelProperty addImageModelProperty(String propertyName, String category, String defaultValue) {
		return addModelProperty(UNDER_SCORE + propertyName, new ModelProperty(propertyName,
				new ImagePropertyDescriptor(UNDER_SCORE + propertyName, propertyName, defaultValue), category));
	}

	protected ModelProperty addImageModelProperty(String propertyName, String category) {
		return addImageModelProperty(propertyName, category, null);
	}

	protected ModelProperty addListModelProperty(String propertyName, String category, String[] list) {
		return addModelProperty(UNDER_SCORE + propertyName, new ModelProperty(propertyName,
				new StringListPropertyDescriptor(UNDER_SCORE + propertyName, propertyName, list), category));
	}

	protected ModelProperty addListModelProperty(String propertyName, String category, String[] list,
			String defaultValue) {
		return addModelProperty(UNDER_SCORE + propertyName, new ModelProperty(propertyName,
				new StringListPropertyDescriptor(UNDER_SCORE + propertyName, propertyName, list, defaultValue),
				category));
	}

	protected ModelProperty addListModelProperty(String propertyName, String category) {
		return addListModelProperty(propertyName, category, null);
	}

	protected void removePropertySheet(String propertyName) {
		ModelProperty mp = getModelProperty(UNDER_SCORE + propertyName);
		mp.setDescriptor(null);
	}

	protected void setAdvancedToModel(String propertyName, boolean isAdvanced) {
		ModelProperty mp = getModelProperty(UNDER_SCORE + propertyName);
		mp.setAdvanced(isAdvanced);
	}

	public Map<Object, Object> getPropertyAttributes() {
		return propertyAttributes;
	}

	// ***********************************************
	/**
	 * @see IModel#setAttribute(String, Object)
	 */
	public void setAttribute(String propertyName, Object value) {
		setPropertyValue(UNDER_SCORE + propertyName, value);
	}

	/** 
	 * @see IModel#getAttribute(String)
	 */
	public Object getAttribute(String propertyName) {
		Object value = getRealPropertyValue(UNDER_SCORE + propertyName);
		return value;
	}

	/**
	 * @see IModel#getAttributeToString(String)
	 */
	public String getAttributeToString(String propertyName) {
		Object value = getRealPropertyValue(UNDER_SCORE + propertyName);
		if (value == null) {
			return "";
		} else {
			return value.toString();
		}
	}

	public int getAttributeToNumber(String propertyName) {
		Object value = getRealPropertyValue(UNDER_SCORE + propertyName);
		return Integer.parseInt(value.toString());
	}

}
