package net.sf.amateras.air.util;

import java.io.File;
import java.io.IOException;
import java.util.ArrayList;
import java.util.List;

import org.eclipse.core.resources.IFile;
import org.eclipse.core.resources.IProject;
import org.eclipse.core.resources.IResource;
import org.eclipse.core.resources.IWorkspace;
import org.eclipse.core.resources.ResourcesPlugin;
import org.eclipse.core.runtime.IPath;
import org.eclipse.core.runtime.Path;
import org.eclipse.jface.window.Window;
import org.eclipse.swt.widgets.Display;
import org.eclipse.swt.widgets.Shell;
import org.eclipse.ui.IEditorInput;
import org.eclipse.ui.PlatformUI;
import org.eclipse.ui.dialogs.ElementListSelectionDialog;
import org.eclipse.ui.model.WorkbenchLabelProvider;
import org.eclipse.ui.part.FileEditorInput;

public class ResourceUtil {

	private ResourceUtil() {
	}

	public static IEditorInput createEditorInput(String absolutePath) {
		IFile[] workspaceFile = getWorkspaceFiles(absolutePath);
		if (workspaceFile != null && workspaceFile.length > 0) {
			IFile file2 = selectWorkspaceFile(workspaceFile);
			if (file2 != null) {
				return new FileEditorInput(file2);
			} else {
				return new FileEditorInput(workspaceFile[0]);
			}
		}
		return null;
	}

	public static IFile[] getWorkspaceFiles(String absolutePath) {
		IWorkspace workspace = ResourcesPlugin.getWorkspace();
		absolutePath = workspace.getRoot().getFullPath().toOSString() + "/" + absolutePath;
		IPath location = Path.fromOSString(absolutePath);
		location = Path.fromPortableString(absolutePath);

		IFile[] files = workspace.getRoot().findFilesForLocation(location);

		files = filterNonExistentFiles(files);
		if (files == null || files.length == 0) {
			return null;
		}

		return files;
	}

	private static IFile[] filterNonExistentFiles(IFile[] files) {
		if (files == null) {
			return null;
		}

		int length = files.length;
		ArrayList<IFile> existentFiles = new ArrayList<IFile>(length);
		for (int i = 0; i < length; i++) {
			if (files[i].exists()) {
				existentFiles.add(files[i]);
			}
		}
		return existentFiles.toArray(new IFile[existentFiles.size()]);
	}

	private static IFile selectWorkspaceFile(final IFile[] files) {
		if (files == null || files.length == 0) {
			return null;
		}
		if (files.length == 1) {
			return files[0];
		}
		final List<IFile> selected = new ArrayList<IFile>();

		Runnable r = new Runnable() {
			public void run() {
				Shell shell = PlatformUI.getWorkbench().getActiveWorkbenchWindow().getShell();
				ElementListSelectionDialog dialog = new ElementListSelectionDialog(shell, new WorkbenchLabelProvider());
				dialog.setElements(files);
				dialog.setTitle("Select Workspace File");
				dialog.setMessage("File may be matched to multiple files in the workspace.");
				if (dialog.open() == Window.OK) {
					selected.add((IFile) dialog.getFirstResult());
				}
			}

		};
		if (Display.getCurrent() == null) {
			Display.getDefault().syncExec(r);
		} else {
			r.run();
		}
		if (selected.size() > 0) {
			return selected.get(0);
		}
		return null;
	}

	public static String getIResourceOSString(IResource f) {
		String fullPath = f.getRawLocation().toOSString();
		File file = new File(fullPath);
		if (file.exists()) {
			return getFileAbsolutePath(file);
		} else {
			IProject project = f.getProject();
			IPath location = project.getLocation();
			File projectFile = location.toFile();
			if (projectFile.exists()) {
				String projectFilePath = getFileAbsolutePath(projectFile);

				if (fullPath.startsWith(projectFilePath)) {
					return fullPath;
				} else {
					if (fullPath.toLowerCase().startsWith(projectFilePath.toLowerCase())) {
						String relativePart = fullPath.substring(projectFilePath.length());

						return projectFilePath + relativePart;
					}
				}
			}
		}
		return fullPath;
	}

	public static String getFileAbsolutePath(File f) {
		try {
			return f.getCanonicalPath();
		} catch (IOException e) {
			return f.getAbsolutePath();
		}
	}

}
