/*
 * Anthy is a Japanese KanaKanji conversion system library
 *
 * TODO
 *  reset_context
 *  deprecated clist widget
 */
#include <gtk/gtk.h>
#include <gtk/gtkimcontext.h>
#include <gtk/gtkimmodule.h>
#include <gtk/gtkimcontextsimple.h>
#include <gdk/gdkkeysyms.h>
#include <anthy/anthy.h>
#include <anthy/input.h>
#include <iconv.h>
#include <string.h>

/* functions */
/* class management */
static void im_anthy_class_init (GtkIMContextClass *class);
static void im_anthy_finalize(GObject *obj);
static void im_anthy_init (GtkIMContext *context);

/* exported module interface */
void im_module_list(const GtkIMContextInfo ***contexts,
		    int *n_contexts);
void im_module_init(GTypeModule *module);
void im_module_exit();
GtkIMContext * im_module_create (const gchar *context_id);


struct CandidateWin {
  GtkWidget *top_win;
  GtkWidget *clist;
  int visible;
  int layout_start;
};

/*
 * The Context
 */
#define MODE_DIRECT 0
#define MODE_HIRA 1
#define MODE_KATA 2

#define NR_CANDIDATES 12

typedef struct _IMAnthyContext
{
  struct _GtkIMContext parent;
  struct _GtkIMContext *slave;
  int mode; /* MODE_HIRA or MODE_DIRECT */
  int is_on;
  struct anthy_input_context *aic;
  struct anthy_input_preedit *pedit;
  GdkWindow *win;
  struct CandidateWin *cwin;
} IMAnthyContext;

typedef struct _IMContextAnthyClass 
{
  GtkIMContextClass parent_class;
} IMContextAnthyClass;


static GType type_im_anthy;
static GObjectClass *parent_class;

static const GtkIMContextInfo im_anthy_info = {
  "anthy",
  "Anthy(Japanese Conversion System)",
  "gtk+",
  "",
  "ja"
};

static const GtkIMContextInfo *info_list[] = {
  &im_anthy_info
};

static const GTypeInfo object_info =
  {
    sizeof(IMContextAnthyClass),
    (GBaseInitFunc) NULL,
    (GBaseFinalizeFunc) NULL,
    (GClassInitFunc) im_anthy_class_init,
    NULL,
    NULL,
    sizeof(IMAnthyContext),
    0,
    (GtkObjectInitFunc) im_anthy_init,
  };

#define IM_ANTHY_CONTEXT(obj) (GTK_CHECK_CAST((obj),type_im_anthy,IMAnthyContext))

/*
 * global variables for immodule
 */
static guint16 gtk_compose_ignore[] = {
  GDK_Shift_L,
  GDK_Shift_R,
  GDK_Control_L,
  GDK_Control_R,
  GDK_Caps_Lock,
  GDK_Shift_Lock,
  GDK_Meta_L,
  GDK_Meta_R,
  GDK_Alt_L,
  GDK_Alt_R,
  GDK_Super_L,
  GDK_Super_R,
  GDK_Hyper_L,
  GDK_Hyper_R,
  GDK_Mode_switch,
  0
};

static struct {
  GtkWidget *win;
  GtkWidget *menu;
} status_bar;

static IMAnthyContext *current_context;
static struct anthy_input_config *config;

/* input and event handling */
static void im_anthy_set_client_window(GtkIMContext *, GdkWindow *);
static void im_anthy_get_preedit_string(GtkIMContext *, gchar **str,
					PangoAttrList **attrs,
					gint *cursor_pos);
static gboolean im_anthy_filter_keypress(GtkIMContext *context,
					 GdkEventKey *key);
static void im_anthy_focus_in(GtkIMContext *);
static void im_anthy_focus_out(GtkIMContext *);
static void im_anthy_commit_cb(GtkIMContext *ic,
			       const gchar *str,
			       IMAnthyContext *ac);
static void im_anthy_reset(GtkIMContext *);


static int init_anthy_lib();
static int proc_anthy_keypress(struct anthy_input_context *, GdkEventKey *key);
static void update_anthy_preedit(IMAnthyContext *, struct anthy_input_context *);
static void do_commit(IMAnthyContext *, char *);
static char *get_preedit_string(struct anthy_input_preedit *pedit,
				PangoAttrList *attr);

static struct CandidateWin *create_candidate_win();
static void release_candidate_win(struct CandidateWin *win);
static void update_candidate_win(IMAnthyContext *);
static void layout_candidates(struct anthy_input_context *aic,
			      struct anthy_input_segment *seg,
			      struct CandidateWin *win,
			      GdkWindow *);
static char *euc_to_utf8(char *);
static void init_status_bar();
static void update_status_bar(IMAnthyContext *ac);
static void add_input_mode(GtkWidget *, char *, int);
static gint select_mode(GtkWidget *, gpointer);
static int is_on_key(GdkEventKey *key);
static int is_off_key(GdkEventKey *key);
static void im_anthy_off(IMAnthyContext *);
/* anthy */

static char *
euc_to_utf8(char *str)
{
  iconv_t conv;
  int len = strlen(str);
  int buflen = len * 6+3;
  char *realbuf = alloca(buflen);
  char *outbuf = realbuf;
  char *inbuf = str;
  bzero(realbuf, buflen);
  conv = iconv_open("UTF-8", "EUC-JP");
  if (conv == (iconv_t) -1) {
    return NULL;
  }
  iconv(conv,&inbuf,&len,&outbuf,&buflen);
  iconv_close(conv);
  return strdup(realbuf);
}

static gint
select_mode(GtkWidget *w, gpointer p)
{
  int m = (int) p;
  if (!current_context) {
    return 0;
  }
  if (m == MODE_DIRECT) {
    im_anthy_off(current_context);
  } else {
    current_context->is_on = 1;
    current_context->mode = m;
  }
  return 0;
}

static void
add_input_mode(GtkWidget *menu, char *name, int mode)
{
  GtkWidget *menu_item;
  menu_item = gtk_menu_item_new_with_label(name);
  gtk_menu_append(GTK_MENU(menu), menu_item);
  gtk_signal_connect(GTK_OBJECT(menu_item), "activate",
		     GTK_SIGNAL_FUNC(select_mode),(void *)mode);
  gtk_widget_show(menu_item);
}

static void
init_status_bar()
{
  GtkWidget *menu;
  static char hiragana[]={0xe3,0x81,0xb2,0xe3,0x82,0x89,
			  0xe3,0x81,0x8c,0xe3,0x81,0xaa,0};
  static char katakana[]={0xe3,0x81,0x8b,0xe3,0x81,0x9f,
			  0xe3,0x81,0x8b,0xe3,0x81,0xaa,0};
  static char direct[]={0xe7,0x9b,0xb4,0xe6,0x8e,0xa5,
			  0xe5,0x85,0xa5,0xe5,0x8a,0x9b,0};
  menu = gtk_menu_new();
  add_input_mode(menu, hiragana, MODE_HIRA);
  add_input_mode(menu, katakana, MODE_KATA);
  add_input_mode(menu, direct, MODE_DIRECT);
  status_bar.menu = gtk_option_menu_new();
  gtk_option_menu_set_menu(GTK_OPTION_MENU(status_bar.menu), menu);
  status_bar.win = gtk_window_new(GTK_WINDOW_POPUP);
  gtk_container_add(GTK_CONTAINER(status_bar.win),status_bar.menu);
  gtk_widget_show(status_bar.menu);
}

static void
update_status_bar(IMAnthyContext *ic)
{
  gint x,y,w,h,d;
  if (!status_bar.win) {
    init_status_bar();
  }
  if (!ic) {
    gtk_widget_hide(status_bar.win);
    return ;
  }
  if (ic->win) {
    gdk_window_get_geometry(ic->win, &x, &y,&w,&h,&d);
    gdk_window_get_origin(ic->win, &x, &y);
    gtk_window_move(GTK_WINDOW(status_bar.win), x+w , y);
    gtk_widget_show(status_bar.win);
  }
  if (!ic->is_on) {
    gtk_option_menu_set_history(GTK_OPTION_MENU(status_bar.menu), 2);
  } else if (ic->mode == MODE_KATA) {
    gtk_option_menu_set_history(GTK_OPTION_MENU(status_bar.menu), 1);
  } else {
    gtk_option_menu_set_history(GTK_OPTION_MENU(status_bar.menu), 0);
  }
}

static struct CandidateWin *
create_candidate_win()
{
  struct CandidateWin *win;
  gchar *titles[2] = {"index", "Candidates"};
  win = (struct CandidateWin *)malloc(sizeof(struct CandidateWin));
  win->top_win = gtk_window_new(GTK_WINDOW_POPUP);
  win->clist = gtk_clist_new_with_titles(2, titles);
  gtk_container_add(GTK_CONTAINER(win->top_win), win->clist);
  gtk_widget_show(win->clist);
  win->visible = 0;
  return win;
}

static void
release_candidate_win(struct CandidateWin *cwin)
{
  gtk_widget_destroy(cwin->clist);
  gtk_widget_destroy(cwin->top_win);
  free(cwin);
}

static void
layout_candidates(struct anthy_input_context *aic,
		  struct anthy_input_segment *seg,
		  struct CandidateWin *cwin,
		  GdkWindow *win)
{
  char idx[NR_CANDIDATES];
  char *buf[2];
  struct anthy_input_segment *cseg;
  int i, count;
  if (!cwin->visible) {
    cwin->layout_start = seg->cand_no;
  }
  while (cwin->layout_start + NR_CANDIDATES <= seg->cand_no) {
    cwin->layout_start += NR_CANDIDATES;
  }
  while (cwin->layout_start > seg->cand_no) {
    cwin->layout_start -= NR_CANDIDATES;
  }
  if (cwin->layout_start < 0) {
    cwin->layout_start = 0;
  }
  gtk_clist_clear(cwin->clist);
  count = 0;
  for (i = cwin->layout_start;
       i < seg->nr_cand && i < cwin->layout_start + NR_CANDIDATES; i++) {
    count ++;
    cseg = anthy_input_get_candidate(aic, i);
    sprintf(idx, "%d", cseg->cand_no);
    buf[0] = idx;
    buf[1] = euc_to_utf8(cseg->str);/* leak */
    gtk_clist_append(cwin->clist, buf);
    anthy_input_free_segment(cseg);
  }
  for (i = count; i < NR_CANDIDATES; i++) {
    buf[0] = "*";
    buf[1] = "";
    gtk_clist_append(cwin->clist, buf);
  }
  cseg = anthy_input_get_candidate(aic, seg->cand_no);
  anthy_input_free_segment(cseg);

  gtk_clist_select_row(cwin->clist, seg->cand_no - cwin->layout_start,0);
  
  if (!cwin->visible) {
    if (win) {
      gint x,y,w,h,d;
      gdk_window_get_geometry(win, &x, &y,&w,&h,&d);
      gdk_window_get_origin(win, &x, &y);
      gtk_window_move(GTK_WINDOW(cwin->top_win), x , y+h);
    }
    cwin->visible = 1;
    cwin->layout_start = seg->cand_no;
    gtk_widget_show(cwin->top_win);
  }
}


static void
update_candidate_win(IMAnthyContext *ac)
{
  struct anthy_input_segment *seg;
  struct anthy_input_segment *cseg = 0;
  for (seg = ac->pedit->segment; seg; seg = seg->next) {
    if (seg->flag & (SF_ENUM|SF_ENUM_REVERSE)) {
      cseg = seg;
    }
  }
  if (cseg) {
    if (!ac->cwin) {
      ac->cwin = create_candidate_win();
    }
    layout_candidates(ac->aic, cseg, ac->cwin, ac->win);
  } else if (ac->cwin && ac->cwin->visible) {
    gtk_widget_hide(ac->cwin->top_win);
    ac->cwin->visible = 0;
  }
}

static void
im_anthy_set_client_window(GtkIMContext *ic, GdkWindow *w)
{
  IMAnthyContext *ac = IM_ANTHY_CONTEXT(ic);
  ac->win = w;
}

static int
proc_anthy_keypress(struct anthy_input_context *aic, GdkEventKey *key)
{
  if (key->state) {
    if (key->state == GDK_CONTROL_MASK) {
      switch (key->keyval) {
      case GDK_a:
	anthy_input_beginning_of_line(aic);
	break;
      case GDK_b:
	anthy_input_move(aic, -1);
	break;
      case GDK_d:
	anthy_input_erase_next(aic);
	break;
      case GDK_e:
	anthy_input_end_of_line(aic);
	break;
      case GDK_f:
	anthy_input_move(aic, 1);
	break;
      case GDK_g:
	anthy_input_quit(aic);
	break;
      case GDK_h:
	anthy_input_erase_prev(aic);
	break;
      case GDK_i:
	anthy_input_resize(aic, -1);
	break;
      case GDK_j:
	anthy_input_commit(aic);
	break;
      case GDK_k:
	anthy_input_cut(aic);
	break;
      case GDK_n:
	anthy_input_next_candidate(aic);
	break;
      case GDK_o:
	anthy_input_resize(aic, 1);
	break;
      case GDK_p:
	anthy_input_prev_candidate(aic);
	break;
      }
    } else if (key->state == GDK_SHIFT_MASK) {
      switch (key->keyval) {
      case GDK_Right:
	anthy_input_resize(aic, 1);
	break;
      case GDK_Left:
	anthy_input_resize(aic, -1);
	break;
      default:
 	anthy_input_key(aic,key->keyval);
      }
    }
    return 0;
  }
  switch (key->keyval) {
  case GDK_space:
    anthy_input_space(aic);
    break;
  case GDK_Return:
    anthy_input_commit(aic);
    break;
  case GDK_Escape:
    anthy_input_quit(aic);
    break;
  case GDK_Right:
    anthy_input_move(aic, 1);
    break;
  case GDK_Left:
    anthy_input_move(aic, -1);
    break;
  case GDK_Down:
    anthy_input_next_candidate(aic);
    break;
  case GDK_Up:
    anthy_input_prev_candidate(aic);
    break;
  case GDK_BackSpace:
    anthy_input_erase_prev(aic);
    break;
  case GDK_Delete:
    break;
    anthy_input_erase_next(aic);
  default:
    anthy_input_key(aic, key->keyval);
  }
  return 1;
}

static char *
get_preedit_string(struct anthy_input_preedit *pedit,
		   PangoAttrList *attrs)
{
  struct anthy_input_segment *seg;
  int len = 0;
  char *origbuf = strdup("");
  char *realbuf;
  int idx = 0;

  for (seg = pedit->segment; seg; seg = seg->next) {
    char *cur;
    if (seg->str) {
      cur = seg->str;
    } else {
      cur = "|";
    }
    realbuf = euc_to_utf8(cur);
    origbuf = (char *)realloc(origbuf, strlen(origbuf) + strlen(realbuf)+1);
    strcat(origbuf,realbuf);
    if (attrs) {
      PangoAttribute *attr;
      int begin, end;
      begin = idx;
      end = idx + strlen(realbuf);
      idx = end;
      if (begin == end) {
	goto no_attr;
      }
      attr = pango_attr_underline_new(PANGO_UNDERLINE_SINGLE);
      attr->start_index = begin;
      attr->end_index = end;
      pango_attr_list_insert(attrs, attr);
      if (seg->flag & SF_CURSOR) {
	attr = pango_attr_foreground_new (0xffff, 0xffff, 0xffff);
	attr->start_index = begin;
	attr->end_index = end;

	pango_attr_list_change (attrs, attr);

	attr = pango_attr_background_new (0, 0, 0);
	attr->start_index = begin;
	attr->end_index = end;

	pango_attr_list_change (attrs, attr);	
      }
    }
  no_attr:
    free(realbuf);
  }

  return origbuf;
}

static void
im_anthy_commit_cb(GtkIMContext *ic,
		   const gchar *str,
		   IMAnthyContext *ac)
{
  g_signal_emit_by_name(ac, "commit", str);
}

static void
im_anthy_get_preedit_string(GtkIMContext *ic, gchar **str,
			    PangoAttrList **attrs,
			    gint *cursor_pos)
{
  IMAnthyContext *ac = IM_ANTHY_CONTEXT(ic);
  char *tmp1, **tmp;
  if (attrs) {
    *attrs = pango_attr_list_new();
  }
  if (cursor_pos) {
    *cursor_pos = 0;
  }
  if (!ac->pedit) {
    *str = euc_to_utf8("");
    return ;
  }
  if (str) {
    tmp = str;
  } else {
    tmp = &tmp1;
  }
  if (attrs && *attrs) {
    *tmp = get_preedit_string(ac->pedit, *attrs);
  } else {
    *tmp = get_preedit_string(ac->pedit, NULL);
  }
  if (!str) {
    free(*tmp);
  }

  if (cursor_pos) {
    *cursor_pos = 0;
  }
  update_candidate_win(ac);
}

static void
do_commit(IMAnthyContext *ic, char *str)
{
  char *realbuf;
  realbuf = euc_to_utf8(str);
  g_signal_emit_by_name(ic, "commit", realbuf);
  free(realbuf);
}

static void
update_anthy_preedit(IMAnthyContext *ic, struct anthy_input_context *ac)
{
  if (ic->pedit) {
    anthy_input_free_preedit(ic->pedit);
    ic->pedit = 0;
  }
  ic->pedit = anthy_input_get_preedit(ic->aic);
  g_signal_emit_by_name (ic, "preedit_changed");
  if (ic->pedit->commit) {
    do_commit(ic, ic->pedit->commit);
  }
}

static void
im_anthy_reset(GtkIMContext *c)
{
}

static void
im_anthy_focus_in(GtkIMContext *c)
{
  current_context = IM_ANTHY_CONTEXT(c);
  update_status_bar(current_context);
}

static void
im_anthy_focus_out(GtkIMContext *c)
{
  update_status_bar(0);
}

static void
im_anthy_off(IMAnthyContext *ac)
{
  ac->is_on = 0;
  update_status_bar(ac);
}

static int
is_on_key(GdkEventKey *key)
{
  if ((key->keyval == GDK_Escape || key->keyval == GDK_Zenkaku_Hankaku) && 
      key->state == GDK_MOD1_MASK) {
    return 1;
  }
  if (key->state == GDK_SHIFT_MASK && key->keyval == GDK_space) {
    return 1;
  }
  if (key->state == GDK_CONTROL_MASK && key->keyval == GDK_j) {
    return 1;
  }
  return 0;
}

static int
is_off_key(GdkEventKey *key)
{
  if ((key->keyval == GDK_Escape || key->keyval == GDK_Zenkaku_Hankaku) && 
      key->state == GDK_MOD1_MASK) {
    return 1;
  }
  if (key->state == GDK_SHIFT_MASK && key->keyval == GDK_space) {
    return 1;
  }
  if (key->keyval == GDK_l) {
    return 1;
  }
  return 0;
}

static gboolean
im_anthy_filter_keypress(GtkIMContext *context,
			 GdkEventKey *key)
{
  IMAnthyContext *ac = IM_ANTHY_CONTEXT(context);
  int i;
  if (key->type == GDK_KEY_RELEASE) {
    return FALSE;
  }
  for (i = 0; gtk_compose_ignore[i]; i++) {
    if(key->keyval == gtk_compose_ignore[i]) {
      return FALSE;
    }
  }
  if (!ac->is_on) {
    if (is_on_key(key)) {
      ac->is_on = 1;
      update_status_bar(ac);
      return FALSE;
    }
    return gtk_im_context_filter_keypress(ac->slave, key);
  }

  if (is_off_key(key)) {
    if (ac->aic->state < ST_EDIT ) {
      im_anthy_off(ac);
      return FALSE;
    } else {
      return FALSE;
    }
  }
  if (!ac->aic) {
    return FALSE;
  }
  if (ac->aic->state == ST_NONE &&
      (key->keyval > 128 || (key->state & GDK_CONTROL_MASK))) {
    return FALSE;
  }
  proc_anthy_keypress(ac->aic, key);
  update_anthy_preedit(ac, ac->aic);
  return TRUE;
}

/*
 * class init
 */
void
im_anthy_class_init (GtkIMContextClass *class)
{
  GtkIMContextClass *im_context_class =
    GTK_IM_CONTEXT_CLASS(class);
  GObjectClass *object_class = G_OBJECT_CLASS(class);

  parent_class = g_type_class_peek_parent (class);

  im_context_class->set_client_window = im_anthy_set_client_window;
  im_context_class->filter_keypress = im_anthy_filter_keypress;
  im_context_class->reset = im_anthy_reset;
  im_context_class->get_preedit_string = im_anthy_get_preedit_string;
  im_context_class->focus_in = im_anthy_focus_in;
  im_context_class->focus_out = im_anthy_focus_out;
  object_class->finalize = im_anthy_finalize;
}

/*
 * The Constructor
 */
void
im_anthy_init (GtkIMContext *context)
{
  IMAnthyContext *ac = IM_ANTHY_CONTEXT(context);
  ac->aic = anthy_input_create_context(config);
  ac->pedit = 0;
  ac->win = 0;
  ac->mode = MODE_HIRA;
  ac->is_on = 0;
  ac->cwin = 0;
  ac->slave = g_object_new(GTK_TYPE_IM_CONTEXT_SIMPLE, NULL);
  g_signal_connect(ac->slave, "commit",
		   G_CALLBACK(im_anthy_commit_cb), ac);
}

/*
 * The Destructor
 */
void
im_anthy_finalize(GObject *obj)
{
  IMAnthyContext *ac = IM_ANTHY_CONTEXT(obj);
  anthy_input_free_context(ac->aic);
  if (ac->pedit) {
    anthy_input_free_preedit(ac->pedit);
  }
  if (ac->cwin) {
    release_candidate_win(ac->cwin);
  }
  g_signal_handlers_disconnect_by_func(ac->slave,
				       im_anthy_commit_cb,
				       ac);
  g_object_unref(ac->slave);
  parent_class->finalize (obj);
}


static int
init_anthy_lib()
{
  if (anthy_input_init()) {
    return -1;
  }
  config = anthy_input_create_config();
  return 0;
}


GtkIMContext *
im_module_create (const gchar *context_id)
{
  if (strcmp (context_id, "anthy") == 0) {
    GtkObject *obj = g_object_new(type_im_anthy, NULL);
    return GTK_IM_CONTEXT (obj);
  }
  return NULL;
}

/*
 *
 */
void
im_module_list(const GtkIMContextInfo ***contexts,
	       int *n_contexts)
{
  *contexts = info_list;
  *n_contexts = G_N_ELEMENTS(info_list);
}

void
im_module_init(GTypeModule *type_module)
{
  int rval;
  rval = init_anthy_lib();
  if (rval == -1) {
    return ;
  }
  type_im_anthy =
    g_type_module_register_type(type_module,
				GTK_TYPE_IM_CONTEXT,
				"GtkIMContextAnthy",
				&object_info, 0);
}

void
im_module_exit()
{
}
