﻿using System;
using System.Text;
using System.Runtime.InteropServices;
using System.Runtime.InteropServices.ComTypes;
using System.Diagnostics;

namespace NaGet.InteropServices
{	
	/// <summary>
	/// ShellLinkの更新フラグ
	/// </summary>
	[Flags]
	public enum ShellLinkResolve : uint
	{
		// AnyMatch = 0x02, // winMe,win2k以降無効
		
		/// <summary>
		/// MSIを呼ぶ
		/// </summary>
		InvokeMSI = 0x80,
		/// <summary>
		/// 追跡禁止
		/// </summary>
		NoLinkInfo = 0x40,
		/// <summary>
		/// リンク先の解決ができないときダイアログを表示しない
		/// </summary>
		NoUi = 0x01,
		NoUiWithMsgPump = 0x101,
		/// <summary>
		/// リンク先のデータ更新を行わない
		/// </summary>
		NoUpdate = 0x07,
		/// <summary>
		/// 検索をしない
		/// </summary>
		NoSearch = 0x10,
		NoTrack = 0x20,
		/// <summary>
		/// リンク先を更新する
		/// </summary>
		Update = 0x04,
	}
	
	[ComImport()]
	[Guid("000214F9-0000-0000-C000-000000000046")]
	[InterfaceType(ComInterfaceType.InterfaceIsIUnknown)]
	public interface IShellLinkW
	{
		void GetPath([MarshalAs(UnmanagedType.LPWStr)] StringBuilder pszFile,
		             int cchMaxPath, IntPtr pfd, uint fFlags);
		
		void GetIDList(out IntPtr ppidl);
		void SetIDList(IntPtr pidl);
		
		void GetDescription([MarshalAs(UnmanagedType.LPWStr)] StringBuilder pszDesc,
		                         int cchMaxPath);
		void SetDescription(string pszDesc);
		
		void GetWorkingDirectory([MarshalAs(UnmanagedType.LPWStr)] StringBuilder pszDir,
		                         int cchMaxPath);
		void SetWorkingDirectory(string pszDir);
		
		void GetArguments([MarshalAs(UnmanagedType.LPWStr)] StringBuilder pszArgs,
		                         int cchMaxPath);
		void SetArguments(string pszArgs);
		
		void GetHotkey(out short pwHotkey);
		void SetHotkey(short pwHotkey);
		
		void GetShowCmd(out uint piShowCmd);
		void SetShowCmd(uint piShowCmd);
		
		void GetIconLocation([MarshalAs(UnmanagedType.LPWStr)] StringBuilder pszIconPath,
		                         int cchMaxPath, out int piIcon);
		void SetIconLocation(string pszIconPath, int iIcon);
		
		void SetRelativePath([MarshalAs(UnmanagedType.LPWStr)] StringBuilder pszPath,
		                         int cchMaxPath, uint dwReserved);
		
		void Resolve(IntPtr hWnd, ShellLinkResolve fFlag);
				
		void SetPath(string pszFile);
	}

	/// <summary>
	/// シェルリンク(ショートカット)のカプセルクラス
	/// </summary>
	public class ShellLink : IDisposable
	{
		/// <summary>
		/// シェルリンクのCOMオブジェクト
		/// </summary>
		protected IShellLinkW shellLink;
		
		/// <summary>
		/// シェルリンクのGUID
		/// </summary>
		public const string ShellLinkGuid = "00021401-0000-0000-C000-000000000046";
		
		protected const int MAX_PATH = 260;
		
		/// <summary>
		/// 新しいシェルリンクを作成する形のコンストラクタ
		/// </summary>
		public ShellLink()
		{
			Type shellLinkType = Type.GetTypeFromCLSID(new Guid(ShellLinkGuid));
			shellLink = (IShellLinkW) Activator.CreateInstance(shellLinkType);
		}
		
		/// <summary>
		/// 既存のシェルリンクを開くコンストラクタ
		/// </summary>
		/// <param name="path">既存のシェルリンクのパス</param>
		public ShellLink(string path) : this()
		{
			if (! System.IO.File.Exists(path)) {
				throw new System.IO.FileNotFoundException("File does not found", path);
			}
			
			ToPersistFile().Load(path, 0);
			Resolve(IntPtr.Zero, ShellLinkResolve.NoUpdate | ShellLinkResolve.NoUi);
		}
		
		/// <summary>
		/// シェルリンクの解析解決
		/// </summary>
		/// <param name="hWnd">親フレームのハンドル</param>
		/// <param name="fFlags">方法</param>
		public void Resolve(IntPtr? hWnd, ShellLinkResolve fFlags)
		{
			shellLink.Resolve(hWnd ?? IntPtr.Zero, fFlags);
		}
		
		/// <summary>
		/// IPersistFileとして取り出す
		/// </summary>
		/// <returns>IPersistFileにキャストされたCOMオブジェクト</returns>
		public IPersistFile ToPersistFile()
		{
			return (IPersistFile) shellLink;
		}
		
		/// <summary>
		/// リンク先パスを得る。コマンドラインを得るためにも
		/// </summary>
		/// <param name="fFlags">パスのタイプ(1:8.3形式; 2:UNCパス; 4:環境変数変換なし)</param>
		/// <returns>パス</returns>
		public string GetPath(uint fFlags)
		{
			StringBuilder sb = new StringBuilder(MAX_PATH);
			shellLink.GetPath(sb, sb.Capacity, IntPtr.Zero, fFlags);
			return sb.ToString();
		}
		
		/// <summary>
		/// リンク先パス(取得時には環境変数は展開されません)
		/// </summary>
		public string Path {
			get { return GetPath(0x04); }
			set { shellLink.SetPath(value); }
		}
		
		/// <summary>
		/// リンク先(引数)
		/// </summary>
		public string Arguments {
			get {
				StringBuilder sb = new StringBuilder(MAX_PATH);
				shellLink.GetArguments(sb, sb.Capacity);
				return sb.ToString();
			}
			set { shellLink.SetArguments(value); }
		}
		
		/// <summary>
		/// 作業フォルダ
		/// </summary>
		public string WorkingDirectory {
			get {
				StringBuilder sb = new StringBuilder(MAX_PATH);
				shellLink.GetWorkingDirectory(sb, sb.Capacity);
				return sb.ToString();
			}
			set { shellLink.SetWorkingDirectory(value); }
		}
		
		/// <summary>
		/// ウィンドウスタイル(ShowCmdのワッパ)
		/// </summary>
		public ProcessWindowStyle WindowStyle
		{
			get {
				uint showcmd;
				shellLink.GetShowCmd(out showcmd);
				switch (showcmd) {
					case 3:
						return ProcessWindowStyle.Maximized;
					case 7:
						return ProcessWindowStyle.Minimized;
					case 1:
					default:
						return ProcessWindowStyle.Normal;
				}
			}
			set {
				switch (value) {
					case ProcessWindowStyle.Normal:
						shellLink.SetShowCmd(1);
						break;
					case ProcessWindowStyle.Maximized:
						shellLink.SetShowCmd(3);
						break;
					case ProcessWindowStyle.Minimized:
					case ProcessWindowStyle.Hidden:
						shellLink.SetShowCmd(7);
						break;
				}
			}
		}
		
		/// <summary>
		/// アイコンの場所を得る
		/// </summary>
		/// <param name="iconPath">アイコンを含むファイルパス</param>
		/// <param name="iconIndex">アイコンのインデックス</param>
		public void GetIconLocation(out string iconPath, out int iconIndex)
		{
			StringBuilder sb = new StringBuilder(MAX_PATH);
			shellLink.GetIconLocation(sb, sb.Capacity, out iconIndex);
			iconPath = sb.ToString();
		}
		
		/// <summary>
		/// アイコンの場所を設定する
		/// </summary>
		/// <param name="iconPath">アイコンを含むファイルパス</param>
		/// <param name="iconIndex">アイコンのインデックス</param>
		public void SetIconLocation(string iconPath, int iconIndex)
		{
			shellLink.SetIconLocation(iconPath, iconIndex);
		}
		
		/// <summary>
		/// COMオブジェクトの開放
		/// </summary>
		public void Dispose()
		{
			if (shellLink != null) {
				Marshal.ReleaseComObject(shellLink);
				shellLink = null;
			}
		}

		/// <summary>
		/// シェルリンクの中身をプロセス情報として取得する。起動の際に利用
		/// </summary>
		/// <returns>プロセス起動情報化されたシェルリンクの情報</returns>
		public ProcessStartInfo ToProcessStartInfo()
		{
			ProcessStartInfo procInfo = new ProcessStartInfo();
			procInfo.FileName = GetPath(0);
			procInfo.Arguments = Arguments;
			procInfo.WorkingDirectory = WorkingDirectory;
			procInfo.WindowStyle = WindowStyle;
			
			return procInfo;
		}
		
		/// <summary>
		/// プロセス情報からシェルリンクオブジェクトを生成
		/// </summary>
		/// <param name="procInfo">プロセス情報</param>
		/// <returns>生成されたシェルリンクオブジェクト</returns>
		public static ShellLink CreateFromProcessStartInfo(ProcessStartInfo procInfo)
		{
			ShellLink shelllink = new ShellLink();
			shelllink.Path = procInfo.FileName;
			shelllink.Arguments = procInfo.Arguments;
			shelllink.WorkingDirectory = procInfo.WorkingDirectory;
			shelllink.WindowStyle = procInfo.WindowStyle;
			
			return shelllink;
		}
		
		/// <summary>
		/// ショートカット先のEXEファイルに対して適切な名前を生成する。
		/// 
		/// 具体的には、アセンブリの製品名をまず優先的に使い、
		/// それがなければ、exeファイルのファイル名(拡張子を除いたもの)を返す。
		/// 
		/// 拡張子はつかないので、lnkファイル名に使う場合は、手動で
		/// <code>".lnk"</code>を追加すること。
		/// </summary>
		/// <returns>拡張子を含まない、適切な名前</returns>
		public string GetSuitableShellLinkNameFor()
		{
			string exeFile = GetPath(0);
			
			try {
				FileVersionInfo vInfo = FileVersionInfo.GetVersionInfo(exeFile);
				if (vInfo.ProductName != null && vInfo.ProductName != string.Empty
				    && vInfo.ProductName.IndexOfAny(System.IO.Path.GetInvalidFileNameChars()) < 0) {
					// 原則、製品名を採用
					return vInfo.ProductName;
				}
			} catch (Exception) {}

			// そのほかの場合は、*.exeファイルの名前をそのまま使用
			return System.IO.Path.GetFileNameWithoutExtension(exeFile);
		}
	}
}
