﻿using System;
using System.IO;
using NaGet.Packages;
using NaGet.Packages.Install;
using NaGet.Net;
using NaGet.SubCommands;

namespace NaGet.SubCommands
{
	public class NaGetUpdate : NaGetTaskSet
	{
		private ProviderList providerList;
		
		private PackageListsManager pkgListMan;
		
		private bool downloadPackageLists = false;
		
		private bool packageListsDownloaded = true;
		
		
		/// <summary>
		/// ダウンロードに使うダウンローダオブジェクト
		/// </summary>
		public Downloader Downloader {
			get {
				if (downloader == null) {
					downloader = new Downloader();
				}
				return downloader;
			}
		}
		
		private Downloader downloader;
		
		private int currentTaskSetIndex = -1;
		
		private bool done = false;
		
		public override int CurrentTaskSetIndex {
			get { return currentTaskSetIndex; }
		}
		
		public override bool Cancelable {
			get { return ! packageListsDownloaded; }
		}
		
		public NaGetUpdate(PackageListsManager pkgListMan)
			: this(pkgListMan, true)
		{
		}
		
		public NaGetUpdate(PackageListsManager pkgMan, bool downloadPackageListsFlag)
		{
			pkgListMan = pkgMan;
			downloadPackageLists = downloadPackageListsFlag;
			
			System.Collections.Generic.List<string> taskSetNames = new System.Collections.Generic.List<string>();
			if (downloadPackageLists) {
				providerList = new ProviderList(NaGet.Env.ProviderListFile);
				
				foreach (string url in providerList.Urls) {
					taskSetNames.Add(string.Format("リスト取得: {0}", url));
				}
				taskSetNames.Add(string.Format("リスト更新: {0}", NaGet.Env.PackageListFile));
			}
			taskSetNames.Add(string.Format("リスト更新: {0}", NaGet.Env.ArchiveInstalledPackageListFile));
			taskSetNames.Add(string.Format("リスト更新: {0}", NaGet.Env.SystemInstalledPackageListFile));
			
			TaskSetNames = taskSetNames.ToArray();
		}

		public override void Run()
		{
			currentTaskSetIndex ++;
			RaiseTaskSetEvent(NaGetTaskSetEventType.STARTED, "リスト更新処理開始");
			
			try {
				// リストのダウンロード
				if (downloadPackageLists) {
					packageListsDownloaded = false;
					try {
						runDownloadPackages();
					} catch (NaGetTaskCanceledException) {
						RaiseTaskSetEvent(NaGetTaskSetEventType.WARNING, "リストのダウンロード処理がキャンセルされました");
					} catch (System.Net.WebException e) {
						RaiseTaskSetEvent(NaGetTaskSetEventType.WARNING, e.Message);
						if (System.Net.NetworkInformation.NetworkInterface.GetIsNetworkAvailable()) {
							RaiseTaskSetEvent(NaGetTaskSetEventType.WARNING, "ネットワークに接続されていません。");
						} else {
							RaiseTaskSetEvent(NaGetTaskSetEventType.WARNING, "ネットワークに接続できませんでした。ネットワークが切断されているか、ファイアウォールによって遮断された可能性があります。");
						}
					} finally {
						currentTaskSetIndex = providerList.Urls.Length + 1;
					}
					packageListsDownloaded = true;
				}
				
				runLocalUpdate();
			} finally {
				done = true;
			}
			
			RaiseTaskSetEvent(NaGetTaskSetEventType.COMPLETED, "終了", 100);
		}
		
		private void runDownloadPackages()
		{
			PackageList<Package> avaiablePackageList = new PackageList<Package>();
			foreach(string provider in providerList.Urls) {
				RaiseTaskSetEvent(NaGetTaskSetEventType.STARTED_TASKSET, TaskSetNames[currentTaskSetIndex]);
				
				string tmpfileName = Path.GetTempFileName();
				try {
					Downloader.Download(provider, tmpfileName);
					
					avaiablePackageList.AddPackages(NaGet.Utils.GetDeserializedObject<PackageList<Package>>(tmpfileName).Packages);
				} finally {
					if (File.Exists(tmpfileName)) {
						File.Delete(tmpfileName);
					}
				}
				
				currentTaskSetIndex ++;
				RaiseTaskSetEvent(NaGetTaskSetEventType.COMPLETED_TASKSET, TaskSetNames[currentTaskSetIndex-1]);
			}
			
			RaiseTaskSetEvent(NaGetTaskSetEventType.STARTED_TASKSET, TaskSetNames[currentTaskSetIndex]);
			pkgListMan.availablePkgList = avaiablePackageList; // Mediatorのリストを更新
			pkgListMan.SaveAvailablePackageList();
			currentTaskSetIndex ++;
			RaiseTaskSetEvent(NaGetTaskSetEventType.COMPLETED_TASKSET, TaskSetNames[currentTaskSetIndex-1]);
		}
		
		private void runLocalUpdate()
		{
			// インストールトリストの更新
			RaiseTaskSetEvent(NaGetTaskSetEventType.STARTED_TASKSET, TaskSetNames[currentTaskSetIndex]);
			pkgListMan.DetectInstalledPkgs();
			pkgListMan.SaveInstalledPackageList();
			currentTaskSetIndex++;
			RaiseTaskSetEvent(NaGetTaskSetEventType.COMPLETED_TASKSET, TaskSetNames[currentTaskSetIndex-1]);
		
			// システムにインストールされているリストの更新
			RaiseTaskSetEvent(NaGetTaskSetEventType.STARTED_TASKSET, TaskSetNames[currentTaskSetIndex]);
			pkgListMan.DetectSystemInstalledPkgs();
			pkgListMan.SaveSystemInstalledPackageList();
			currentTaskSetIndex++;
			RaiseTaskSetEvent(NaGetTaskSetEventType.COMPLETED_TASKSET, TaskSetNames[currentTaskSetIndex-1]);
		}
		
		public override bool Cancel()
		{
			return Downloader.Cancel();
		}

		public override bool Done {
			get { return done; }
		}
	}
}
