﻿using System;
using System.Windows.Forms;
using System.Collections.Generic;
using System.Text;
using System.Text.RegularExpressions;
using NaGet.Packages;

namespace AppliStation.PackageInfo
{
	/// <summary>
	/// パッケージ情報アクセスのための便利メソッド群
	/// </summary>
	public sealed class PackageDescripter
	{
		/// <summary>
		/// インストーラタイプに対応する文字列を生成して返す
		/// </summary>
		/// <param name="instType">インストーラタイプ</param>
		/// <returns>ラベル文字列</returns>
		public static string GetInstallerTypeLabel(InstallerType instType)
		{
			switch (instType) {
				case InstallerType.ARCHIVE:
					return "自己解凍書庫または書庫";
				case InstallerType.EXEC_INSTALLER:
					return "インストーラー形式";
				case InstallerType.MSI_PACKAGE:
					return "MSIファイル形式";
				case InstallerType.ITSELF:
					return "単体の実行ファイル";
				default:
					return "不明なインストール形式(インストール不能)";
			}
		}
		
		/// <summary>
		/// OSのラベル文字列を生成する
		/// </summary>
		/// <param name="platform">プラットフォーム</param>
		/// <returns>platformの示すOSバージョン文字列</returns>
		public static string GetOSValueLabel(Platform platform)
		{
			if (platform == null || string.IsNullOrEmpty(platform.Os)) {
				return "任意のWindows";
			} else {
				return platform.Os.Replace("WIN", string.Empty);
			}
		}
		
		/// <summary>
		/// アーキテクチャのラベル文字列を生成する
		/// </summary>
		/// <param name="platform">プラットフォーム</param>
		/// <returns>platformの示すアーキテクチャ文字列</returns>
		public static string GetArchValueLabel(Platform platform)
		{
			switch ((platform != null)? platform.Arch : System.Reflection.ProcessorArchitecture.X86) {
				case System.Reflection.ProcessorArchitecture.Amd64:
					return "x64 (64ビット環境)";
				case System.Reflection.ProcessorArchitecture.IA64:
					return "Itanium";
				case System.Reflection.ProcessorArchitecture.None:
				case System.Reflection.ProcessorArchitecture.MSIL:
					return "アーキテクチャ依存なし";
				case System.Reflection.ProcessorArchitecture.X86:
				default:
					return "x86 (32ビット環境)";
			}
		}
		
		/// <summary>
		/// ライセンスのラベル文字列とリンク情報を生成する
		/// </summary>
		/// <param name="pkg">パッケージ</param>
		/// <param name="text">ラベル文字列</param>
		/// <param name="links">リンク文字列</param>
		public static void GetLicenseValueLabel(Package pkg, out string text, out LinkLabel.Link[] links)
		{
			string license = pkg.License;
			string publicSiteUrl = (pkg.Url != null)? pkg.Url.Href : null;
			
			List<LinkLabel.Link> linkArray = new List<LinkLabel.Link>();
			
			switch (license ?? string.Empty) {
				case "":
					if (string.IsNullOrEmpty(publicSiteUrl)) {
						text = "フリーソフトウェア";
					} else {
						text = "フリーソフトウェア (詳細は公式サイトなどで確認してください)";
						linkArray.Add(createLinkToLinkLabel(text, "公式サイト", publicSiteUrl));
					}
					break;
				#region オープンソースかつコピーレフト
				case "GPL":
				case "GPLv2":
				case "GPLv2+":
				case "GPLv3":
				case "GPLv3+":
					text = string.Format("{0} (オープンソース)", license);
					if (license.StartsWith("GPLv3")) {
						linkArray.Add(new LinkLabel.Link(0, license.Length, @"http://www.gnu.org/licenses/gpl-3.0.html"));
					} else {
						linkArray.Add(new LinkLabel.Link(0, license.Length, @"http://www.gnu.org/licenses/gpl-2.0.html"));
						linkArray.Add(createLinkToLinkLabel(text, "オープンソース",
						                                @"http://creativecommons.org/licenses/GPL/2.0/deed.ja"));
					}
					break;
				case "LGPL":
				case "LGPLv2":
				case "LGPLv2+":
				case "LGPLv3":
				case "LGPLv3+":
					text = string.Format("{0} (オープンソース)", license);
					if (license.StartsWith("GPLv3")) {
						linkArray.Add(new LinkLabel.Link(0, license.Length, @"http://www.gnu.org/licenses/lgpl-3.0.html"));
					} else {
						linkArray.Add(new LinkLabel.Link(0, license.Length, @"http://www.gnu.org/licenses/lgpl-2.1.html"));
						linkArray.Add(createLinkToLinkLabel(text, "オープンソース",
						                                @"http://creativecommons.org/licenses/LGPL/2.1/deed.ja"));
					}
					break;
				case "AGPLv3":
				case "AGPLv3+":
					text = string.Format("{0} (オープンソース)", license);
					linkArray.Add(new LinkLabel.Link(0, license.Length, @"http://www.gnu.org/licenses/agpl-3.0.txt"));
					break;
				case "MPL":
				case "MPLv1":
				case "MPLv1+":
					text = "MPL (オープンソース)";
					linkArray.Add(new LinkLabel.Link(0, "MPL".Length, @"http://www.mozilla.org/MPL/MPL-1.1.html"));
					break;
				case "CPL":
					text = "CPL (オープンソース)";
					linkArray.Add(new LinkLabel.Link(0, "CPL".Length, @"http://opensource.org/licenses/cpl1.0.php"));
					break;
				case "Artistic":
					text = "Artisticライセンス (オープンソース)";
					linkArray.Add(new LinkLabel.Link(0, "Artisticライセンス".Length, @"http://www.perlfoundation.org/artistic_license_2_0"));
					break;
				#endregion
				#region オープンソースだが、コピーレフトでない著名なライセンス
				case "MIT":
					text = "MITライセンス (オープンソース)";
					linkArray.Add(new LinkLabel.Link(0, "MITライセンス".Length, @"http://www.opensource.org/licenses/mit-license.php"));
					break;
				case "BSD":
					text = "修正済BSDライセンス (オープンソース)";
					linkArray.Add(new LinkLabel.Link(0, "修正済BSDライセンス".Length, @"http://www.opensource.org/licenses/bsd-license.php"));
					break;
				case "Apache":
				case "ASLv2":
				case "ASLv2+":
					text = "Apacheライセンス (オープンソース)";
					linkArray.Add(new LinkLabel.Link(0, "Apacheライセンス".Length, @"http://www.apache.org/licenses/LICENSE-2.0.txt"));
					break;
				case "zlib":
					text = "zlibライセンス (オープンソース)";
					linkArray.Add(new LinkLabel.Link(0, "zlibライセンス".Length, @"http://sourceforge.jp/projects/opensource/wiki/licenses%2Fzlib_libpng_license"));
					break;
				#endregion
				#region 実質Public Domainなライセンス
				case "WTFPL":
				case "WTFPLv2":
					text = "WTFPL";
					linkArray.Add(new LinkLabel.Link(0, text.Length, @"http://sam.zoy.org/wtfpl/"));
					break;
				case "NYSL":
					text = "煮るなり焼くなり好きにしろライセンス";
					linkArray.Add(new LinkLabel.Link(0, text.Length, @"http://www.kmonos.net/nysl/NYSL_withfaq.TXT"));
					break;
				case "Public Domain":
					text = "パブリックドメイン (著作権帰属なし)";
					break;
				#endregion
				default:
					if (license.StartsWith("http://")) {
						text = license;
						linkArray.Add(new LinkLabel.Link(0, license.Length, license));
					} else {
						Regex regex = new Regex(@"^(?<msg>.*)<(?<url>https?://.*)>$", RegexOptions.IgnoreCase);
						Match match = regex.Match(license);
						
						if (match.Success) {
							text = match.Groups["msg"].Value;
							linkArray.Add(new LinkLabel.Link(0, text.Length, match.Groups["url"].Value));
						} else {
							text = license;
						}
					}
					break;
			}
			
			links = linkArray.ToArray();
		}
		
		/// <summary>
		/// ライセンスのラベル文字列を生成する
		/// </summary>
		/// <param name="pkg">パッケージ</param>
		/// <returns>ラベル文字列</returns>
		public static string GetLicenseValueLabel(Package pkg)
		{
			string label;
			LinkLabel.Link[] links;
			
			GetLicenseValueLabel(pkg, out label, out links);
			
			return label;
		}
		
		/// <summary>
		/// リンクラベルのリンク生成便利メソッド
		/// </summary>
		/// <param name="text">ラベルテキスト</param>
		/// <param name="word">リンク対象文字列</param>
		/// <param name="link">リンクオブジェクト</param>
		private static LinkLabel.Link createLinkToLinkLabel(string text, string word, string link)
		{
			int pos = text.IndexOf(word, 0);
			if (pos >= 0) {
				return new LinkLabel.Link(pos, word.Length, link);
			} else throw new ArgumentException(string.Format("label text \"{0}\" does not contain \"{1}\"!", text, word));
		}
	}
}
