package jp.sourceforge.armadillo;

import java.io.*;
import java.text.*;
import java.util.*;

/**
 * R}hB
 * R\[sꂽ̐U蕪sB
 */
public final class Command {

    /**
     * Command̐B
     */
    private Command() {
        // empty
    }

    /**
     * ̎sB
     * @param args p[^
     * @throws CommandException R}h삪s̏ꍇ
     * @throws IOException o̓G[ꍇ
     * @throws ParseException IvV̓G[
     */
    private static void execute(String[] args) throws IOException, ParseException {
        if (args.length < 2) {
            throw new CommandException("It needs at least 2 parameters");
        }
        OptionParser parser = new OptionParser(args);
        char mode = parser.parseMode();
        File archiveFile = parser.parseArchiveFile();
        boolean verbose = parser.parseVerbose();
        String outDirPath = parser.parseOutputDirectory();
        String[] targetNames = parser.getRest();
        if (mode != 'x' && outDirPath.length() > 0) {
            System.out.println("warn: directory option is unnecessary in this mode");
        }
        switch (mode) {
            case 'c':
                if (mode == 'c' && archiveFile.exists()) {
                    System.out.print("archive file '"
                                     + archiveFile.getName()
                                     + "' already exists, overwrite? (y/N)");
                    String answer = Utilities.getInput();
                    if (answer == null || !answer.equals("y")) {
                        if (verbose) {
                            System.out.println("cancelled.");
                        }
                        return;
                    }
                }
                archive(archiveFile, targetNames, verbose);
                break;
            case 't':
                list(archiveFile, targetNames, verbose);
                break;
            case 'x':
                if (outDirPath.length() == 0) {
                    expand(archiveFile, new File("."), targetNames, verbose);
                } else {
                    File outDir = new File(outDirPath);
                    if (outDir.isDirectory() && outDir.exists()) {
                        expand(archiveFile, outDir, targetNames, verbose);
                    } else {
                        throw new CommandException("directory '" + outDirPath + "' does not exist");
                    }
                }
                break;
            default:
        }
    }

    /**
     * A[JCut@C쐬B
     * @param archiveFile A[JCut@C
     * @param targetNames Ώۖ̃Xg
     * @param verbose verbose[h
     * @throws IOException o̓G[ꍇ
     */
    private static void archive(File archiveFile, String[] targetNames, boolean verbose) throws IOException {
        if (verbose) {
            System.out.println("armadillo archiving started.");
        }
        ArchiveType type = ArchiveType.getType(archiveFile);
        if (type == ArchiveType.UNKNOWN) {
            throw new CommandException("invalid file suffix: " + archiveFile.getName());
        }
        if (targetNames.length > 0) {
            FileArchiver archiver = new FileArchiver(archiveFile);
            try {
                for (int i = 0; i < targetNames.length; i++) {
                    archive(archiver, new File(targetNames[i]), verbose);
                }
            } finally {
                archiver.close();
            }
        } else {
            throw new CommandException("archive requires targets");
        }
        if (verbose) {
            System.out.println("armadillo archiving terminated.");
        }
    }

    /**
     * t@CA[JCuB
     * @param archiver Archiver
     * @param src t@C
     * @param verbose verbose[h
     * @throws IOException o̓G[ꍇ
     */
    private static void archive(FileArchiver archiver, File src, boolean verbose) throws IOException {
        String entryName = Utilities.getEntryName(src);
        ArchiveEntry entry = new ArchiveEntry(entryName);
        try {
            entry.lastModified = src.lastModified();
            if (src.isDirectory()) {
                if (entryName.equals(".")) {
                    return;
                }
                archiver.addEntry(entry, src);
                if (verbose) {
                    System.out.println("  " + entryName + " ... STORED");
                }
                File[] files = src.listFiles();
                for (int i = 0, n = files.length; i < n; i++) {
                    archive(archiver, files[i], verbose);
                }
            } else {
                if (verbose) {
                    System.out.print("  " + entryName + " ... ");
                }
                archiver.addEntry(entry, src);
                if (verbose) {
                    StringBuffer buffer = new StringBuffer();
                    long size = entry.size;
                    long compressedSize = entry.compressedSize;
                    assert compressedSize >= 0 && size >= 0;
                    if (size == 0 || compressedSize == size) {
                        buffer.append("STORED (");
                        buffer.append(size);
                    } else {
                        buffer.append("COMPRESSED (");
                        int rate = compressedSize == size
                                ? 100
                                : (int)(compressedSize * 100 / size);
                        buffer.append(rate + "%, " + size + " => " + compressedSize);
                    }
                    buffer.append(')');
                    System.out.println(buffer);
                }
            }
        } catch (IOException ex) {
            IOException exception = new IOException(ex.getMessage() + " : entry=" + entryName);
            exception.setStackTrace(ex.getStackTrace());
            exception.initCause(ex.getCause());
            throw exception;
        }
    }

    /**
     * A[JCut@CWJB
     * @param archiveFile A[JCut@C
     * @param outputDirectory o͐fBNg
     * @param targetNames Ώۖ̃Xg
     * @param verbose verbose[h̏ꍇ <code>true</code>
     * @throws IOException o̓G[ꍇ
     */
    private static void expand(File archiveFile,
                               File outputDirectory,
                               String[] targetNames,
                               boolean verbose) throws IOException {
        if (verbose) {
            System.out.println("armadillo expansion started.");
            System.out.println("  * output directory = [" + outputDirectory + ']');
        }
        boolean all;
        Set nameSet;
        if (targetNames != null && targetNames.length > 0) {
            all = false;
            nameSet = new HashSet(Arrays.asList(targetNames));
        } else {
            all = true;
            nameSet = Collections.EMPTY_SET;
        }
        FileExpander expander = new FileExpander(archiveFile);
        try {
            Set dirEntrySet = new HashSet();
            for (Iterator it = expander.iterator(); it.hasNext();) {
                ArchiveEntry entry = (ArchiveEntry)it.next();
                if (entry.isNull()) {
                    break;
                }
                String name = entry.name;
                if (!all && !nameSet.contains(name)) {
                    continue;
                }
                if (verbose) {
                    System.out.print("  " + name + " ... ");
                }
                File newfile = new File(outputDirectory, name);
                if (entry.isDirectory()) {
                    if (!newfile.exists()) {
                        newfile.mkdirs();
                    }
                    dirEntrySet.add(entry);
                    if (verbose) {
                        System.out.println("CREATED");
                    }
                    continue;
                }
                File parent = newfile.getParentFile();
                if (!parent.exists()) {
                    if (!parent.mkdirs()) {
                        throw new IOException("can't mkdir '" + parent + "'");
                    }
                }
                OutputStream fos = new FileOutputStream(newfile);
                try {
                    long size = expander.expand(fos);
                    if (verbose) {
                        System.out.println("EXTRACTED (" + size + ")");
                    }
                } finally {
                    fos.close();
                }
                newfile.setLastModified(entry.lastModified);
            }
            /*
             * fBNg̃^CX^v́ATuGgXV
             * ɍXVĂ܂̂ŁAŌɐݒ肷Kv
             */
            for (Iterator it = dirEntrySet.iterator(); it.hasNext();) {
                ArchiveEntry entry = (ArchiveEntry)it.next();
                File dir = new File(outputDirectory, entry.name);
                dir.setLastModified(entry.lastModified);
            }
        } finally {
            expander.close();
        }
        if (verbose) {
            System.out.println("armadillo expansion terminated.");
        }
    }

    /**
     * A[JCut@C̓e\B
     * @param archiveFile A[JCut@C
     * @param targetNames Ώۖ̃Xg
     * @param verbose verbose[h̏ꍇ <code>true</code>
     * @throws IOException o̓G[ꍇ
     */
    private static void list(File archiveFile, String[] targetNames, boolean verbose) throws IOException {
        if (verbose) {
            System.out.println("armadillo listing started.");
        }
        boolean all;
        Set nameSet;
        if (targetNames != null && targetNames.length > 0) {
            all = false;
            nameSet = new HashSet(Arrays.asList(targetNames));
        } else {
            all = true;
            nameSet = Collections.EMPTY_SET;
        }
        DateFormat df = new SimpleDateFormat("yyyy-MM-dd HH:mm:ss");
        NumberFormat nf = new DecimalFormat("###.0");
        FileExpander expander = new FileExpander(archiveFile);
        try {
            for (Iterator it = expander.iterator(); it.hasNext();) {
                ArchiveEntry entry = (ArchiveEntry)it.next();
                if (entry.isNull()) {
                    break;
                }
                String name = entry.name;
                if (!all && !nameSet.contains(name)) {
                    continue;
                }
                String line = "";
                if (entry.isDirectory()) {
                    line += 'D';
                    line += padSpace("", (verbose ? 24 : 8), false);
                } else {
                    long size = entry.size;
                    line += padSpace(String.valueOf(size), 8, true);
                    if (verbose) {
                        line += ' ';
                        long compressedSize = entry.compressedSize;
                        String compSizeString = (compressedSize > 0)
                                ? String.valueOf(compressedSize)
                                : "";
                        line += padSpace(compSizeString, 8, true);
                        line += ' ';
                        if (size > 0 && compressedSize > 0) {
                            float rate = (compressedSize / 1f / size) * 100;
                            line += padSpace(nf.format(rate), 5, true);
                            line += '%';
                        } else {
                            line += "------";
                        }
                    }
                    line += ' ';
                }
                line += padSpace(entry.method, 6, false).substring(0, 6);
                line += ' ';
                line += df.format(new Date(entry.lastModified));
                line += ' ';
                line += name;
                System.out.println(line);
            }
        } finally {
            expander.close();
        }
        if (verbose) {
            System.out.println("armadillo listing terminated.");
        }
    }

    /**
     * Xy[X𖄂߂B
     * 񂪎w肵蒷ꍇ͂̂܂ܕԂB
     * @param s 
     * @param length ߂Ƃ̒
     * @param forward <code>true</code>̏ꍇAXy[X𕶎̑Oɖ߂
     * @return Xy[X߂ꂽ
     */
    private static String padSpace(String s, int length, boolean forward) {
        if (s.length() > length) {
            return s;
        }
        StringBuffer buffer = new StringBuffer("        "); // 8
        while (buffer.length() < length) {
            buffer.append(buffer);
        }
        String space = buffer.substring(s.length(), length);
        return forward ? space + s : s + space;
    }

    /**
     * R}hG[B 
     */
    private static final class CommandException extends RuntimeException {

        /**
         * CommandException̐B
         * @param message
         */
        CommandException(String message) {
            super(message);
        }

    }

    /**
     * wv̕\B
     */
    private static void printHelp() {
        Properties p = new Properties();
        InputStream is = Command.class.getResourceAsStream("about.properties");
        try {
            try {
                p.load(is);
            } finally {
                is.close();
            }
        } catch (IOException ex) {
            throw new RuntimeException(ex);
        }
        String s = MessageFormat.format(p.getProperty("about"),
                                        new String[]{Version.getVersionString(false)});
        System.out.println(s);
    }

    /**
     * R}h̎sB
     * @param args p[^
     * @throws Exception Cӂ̗O
     */
    public static void main(String[] args) throws Exception {
        String caption;
        String message;
        try {
            if (args.length < 1 || args[0].equals("-h") || args[0].equals("--help")) {
                printHelp();
            } else {
                execute(args);
            }
            return; // normal end
        } catch (ParseException ex) {
            caption = "option error";
            message = ex.getMessage();
        } catch (CommandException ex) {
            caption = "command error";
            message = ex.getMessage();
        } catch (IOException ex) {
            caption = "I/O error";
            message = ex.getMessage();
            String propName = "jp.sourceforge.armadillo.debug";
            if (!System.getProperty(propName, "false").equalsIgnoreCase("false")) {
                System.err.println(caption);
                throw ex;
            }
        }
        System.err.println();
        String s = "armadillo: " + caption;
        if (message != null) {
            s += " - " + message;
        }
        System.err.println(s);
        System.exit(1);
    }

}
