package jp.sourceforge.armadillo;

import java.io.*;
import java.nio.channels.*;

import jp.sourceforge.armadillo.Utilities.*;
import jp.sourceforge.armadillo.io.*;
import jp.sourceforge.armadillo.lzh.*;

/**
 * LZHA[JCoB
 */
public final class LzhArchiver extends CharacterEncoding implements Archiver {

    private static final byte HEADER_LEVEL = LzhHeader.HEADER_LEVEL_2;

    private LzhOutputStream los;
    private RandomAccessFile rfile;

    /**
     * LzhArchiver̐B
     * @param os OutputStream
     */
    public LzhArchiver(OutputStream os) {
        this.los = new LzhOutputStream((os instanceof BufferedOutputStream)
                ? os
                : new BufferedOutputStream(os));
    }

    /**
     * LzhArchiver̐B
     * @param file o͐t@C
     * @throws IOException I/OG[
     */
    public LzhArchiver(File file) throws IOException {
        RandomAccessFile r = new RandomAccessFile(file, "rw");
        try {
            r.setLength(0L); // overwrite
            this.los = new LzhOutputStream(Channels.newOutputStream(r.getChannel()));
            this.rfile = r;
        } catch (Exception ex) {
            r.close();
            IOException e = new IOException();
            e.initCause(ex);
            throw e;
        }
    }

    /* @see jp.sourceforge.armadillo.Archiver#addEntry(jp.sourceforge.armadillo.ArchiveEntry, java.io.File) */
    public void addEntry(ArchiveEntry ae, File file) throws IOException {
        if (file.isDirectory()) {
            addDirectoryEntry(ae);
        } else {
            addFileEntry(ae, file);
        }
        ae.setAdded(true);
    }

    /* @see jp.sourceforge.armadillo.Archiver#addEntry(jp.sourceforge.armadillo.ArchiveEntry,
                                                       java.io.InputStream, long) */
    public void addEntry(ArchiveEntry ae, InputStream is, long length) throws IOException {
        if (ae.isDirectory()) {
            addDirectoryEntry(ae);
        } else {
            File file = IOUtilities.createTemporaryFile();
            if (length > 0 && is != null) {
                OutputStream os = new FileOutputStream(file);
                try {
                    IOUtilities.transfer(is, os, length);
                } finally {
                    os.close();
                }
            }
            addFileEntry(ae, file);
            file.delete();
        }
        ae.setAdded(true);
    }

    /**
     * fBNgGg̒ǉB
     * @param ae ArchiveEntry
     * @throws IOException o̓G[ꍇ
     */
    private void addDirectoryEntry(ArchiveEntry ae) throws IOException {
        LzhEntry entry = createEntry(ae);
        entry.setHeaderLevel(HEADER_LEVEL);
        entry.setMethod(LzhMethod.LHD);
        entry.setCrc((short)0);
        entry.setSize(0L);
        entry.setCompressedSize(0L);
        los.putNextEntry(entry);
        los.closeEntry();
        ae.crc = 0L;
        ae.size = 0L;
        ae.compressedSize = 0L;
    }

    /**
     * t@CGg̒ǉB
     * @param ae ArchiveEntry
     * @param file t@C
     * @throws IOException o̓G[ꍇ
     */
    private void addFileEntry(ArchiveEntry ae, File file) throws IOException {
        LzhEntry entry = createEntry(ae);
        entry.setHeaderLevel(HEADER_LEVEL);
        entry.setOsIdentifier('J');
        final long fileSize = file.length();
        if (fileSize > 0) {
            entry.setMethod(LzhMethod.LH5);
            entry.setSize(fileSize);
            RandomAccessFile r = new RandomAccessFile(file, "r");
            try {
                long size;
                if (rfile == null) {
                    FileChannel fc = r.getChannel();
                    try {
                        tryOut(entry, fc);
                        if (entry.getCompressedSize() >= fileSize) {
                            entry.setMethod(LzhMethod.LH0);
                        }
                    } catch (LzhQuit ex) {
                        entry.setMethod(LzhMethod.LH0);
                    }
                    los.putNextEntry(entry);
                    size = fc.transferTo(0L, fileSize, Channels.newChannel(los));
                    assert size == fileSize;
                    los.closeEntry();
                } else {
                    los.flush();
                    long first = rfile.getFilePointer();
                    assert entry.getCompressedSize() == 0L;
                    los.putNextEntry(entry);
                    long eoh = rfile.getFilePointer();
                    boolean quit;
                    try {
                        WritableByteChannel w = Channels.newChannel(los);
                        size = r.getChannel().transferTo(0, fileSize, w);
                        quit = false;
                        assert size == fileSize;
                    } catch (LzhQuit ex) {
                        size = 0;
                        quit = true;
                    }
                    los.closeEntry();
                    entry.setCrc(los.getCrc());
                    long last = rfile.getFilePointer();
                    long compressedSize = last - eoh;
                    rfile.seek(first);
                    if (compressedSize >= fileSize || quit) {
                        entry.setMethod(LzhMethod.LH0);
                        entry.setCompressedSize(fileSize);
                        FileChannel w = rfile.getChannel();
                        los.writeHeader(entry, Channels.newOutputStream(w));
                        size = r.getChannel().transferTo(0, fileSize, w);
                        rfile.setLength(rfile.getFilePointer());
                        assert size == fileSize;
                    } else {
                        entry.setCompressedSize(last - first);
                        los.writeHeader(entry, Channels.newOutputStream(rfile.getChannel()));
                        rfile.seek(last);
                    }
                }
                assert size == fileSize;
                ae.crc = entry.getCrc();
                ae.size = entry.getSize();
                ae.compressedSize = entry.getCompressedSize();
            } finally {
                r.close();
            }
        } else {
            los.putNextEntry(entry);
            los.closeEntry();
            ae.crc = 0L;
            ae.size = 0L;
            ae.compressedSize = 0L;
        }
    }

    /**
     * ksB
     * @param entry LzhEntry
     * @param fch InputStream
     * @throws IOException o̓G[ꍇ
     */
    private static void tryOut(LzhEntry entry, FileChannel fch) throws IOException {
        VolumetricOutputStream volumetric = new VolumetricOutputStream();
        LzhOutputStream los = new LzhOutputStream(volumetric);
        try {
            los.putNextEntry(entry);
            fch.transferTo(0L, Long.MAX_VALUE, Channels.newChannel(los));
            los.closeEntry();
        } finally {
            los.close();
        }
        entry.setCrc(los.getCrc());
        entry.setCompressedSize(volumetric.getSize() - 1);
    }

    /**
     * Gg̐B
     * @param ae ArchiveEntry
     * @return LzhEntry
     */
    private LzhEntry createEntry(ArchiveEntry ae) {
        String name = ae.name;
        LzhEntry entry = new LzhEntry(name);
        if (usesEncoding) {
            entry.setNameAsBytes(encode(name));
        }
        long fileLastModified = ae.lastModified;
        entry.setFTime(FTime.convert(fileLastModified));
        entry.setTimeT(TimeT.convert(fileLastModified));
        entry.setFileTime(FILETIME.convert(fileLastModified));
        return entry;
    }

    /* @see jp.sourceforge.armadillo.Archiver#close() */
    public void close() throws IOException {
        try {
            los.close();
        } finally {
            if (rfile != null) {
                rfile.close();
            }
        }
    }

}
