package jp.sourceforge.armadillo.zip;

import java.io.*;
import java.util.zip.*;

import jp.sourceforge.armadillo.io.*;

/**
 * ZIP`A[JCȕo̓Xg[B
 */
public final class ZipOutputStream extends ArchiveOutputStream {

    private ZipHeader header;
    private Deflater deflater;
    private CRC32 crc;
    private ZipEntry nextEntry;

    /**
     * ZipOutputStream̐B
     * ftHgZbggpB
     * @param os OutputStream
     */
    public ZipOutputStream(OutputStream os) {
        super(os);
        this.header = new ZipHeader();
        this.deflater = new Deflater(Deflater.DEFAULT_COMPRESSION, true);
        this.crc = new CRC32();
        frontStream = os;
    }

    /**
     * GgǉB
     * wb_͑ɏ܂B
     * @param entry ZipEntry
     * @throws IOException Gg݂̏Ɏsꍇ
     */
    public void putNextEntry(ZipEntry entry) throws IOException {
        ensureOpen();
        if (nextEntry != null) {
            closeEntry();
        }
        if (entry.method == -1) {
            entry.method = ZipEntry.DEFLATED;
        }
        switch (entry.method) {
            case ZipEntry.DEFLATED:
                entry.version = 20;
                if (entry.crc != -1 && entry.compressedSize != -1 && entry.size != -1) {
                    entry.option = 0;
                } else if (entry.crc == -1 && entry.compressedSize == -1 && entry.size == -1) {
                    entry.option = 8;
                } else {
                    throw new ZipException("invalid ZipEntry (crc, compressed-size or size)");
                }
                frontStream = new DeflaterOutputStream(out, deflater);
                break;
            case ZipEntry.STORED:
                long size = (entry.size >= 0) ? entry.size : entry.compressedSize;
                if (size < 0 || entry.crc < 0) {
                    throw new ZipException("invalid ZipEntry (crc, compressed-size or size)");
                }
                entry.version = 10;
                entry.compressedSize = size;
                entry.size = size;
                break;
            default:
                throw new ZipException("unsupported compression method: " + entry.method);
        }
        nextEntry = entry;
        header.write(out, entry);
    }

    /**
     * ݂̃GgB
     * @throws IOException o̓G[ꍇ
     */
    public void closeEntry() throws IOException {
        ensureOpen();
        flush();
        if (frontStream instanceof DeflaterOutputStream) {
            DeflaterOutputStream deflaterOutputStream = (DeflaterOutputStream)frontStream;
            deflaterOutputStream.finish();
            deflaterOutputStream.flush();
            frontStream = out;
        }
        if (nextEntry.hasEXT()) {
            nextEntry.crc = crc.getValue();
            nextEntry.compressedSize = deflater.getTotalOut();
            nextEntry.size = deflater.getTotalIn();
            header.writeEXT(out, nextEntry);
        } else if (nextEntry.method == ZipEntry.DEFLATED) {
            if (crc.getValue() != nextEntry.crc
                || deflater.getTotalOut() != nextEntry.compressedSize
                || deflater.getTotalIn() != nextEntry.size) {
                throw new ZipException("invalid header info");
            }
        } else if (nextEntry.size > 0) {
            if (crc.getValue() != nextEntry.crc) {
                throw new ZipException("invalid header info");
            }
        }
        deflater.reset();
        crc.reset();
        nextEntry = null;
    }

    /* @see jp.sourceforge.armadillo.io.ArchiveOutputStream#write(int) */
    public void write(int b) throws IOException {
        super.write(b);
        crc.update(b);
    }

    /* @see jp.sourceforge.armadillo.io.ArchiveOutputStream#write(byte[], int, int) */
    public void write(byte[] b, int off, int len) throws IOException {
        super.write(b, off, len);
        crc.update(b, off, len);
    }

    /* @see jp.sourceforge.armadillo.io.ArchiveOutputStream#close() */
    public void close() throws IOException {
        try {
            flush();
            header.writeEND(out);
            deflater.end();
        } finally {
            header = null;
            deflater = null;
            crc = null;
            nextEntry = null;
            super.close();
        }
    }

}
