package jp.sfjp.armadillo;

import java.io.*;
import java.lang.reflect.*;
import java.util.*;
import java.util.zip.*;
import jp.sfjp.armadillo.archive.*;
import jp.sfjp.armadillo.archive.cab.*;
import jp.sfjp.armadillo.archive.lzh.*;
import jp.sfjp.armadillo.archive.tar.*;
import jp.sfjp.armadillo.archive.zip.ZipFile;
import jp.sfjp.armadillo.file.*;
import jp.sfjp.armadillo.io.*;

public final class ArmadilloCommands {

    private static Logger log = Logger.getLogger(ArmadilloCommands.class);

    private ArmadilloCommands() { // empty
    }

    public static String getVersionString() {
        return getVersionString(false);
    }

    public static String getVersionString(boolean throwsError) {
        try {
            InputStream is = ArmadilloCommands.class.getResourceAsStream("version");
            if (is != null)
                try {
                    final Scanner r = new Scanner(is);
                    if (r.hasNextLine())
                        return r.nextLine();
                }
                finally {
                    is.close();
                }
        }
        catch (Exception ex) {
            if (throwsError)
                throw new IllegalStateException(ex);
            // ignore
        }
        return "";
    }

    public static void createArchive(File srcdir, File dst0) throws IOException {
        createArchive(srcdir, dst0, ProgressNotifier.NullObject);
    }

    public static void createArchive(File srcdir, File dst0, ProgressNotifier notifier) throws IOException {
        File dst = getAlternativeFileIfExists(dst0);
        ArchiveType type = ArchiveType.of(dst.getName());
        List<File> files = getAllChildrenFile(srcdir);
        ArchiveCreator ac = getArchiveCreator(dst, type);
        try {
            for (final File file : files) {
                final String pathForEntryName = getRelativePath(srcdir, file);
                if (pathForEntryName.length() == 0)
                    continue;
                ArchiveEntry entry = ac.newEntry(pathForEntryName);
                final long fileSize = file.length();
                entry.setSize(fileSize);
                entry.setLastModified(file.lastModified());
                ac.addEntry(entry, file);
                if (fileSize > 0)
                    notifier.notifyProgress(fileSize);
            }
        }
        finally {
            ac.close();
        }
    }

    public static void extractAll(File src) throws IOException {
        extractAll(src, ProgressNotifier.NullObject);
    }

    public static void extractAll(File src, File dstdir) throws IOException {
        extractAll(src, dstdir, ProgressNotifier.NullObject);
    }

    public static void extractAll(File src, ProgressNotifier notifier) throws IOException {
        extractAll(src, getAlternativeFileIfExists(getDestinationDirectory(src)), notifier);
    }

    public static void extractAll(File src, File dstdir, ProgressNotifier notifier) throws IOException {
        Set<String> targetSet = Collections.emptySet();
        extract(src, dstdir, targetSet, notifier);
    }

    public static void extract(File src, Set<String> targetSet, ProgressNotifier notifier) throws IOException {
        extract(src, getAlternativeFileIfExists(getDestinationDirectory(src)), targetSet, notifier);
    }

    public static void extract(File src,
                               File dstdir,
                               Set<String> targetSet,
                               ProgressNotifier notifier) throws IOException {
        if (dstdir.exists())
            throw new IOException("the directory already exists: " + dstdir);
        final boolean all = targetSet.isEmpty();
        int targetCount = targetSet.size();
        ArchiveType type = ArchiveType.of(src.getName());
        Set<ArchiveEntry> dirEntrySet = new HashSet<ArchiveEntry>();
        ArchiveExtractor ax = getArchiveExtractor(src, type);
        try {
            assert !dstdir.exists();
            if (!dstdir.mkdirs())
                throw new IllegalStateException("failed to mkdirs");
            while (true) {
                ArchiveEntry entry = ax.nextEntry();
                if (entry == ArchiveEntry.NULL)
                    break;
                if (entry.isDirectory())
                    dirEntrySet.add(entry);
                if (!all && !targetSet.contains(entry.getName()))
                    continue;
                if (!all)
                    --targetCount;
                final File newfile = getRelativeFile(dstdir, entry);
                if (entry.isDirectory()) {
                    if (!newfile.exists())
                        newfile.mkdirs();
                    continue;
                }
                if (newfile.exists())
                    throw new IOException("file '" + newfile + "' already exists");
                final File parent = newfile.getParentFile();
                if (!parent.exists())
                    if (!parent.mkdirs())
                        throw new IOException("can't mkdir '" + parent + "'");
                final long inc = (entry.getCompressedSize() > 0)
                        ? entry.getCompressedSize()
                        : (long)(entry.getSize() * 0.18f);
                final long inc1 = (long)(inc * 0.3f);
                final long inc2 = inc - inc1;
                assert inc1 + inc2 == inc;
                notifier.notifyProgress(inc1);
                final long writtenSize;
                if (newfile.exists()) // second check
                    throw new IOException("file '" + newfile + "' already exists");
                OutputStream fos = new FileOutputStream(newfile);
                try {
                    writtenSize = ax.extract(fos);
                }
                catch (Exception ex) {
                    throw new IllegalStateException("illegal state", ex);
                }
                finally {
                    fos.close();
                }
                assert writtenSize == entry.getSize() : String.format("%s: written=%d, entry=%d",
                                                                      entry.getName(),
                                                                      writtenSize,
                                                                      entry.getSize());
                newfile.setLastModified(entry.getLastModified());
                notifier.notifyProgress(inc2);
                if (!all && targetCount <= 0)
                    break;
            }
            notifier.notifyFinished();
            /*
             * Finally, set timestamp on all directories.
             */
            for (final ArchiveEntry entry : dirEntrySet) {
                File dir = new File(dstdir, entry.getName());
                if (dir.exists())
                    dir.setLastModified(entry.getLastModified());
            }
        }
        finally {
            ax.close();
        }
    }

    public static boolean validate(File src) throws IOException {
        ArchiveType type = ArchiveType.of(src.getName());
        ArchiveExtractor ax = getArchiveExtractor(src, type);
        try {
            while (true) {
                ArchiveEntry entry = ax.nextEntry();
                if (entry == ArchiveEntry.NULL)
                    break;
                assert entry != null : "return null entry: extractor="
                                       + ax.getClass().getSimpleName();
                if (entry.isDirectory())
                    continue;
                VolumetricOutputStream vos = new VolumetricOutputStream();
                try {
                    if (ax.extract(vos) != entry.getSize())
                        return false;
                }
                finally {
                    vos.close();
                }
            }
        }
        catch (IOException ex) {
            throw ex;
        }
        catch (Exception ex) {
            log.error(ex);
            return false;
        }
        finally {
            ax.close();
        }
        return true;
    }

    public static List<ArchiveEntry> extractArchiveEntries(File src, ArchiveType type) throws IOException {
        List<ArchiveEntry> a = new ArrayList<ArchiveEntry>();
        ArchiveExtractor ax = getArchiveExtractor(src, type);
        try {
            while (true) {
                ArchiveEntry entry = ax.nextEntry();
                if (entry == ArchiveEntry.NULL)
                    break;
                a.add(entry);
            }
        }
        finally {
            ax.close();
        }
        return a;
    }

    public static long transferAll(InputStream is, OutputStream os) throws IOException {
        long totalSize = 0;
        byte[] buffer = new byte[8192];
        for (int readLength; (readLength = is.read(buffer)) >= 0;) {
            assert readLength != 0 : "Read Zero";
            os.write(buffer, 0, readLength);
            totalSize += readLength;
        }
        os.flush();
        return totalSize;
    }

    public static String getRelativePath(File dir, File f) throws IOException {
        final String filePath = f.getCanonicalFile().getAbsolutePath();
        final String dirPath = dir.getCanonicalFile().getAbsolutePath();
        if (dirPath.equals(filePath))
            return "";
        if (!filePath.startsWith(dirPath))
            throw new IllegalStateException(String.format("not relative: dir=%s, f=%s", dir, f));
        final int dirLen = dirPath.length() + (dirPath.endsWith("/") ? 0 : 1);
        final String s = filePath.substring(dirLen).replace('\\', '/');
        if (f.isDirectory() && !filePath.endsWith("/"))
            return s + "/";
        else
            return s;
    }

    public static File getRelativeFile(File dir, ArchiveEntry entry) {
        final String name = entry.getName();
        if (name.startsWith("/"))
            return new File(dir, name);
        else if (name.matches("(?i)[A-Z]\\:[\\\\/].*"))
            return new File(dir, name.substring(2));
        return new File(dir, name);
    }

    public static List<File> getAllChildrenFile(File dir) {
        if (!dir.isDirectory())
            throw new IllegalArgumentException("file is not dir: " + dir);
        List<File> a = new ArrayList<File>();
        getAllChildrenFile(a, dir);
        return a;
    }

    private static void getAllChildrenFile(List<File> a, File f) {
        a.add(f);
        if (f.isDirectory()) {
            File[] children = f.listFiles();
            if (children != null)
                for (File child : children)
                    getAllChildrenFile(a, child);
        }
    }

    public static long getTotalSize(File dir) {
        TotalFileSizeFindAction action = new TotalFileSizeFindAction();
        Find.find(dir, action);
        return action.totalSize;
    }

    static final class TotalFileSizeFindAction implements FindAction {

        long totalSize;

        @Override
        public void act(File file) {
            totalSize += file.length();
        }

    }

    public static File getDestinationDirectory(File f) {
        return getAlternativeFileIfExists(new File(f.getPath() + ".tmp"));
    }

    public static File getAlternativeFileIfExists(File f) {
        if (!f.exists())
            return f;
        File parent = f.getParentFile();
        final String name = f.getName();
        final String mainName;
        final String ext;
        if (f.isDirectory()) {
            mainName = name;
            ext = "";
        }
        else {
            final int index = name.indexOf('.');
            if (index == 0)
                throw new IllegalArgumentException("dot file not supported at getAlternativeFileIfExists");
            if (index > 0) {
                mainName = name.substring(0, index);
                ext = name.substring(index + 1);
            }
            else {
                mainName = name;
                ext = "";
            }
        }
        for (int i = 1; i <= 1000; i++) {
            final String newName = String.format("%s(%d).%s", mainName, i, ext);
            File newFile = new File(parent, newName);
            if (!newFile.exists())
                return newFile;
        }
        throw new RuntimeException("over 100 times to try to resolve alternative file.");
    }

    public static ArchiveCreator getArchiveCreator(File dst, ArchiveType type) throws IOException {
        switch (type) {
            case TAR:
                return new TarArchiveCreator(new BufferedOutputStream(new FileOutputStream(dst)));
            case TARGZ:
                return new TarArchiveCreator(new GZIPOutputStream(new BufferedOutputStream(new FileOutputStream(dst))));
            case TARZ:
            case TARBZ2:
            case TARXZ:
                return new TarArchiveCreator(getOutputStream(dst, type));
            case ZIP:
                return new ZipFile(dst, true);
            case LZH:
                return new LzhFile(dst, true);
            case CAB:
                return new CabArchiveCreator(new BufferedOutputStream(new FileOutputStream(dst)));
            default:
        }
        throw new IllegalStateException("File: " + dst + ", ArchiverType: " + type);
    }

    private static OutputStream getOutputStream(File file, ArchiveType type) throws IOException {
        try {
            final String cname = System.getProperty("plugin.stream.o." + type, "");
            if (cname.length() == 0)
                throw new UnsupportedOperationException("File: " + file + ", ArchiverType: " + type);
            final Class<?> c = Class.forName(cname);
            Constructor<?> ctor = c.getConstructor(new Class<?>[]{InputStream.class});
            return (OutputStream)ctor.newInstance(new BufferedOutputStream(new FileOutputStream(file)));
        }
        catch (IOException ex) {
            throw ex;
        }
        catch (Exception ex) {
            throw new UnsupportedOperationException("File: " + file + ", ArchiverType: " + type, ex);
        }
    }

    public static ArchiveExtractor getArchiveExtractor(File src, ArchiveType type) throws IOException {
        switch (type) {
            case TAR:
                return new TarArchiveExtractor(new BufferedInputStream(new FileInputStream(src)));
            case TARGZ:
                return new TarArchiveExtractor(new GZIPInputStream(new BufferedInputStream(new FileInputStream(src))));
            case TARZ:
            case TARBZ2:
            case TARXZ:
                return new TarArchiveExtractor(new BufferedInputStream(getInputStream(src, type)));
            case ZIP:
                return new ZipFile(src, true);
            case LZH:
                return new LzhFile(src, true);
            case CAB:
                return new CabArchiveExtractor(new BufferedInputStream(new FileInputStream(src)));
            default:
        }
        throw new IllegalStateException("File: " + src + ", ArchiverType: " + type);
    }

    private static InputStream getInputStream(File file, ArchiveType type) throws IOException {
        try {
            final String cname = System.getProperty("plugin.stream.i." + type, "");
            if (cname.length() == 0)
                throw new UnsupportedOperationException("File: " + file + ", ArchiverType: " + type);
            final Class<?> c = Class.forName(cname);
            Constructor<?> ctor = c.getConstructor(new Class<?>[]{InputStream.class});
            return (InputStream)ctor.newInstance(new FileInputStream(file));
        }
        catch (IOException ex) {
            throw ex;
        }
        catch (Exception ex) {
            throw new UnsupportedOperationException("File: " + file + ", ArchiverType: " + type, ex);
        }
    }

}
