package jp.sfjp.armadillo.archive;

import java.io.*;
import java.nio.charset.*;
import java.util.*;

public abstract class ArchiveEntry implements ArchiveEntryInterface {

    public static final ArchiveEntry NULL = new NullArchiveEntry();

    private boolean added;
    private String name;
    private byte[] nameb;
    private boolean initialized;

    protected ArchiveEntry(boolean initialized) {
        this.initialized = initialized;
    }

    public static ArchiveEntry orNull(ArchiveEntry entry) {
        return (entry == null) ? NULL : entry;
    }

    public ArchiveEntry copyFrom(ArchiveEntry an) {
        copyAtoB(an, this);
        return an;
    }

    public ArchiveEntry copyTo(ArchiveEntry an) {
        copyAtoB(this, an);
        return an;
    }

    private static void copyAtoB(ArchiveEntry a, ArchiveEntry b) {
        b.name = a.name;
        b.nameb = a.nameb;
        b.initialized = a.initialized;
        b.setSize(a.getSize());
        b.setCompressedSize(a.getCompressedSize());
        b.setLastModified(a.getLastModified());
    }

    public boolean isNull() {
        return this == NULL;
    }

    public boolean isAdded() {
        return added;
    }

    @Override
    public void setAdded(boolean added) {
        this.added = added;
    }

    public final String name() {
        if (name == null)
            if (nameb == null)
                throw new IllegalStateException("not initialized: " + getClass());
            else
                name = new String(nameb);
        return name;
    }

    @Override
    public final String getName() {
        return name();
    }

    public final byte[] getNameAsBytes() {
        if (nameb == null)
            if (name == null)
                throw new IllegalStateException("not initialized: " + getClass());
            else
                nameb = name.getBytes();
        return Arrays.copyOf(nameb, nameb.length);
    }

    public final byte[] getNameAsBytes(Charset charset) {
        if (nameb == null)
            nameb = name.getBytes(charset);
        return Arrays.copyOf(nameb, nameb.length);
    }

    public final void setName(String name) {
        this.name = name;
        this.nameb = null;
        this.initialized = true;
    }

    public final void setName(String name, Charset charset) {
        this.name = name;
        this.nameb = name.getBytes(charset);
        this.initialized = true;
    }

    public final void setName(byte[] nameb) {
        this.name = null;
        this.nameb = Arrays.copyOf(nameb, nameb.length);
        this.initialized = true;
    }

    public final boolean equalsName(ArchiveEntry an) {
        return Arrays.equals(getNameAsBytes(), an.getNameAsBytes());
    }

    public final void setFileInfo(File file) {
        assert isDirectory() == file.isDirectory();
        setSize(file.length());
        setLastModified(file.lastModified());
    }

    @Override
    public boolean isDirectory() {
        if (initialized)
            return getName().endsWith("/");
        return false;
    }

    @Override
    public long getSize() {
        throw new UnsupportedOperationException("ArchiveEntry#getSize");
    }

    @Override
    public void setSize(long size) {
        throw new UnsupportedOperationException("ArchiveEntry#setSize");
    }

    @Override
    public long getCompressedSize() {
        throw new UnsupportedOperationException("ArchiveEntry#getCompressedSize");
    }

    @Override
    public void setCompressedSize(long size) {
        throw new UnsupportedOperationException("ArchiveEntry#setCompressedSize");
    }

    @Override
    public long getLastModified() {
        throw new UnsupportedOperationException("ArchiveEntry#getLastModified");
    }

    @Override
    public void setLastModified(long time) {
        throw new UnsupportedOperationException("ArchiveEntry#setLastModified");
    }

    public String getMethodName() {
        return "";
    }

    @Override
    public String toString() {
        return String.format("%s:%s", getClass().getSimpleName(), getName());
    }

    private static final class NullArchiveEntry extends ArchiveEntry {

        public NullArchiveEntry() {
            super(false);
        }

    }

}
