package jp.sfjp.armadillo.archive;

import java.io.*;
import jp.sfjp.armadillo.archive.cab.*;
import jp.sfjp.armadillo.archive.lzh.*;
import jp.sfjp.armadillo.archive.tar.*;
import jp.sfjp.armadillo.archive.zip.*;

public abstract class DumpArchiveHeader {

    public abstract void dump(InputStream is, PrintWriter out) throws IOException;

    protected void printOffset(PrintWriter out, long offset) {
        out.printf("%n---------- offset %1$d ( 0x%1$08X )%n", offset);
    }

    protected void printEnd(PrintWriter out, String name, long offset) {
        out.printf("%n---------- end of %1$s : position = %2$d ( 0x%2$08X )%n", name, offset);
    }

    protected void printHeaderName(PrintWriter out, String name) {
        out.printf("%n%s%n", name);
    }

    protected static void warn(PrintWriter out, String fmt, Object... args) {
        out.printf("warn: " + fmt + "%n", args);
    }

    public static DumpArchiveHeader getInstance(String filename) {
        switch (ArchiveType.of(filename)) {
            case CAB:
                return new DumpCabHeader();
            case LZH:
                return new DumpLzhHeader();
            case TAR:
                return new DumpTarHeader();
            case ZIP:
                return new DumpZipHeader();
            default:
                throw new IllegalArgumentException("unknown type: " + filename);
        }
    }

    public static void execute(File f) throws IOException {
        execute(f, new PrintWriter(System.out, true));
    }

    public static void execute(File f, PrintWriter out) throws IOException {
        FileInputStream is = new FileInputStream(f);
        try {
            getInstance(f.getName()).dump(is, out);
        }
        finally {
            is.close();
        }
    }

    public static void main(String... args) throws IOException {
        if (args.length != 1) {
            System.out.println("usage: DumpArchiveHeader file [ file2 ... fileN ]");
            return;
        }
        for (final String arg : args) {
            final File f = new File(arg);
            if (!f.exists()) {
                System.out.printf("warn: file [%s] not found %n", arg);
                continue;
            }
            try {
                execute(f);
            }
            catch (Exception ex) {
                System.err.printf("%n--- at file [%s]%n", f.getAbsolutePath());
                ex.printStackTrace();
            }
        }
    }

}
