package jp.sfjp.armadillo.archive.cab;

import java.util.zip.*;

/**
 * Checksum for Cab file.
 * @see <a href="http://msdn.microsoft.com/en-us/library/bb267310.aspx">
 *      Microsoft Cabinet Format (http://msdn.microsoft.com/en-us/library/bb267310.aspx)</a>
 */
public final class CabChecksum implements Checksum {

    private final int seed;

    private int value;
    private int p;
    private int tmp;

    public CabChecksum() {
        this(0);
    }

    public CabChecksum(int seed) {
        this.seed = seed;
        reset();
    }

    @Override
    public void update(int b) {
        switch (++p % 4) {
            case 0:
                tmp = b;
                break;
            case 1:
                tmp |= b << 8;
                break;
            case 2:
                tmp |= b << 16;
                break;
            case 3:
                tmp |= b << 24;
                value ^= tmp;
                tmp = 0;
                break;
            default:
        }
    }

    @Override
    public void update(byte[] b, int off, int len) {
        for (int i = off; i < len; i++)
            update(b[i] & 0xFF);
    }

    @Override
    public long getValue() {
        final int tmpValue;
        final int mod = (p + 1) % 4;
        if (mod == 0)
            tmpValue = value;
        else {
            int x = 0;
            switch (mod) {
                case 3:
                    x |= (tmp << 16) & 0xFF0000;
                    x |= tmp & 0x00FF00;
                    x |= (tmp >> 16) & 0x0000FF;
                    assert x == (x & 0xFFFFFF);
                    break;
                case 2:
                    x |= (tmp << 8) & 0xFF00;
                    x |= (tmp >> 8) & 0x00FF;
                    assert x == (x & 0xFFFF);
                    break;
                case 1:
                    x = tmp;
                    assert x == (x & 0xFF);
                    break;
                default:
                    break;
            }
            tmpValue = value ^ x;
        }
        assert tmpValue == (tmpValue & 0xFFFFFFFF);
        return tmpValue;
    }

    @Override
    public void reset() {
        value = seed;
        p = -1;
    }

}
