package jp.sfjp.armadillo.archive.cab;

import java.io.*;
import java.nio.*;
import java.nio.channels.*;
import java.util.*;
import java.util.zip.*;

/**
 * Cab Header.
 * @see <a href="http://msdn.microsoft.com/en-us/library/bb267310.aspx">
 *      Microsoft Cabinet Format (http://msdn.microsoft.com/en-us/library/bb267310.aspx)</a>
 */
public final class CabHeader {

    // signature 'MSCF'
    static final int SIGNATURE = 0x4D534346;

    private static final byte MAJOR_VERSION = 1;
    private static final byte MINOR_VERSION = 3;

    private final ByteBuffer buffer;
    // on read
    private boolean initialized;
    private final List<CabCfFile> fileEntriesOnRead;
    private int entryIndex;

    private short[] folderCompressType;

    public CabHeader() {
        this.buffer = ByteBuffer.allocate(8192);
        this.initialized = false;
        this.fileEntriesOnRead = new ArrayList<CabCfFile>();
    }

    public boolean initialize(InputStream is) throws IOException {
        if (initialized)
            return false;
        try {
            readFileHeader(is);
            return true;
        }
        catch (ArrayIndexOutOfBoundsException ex) {
            throw new CabException("readFileHeader", ex);
        }
        catch (BufferUnderflowException ex) {
            throw new CabException("readFileHeader", ex);
        }
    }

    @SuppressWarnings("unused")
    private void readFileHeader(InputStream is) throws IOException {
        // Cabinet File header
        final int signature; // cabinet file signature
        final int reserved1; // reserved
        final int cbCabinet; // size of this cabinet file in bytes
        final int reserved2; // reserved
        final int coffFiles; // offset of the first CFFILE entry
        final int reserved3; // reserved
        final byte versionMinor; // cabinet file format version, minor
        final byte versionMajor; // cabinet file format version, major
        final short cFolders; // number of CFFOLDER entries in this cabinet
        final short cFiles; // number of CFFILE entries in this cabinet
        final short flags; // cabinet file option indicators
        final short setID; // must be the same for all cabinets in a set
        final short iCabinet; // number of this cabinet file in a set
        // (optional fields not supported)
        // ---
        buffer.clear();
        buffer.limit(36);
        Channels.newChannel(is).read(buffer);
        if (buffer.position() == 0)
            throw new CabException("reading archive header, but position is zero");
        buffer.rewind();
        buffer.order(ByteOrder.BIG_ENDIAN);
        signature = buffer.getInt();
        if (signature != SIGNATURE)
            throw new CabException("bad signature: 0x%X", signature);
        buffer.order(ByteOrder.LITTLE_ENDIAN);
        reserved1 = buffer.getInt();
        cbCabinet = buffer.getInt();
        reserved2 = buffer.getInt();
        coffFiles = buffer.getInt();
        reserved3 = buffer.getInt();
        versionMinor = buffer.get();
        versionMajor = buffer.get();
        cFolders = buffer.getShort();
        cFiles = buffer.getShort();
        flags = buffer.getShort();
        setID = buffer.getShort();
        iCabinet = buffer.getShort();
        if (iCabinet != 0)
            throw new CabException("iCabinet not supported: 0x%d", iCabinet);
        folderCompressType = new short[cFolders];
        // read CFFOLDERs
        for (int i = 0; i < cFolders; i++) {
            // CFFOLDER header
            final int coffCabStart; // offset of the first CFDATA block in this folder
            final short cCFData; // number of CFDATA blocks in this folder
            final short typeCompress; // compression type indicator
            final byte[] abReserve; // (optional) per-folder reserved area
            // ---
            buffer.clear();
            buffer.limit(8);
            Channels.newChannel(is).read(buffer);
            if (buffer.position() == 0)
                throw new CabException("reading CFFOLDER header, but position is zero");
            buffer.rewind();
            coffCabStart = buffer.getInt();
            cCFData = buffer.getShort();
            typeCompress = buffer.getShort();
            abReserve = new byte[0];
            folderCompressType[i] = typeCompress;
        }
        // read CFFILEs
        for (int i = 0; i < cFiles; i++) {
            // CFFILE header
            final int cbFile; // uncompressed size of this file in bytes
            final int uoffFolderStart; // uncompressed offset of this file in the folder
            final short iFolder; // index into the CFFOLDER area
            final short date; // date stamp for this file
            final short time; // time stamp for this file
            final short attribs; // attribute flags for this file
            final byte[] szName; // name of this file
            // ---
            buffer.clear();
            buffer.limit(16);
            Channels.newChannel(is).read(buffer);
            if (buffer.position() == 0)
                throw new CabException("reading CFFILE header, but position is zero");
            buffer.rewind();
            cbFile = buffer.getInt();
            uoffFolderStart = buffer.getInt();
            iFolder = buffer.getShort();
            date = buffer.getShort();
            time = buffer.getShort();
            attribs = buffer.getShort();
            szName = readName(is, 256);
            CabCfFile entry = new CabCfFile(szName);
            entry.uncompressedSize = cbFile;
            entry.uncompressedOffset = uoffFolderStart;
            entry.folderIndex = iFolder;
            entry.date = date;
            entry.time = time;
            entry.attributes = attribs;
            entry.setName(szName);
            fileEntriesOnRead.add(entry);
        }
        initialized = true;
    }

    private byte[] readName(InputStream is, int limit) throws IOException {
        byte[] bytes = new byte[limit];
        int readSize = 0;
        for (int i = 0; i < limit; i++) {
            int b = is.read();
            if (b <= 0)
                break;
            bytes[i] = (byte)(b & 0xFF);
            ++readSize;
        }
        if (readSize < limit) {
            byte[] r = new byte[readSize];
            System.arraycopy(bytes, 0, r, 0, readSize);
            return r;
        }
        else
            return bytes;
    }

    public CabEntry read(InputStream is) throws IOException {
        if (!initialized)
            initialize(is);
        if (entryIndex >= fileEntriesOnRead.size())
            return null;
        return fileEntriesOnRead.get(entryIndex++);
    }

    @SuppressWarnings("unused")
    public void write(OutputStream os, List<CabCfFolder> folders) throws IOException {
        List<CabCfFile> files = new ArrayList<CabCfFile>();
        for (CabCfFolder folder : folders)
            files.addAll(folder.files);
        // Cabinet File header
        final int signature; // cabinet file signature
        final int reserved1; // reserved
        final int cbCabinet; // size of this cabinet file in bytes
        final int reserved2; // reserved
        final int coffFiles; // offset of the first CFFILE entry
        final int reserved3; // reserved
        final byte versionMinor; // cabinet file format version, minor
        final byte versionMajor; // cabinet file format version, major
        final short cFolders; // number of CFFOLDER entries in this cabinet
        final short cFiles; // number of CFFILE entries in this cabinet
        final short flags; // cabinet file option indicators
        final short setID; // must be the same for all cabinets in a set
        final short iCabinet;// number of this cabinet file in a set
        // (optional fields not supported)
        // ---
        assert folders.size() <= Short.MAX_VALUE;
        assert files.size() <= Short.MAX_VALUE;
        signature = SIGNATURE;
        reserved1 = 0;
        cbCabinet = calculateFileSize(folders, files);
        reserved2 = 0;
        reserved3 = 0;
        coffFiles = calculateOffsetOfFirstCffile(folders);
        versionMinor = MINOR_VERSION;
        versionMajor = MAJOR_VERSION;
        cFolders = (short)folders.size();
        cFiles = (short)files.size();
        flags = 0;
        setID = 0; // ignore
        iCabinet = 0;
        buffer.clear();
        // Header
        buffer.order(ByteOrder.BIG_ENDIAN);
        buffer.putInt(SIGNATURE);
        buffer.order(ByteOrder.LITTLE_ENDIAN);
        buffer.putInt(reserved1);
        buffer.putInt(cbCabinet);
        buffer.putInt(reserved2);
        buffer.putInt(coffFiles);
        buffer.putInt(reserved3);
        buffer.put(versionMinor);
        buffer.put(versionMajor);
        buffer.putShort(cFolders);
        buffer.putShort(cFiles);
        buffer.putShort(flags);
        buffer.putShort(setID);
        buffer.putShort(iCabinet);
        buffer.flip();
        Channels.newChannel(os).write(buffer);
        // Folders
        int folderOffset = calculateOffsetOfCabStart(folders);
        for (CabCfFolder folder : folders) {
            // CFFOLDER header
            final int coffCabStart; // offset of the first CFDATA block in this folder
            final short cCFData; // number of CFDATA blocks in this folder
            final short typeCompress; // compression type indicator
            final byte[] abReserve; // (optional) per-folder reserved area
            // ---
            assert folder.arrayOfCfData.size() <= Short.MAX_VALUE;
            coffCabStart = folderOffset;
            cCFData = (short)folder.arrayOfCfData.size();
            typeCompress = folder.method;
            buffer.clear();
            buffer.putInt(coffCabStart);
            buffer.putShort(cCFData);
            buffer.putShort(typeCompress);
            buffer.flip();
            Channels.newChannel(os).write(buffer);
            folderOffset += folder.getCompressedDataSize();
        }
        // Files
        for (CabCfFile entry : files) {
            // CFFOLDER header
            final int cbFile; // uncompressed size of this file in bytes
            final int uoffFolderStart; // uncompressed offset of this file in the folder
            final short iFolder; // index into the CFFOLDER area
            final short date; // date stamp for this file
            final short time; // time stamp for this file
            final short attribs; // attribute flags for this file
            final byte[] szName; // name of this file
            // ---
            cbFile = entry.uncompressedSize;
            uoffFolderStart = entry.uncompressedOffset;
            iFolder = entry.folderIndex;
            date = entry.date;
            time = entry.time;
            attribs = entry.attributes;
            szName = entry.getNameAsBytes();
            buffer.clear();
            buffer.putInt(cbFile);
            buffer.putInt(uoffFolderStart);
            buffer.putShort(iFolder);
            buffer.putShort(date);
            buffer.putShort(time);
            buffer.putShort(attribs);
            buffer.put(szName);
            buffer.put((byte)0); // end of szName
            buffer.flip();
            Channels.newChannel(os).write(buffer);
        }
    }

    @SuppressWarnings("unused")
    public void writeCfDataHeader(OutputStream os, CabCfData cfdata) throws IOException {
        // CFDATA header
        final int csum; // checksum of this CFDATA entry
        final short cbData; // number of compressed bytes in this block
        final short cbUncomp; // number of uncompressed bytes in this block
        final byte[] abReserve; // (optional) per-datablock reserved area
        // ---
        if (!cfdata.finished)
            throw new IOException("stream is not closed yet");
        assert cfdata.uncompsize >= 0 && cfdata.uncompsize <= 32768;
        byte[] bytes = cfdata.bos.toByteArray();
        final short cbData0 = (short)(bytes.length & 0xFFFF);
        final short cbUncomp0 = (cfdata.uncompsize == 32768)
                ? Short.MIN_VALUE
                : (short)cfdata.uncompsize;
        buffer.order(ByteOrder.LITTLE_ENDIAN);
        buffer.clear();
        csum = calcucateChecksum(bytes, cbData0, cbUncomp0);
        cbData = cbData0;
        cbUncomp = cbUncomp0;
        buffer.putInt(csum);
        buffer.putShort(cbData);
        buffer.putShort(cbUncomp);
        buffer.flip();
        Channels.newChannel(os).write(buffer);
    }

    private int calcucateChecksum(byte[] bytes, short compsize, short uncompsize) {
        final int csum1 = calcucateChecksum0(bytes, compsize, 0);
        ByteBuffer buffer = ByteBuffer.allocate(4).order(ByteOrder.LITTLE_ENDIAN);
        buffer.putShort(compsize);
        buffer.putShort(uncompsize);
        buffer.rewind();
        byte[] bytes2 = new byte[4];
        buffer.get(bytes2);
        return calcucateChecksum0(bytes2, 4, csum1);
    }

    private static int calcucateChecksum0(byte[] bytes, int cb, int seed) {
        Checksum cksum = new CabChecksum(seed);
        cksum.update(bytes, 0, cb);
        return (int)(cksum.getValue() & 0xFFFFFFFF);
    }

    static int calculateFileSize(List<CabCfFolder> folders, List<CabCfFile> files) {
        int x = 0;
        // HEADER
        x += 36;
        // CFFOLDER
        x += folders.size() * 8;
        // CFFILE
        for (final CabCfFile f : files)
            x += 16 + f.getName().getBytes().length + 1;
        // CFDATA
        for (CabCfFolder folder : folders)
            x += folder.getCompressedDataSize() + 8;
        return x;
    }

    static int calculateOffsetOfFirstCffile(List<CabCfFolder> folders) {
        final int sizeOfCfheader = 36;
        return sizeOfCfheader + folders.size() * 8;
    }

    static int calculateOffsetOfCabStart(List<CabCfFolder> folders) {
        int size = 0;
        size += calculateOffsetOfFirstCffile(folders);
        for (CabCfFolder folder : folders)
            for (CabEntry entry : folder.files)
                size += 16 + entry.getName().getBytes().length + 1;
        return size;
    }

}
