package jp.sfjp.armadillo.archive.cab;

import java.io.*;
import java.nio.*;
import java.nio.channels.*;
import java.util.zip.*;
import jp.sfjp.armadillo.archive.*;
import jp.sfjp.armadillo.io.*;

public final class CabInputStream extends ArchiveInputStream {

    private final CabHeader header;

    private int index = 0;

    public CabInputStream(InputStream is) {
        super(is);
        this.header = new CabHeader();
    }

    public CabInputStream(InputStream is, boolean readHeaderFirst) throws IOException {
        this(is);
        if (readHeaderFirst)
            header.initialize(is);
    }

    public CabEntry getNextEntry() throws IOException {
        ensureOpen();
        CabEntry entry = header.read(in);
        if (entry == null)
            return null;
        final boolean first = index == 0;
        ++index;
        remaining = entry.getSize();
        if (first) {
            final short imethod = 1;
            final CabCompressionType type = CabCompressionType.of(imethod);
            switch (type) {
                case No:
                    frontStream = in;
                    break;
                case MSZIP:
                    frontStream = new CfDataInflateInputStream(in);
                    break;
                default:
                    throw new CabException("Unsupported compression type: %s(0x%X)", type, imethod);
            }
        }
        return entry;
    }

    static final class CfDataInflateInputStream extends FilterInputStream {

        private static final int BLOCK_SIZE = 32768;

        private final Inflater inflater;
        private byte[] inflaterBuffer;

        CfDataInflateInputStream(InputStream is) {
            this(is, new Inflater(true));
        }

        CfDataInflateInputStream(InputStream is, Inflater inflater) {
            super(is);
            this.inflater = inflater;
        }

        @Override
        public int read() throws IOException {
            byte[] bytes = new byte[1];
            if (read(bytes, 0, 1) < 1)
                return -1;
            return bytes[0];
        }

        @Override
        public int read(byte[] b, int off, int len) throws IOException {
            if (inflaterBuffer == null)
                refill();
            int p = off;
            int q = len;
            while (q > 0) {
                final int r;
                try {
                    r = inflater.inflate(b, p, q);
                }
                catch (DataFormatException ex) {
                    throw new CabException("data format error", ex);
                }
                if (r < 0)
                    throw new CabException("illegal state");
                if (r == 0) {
                    if (inflater.needsDictionary())
                        throw new CabException("change dictionary not supported");
                    if (inflater.finished())
                        if (inflater.getTotalOut() == BLOCK_SIZE)
                            refill();
                        else
                            break;
                    if (inflater.needsInput())
                        refill();
                }
                p += r;
                q -= r;
            }
            assert q >= 0;
            return len - q;
        }

        void refill() throws IOException {
            byte[] tmp = new byte[BLOCK_SIZE];
            final int r = readNextBlock(tmp, 0);
            if (r < 1)
                throw new CabException("illegal state");
            final int remaining = inflater.getRemaining();
            byte[] newBuffer = new byte[remaining + r];
            if (remaining > 0) {
                assert inflaterBuffer != null;
                final int offset = inflaterBuffer.length - remaining;
                System.arraycopy(inflaterBuffer, offset, newBuffer, 0, remaining);
            }
            System.arraycopy(tmp, 0, newBuffer, remaining, r);
            inflater.reset();
            inflater.setInput(newBuffer);
            inflaterBuffer = newBuffer;
        }

        @SuppressWarnings("unused")
        int readNextBlock(byte[] bytes, int offset) throws IOException {
            // CFDATA header
            final int csum; // checksum of this CFDATA entry
            final short cbData; // number of compressed bytes in this block
            final short cbUncomp; // number of uncompressed bytes in this block
            final byte[] abReserve; // (optional) per-datablock reserved area
            // ---
            ByteBuffer buffer = ByteBuffer.allocate(8).order(ByteOrder.LITTLE_ENDIAN);
            buffer.clear();
            Channels.newChannel(in).read(buffer);
            if (buffer.position() == 0)
                return -1; // EOF?
            if (buffer.position() != 8)
                throw new CabException("reading CFDATA header error: %d", buffer.position());
            buffer.rewind();
            csum = buffer.getInt();
            cbData = buffer.getShort();
            cbUncomp = buffer.getShort();
            byte[] tmp = new byte[cbData];
            final int r = IOUtilities.read(in, tmp, 0, cbData);
            if (r != cbData)
                throw new CabException("failed to inflate (fill)");
            // skip CK
            if (tmp[0] != 'C' || tmp[1] != 'K')
                throw new CabException("bad CFDATA block (%02X%02X)", tmp[0], tmp[1]);
            System.arraycopy(tmp, 2, bytes, offset, r - 2);
            return r - 2;
        }

    }

}
