package jp.sfjp.armadillo.archive.cab;

import java.io.*;
import java.util.*;
import jp.sfjp.armadillo.archive.*;

public final class CabOutputStream extends ArchiveOutputStream {

    private final CabHeader header;
    private final List<CabCfFolder> folders;
    private CabCfFolder currentFolder;

    public CabOutputStream(OutputStream out) {
        super(out);
        this.header = new CabHeader();
        this.folders = new ArrayList<CabCfFolder>();
        this.currentFolder = new CabCfFolder("*");
    }

    public void putNextEntry(CabEntry entry) throws IOException {
        ensureOpen();
        final String name = entry.getName();
        if (name.length() > 255)
            throw new IllegalArgumentException("too long name: " + name);
        if (entry.isDirectory()) {
            CabCfFolder folder = (CabCfFolder)entry;
            if (folders.isEmpty() || folder.method != currentFolder.method)
                currentFolder.method = folder.method;
        }
        else {
            if (folders.isEmpty())
                changeFolder(getParentPath(entry));
            CabCfFile file = (CabCfFile)entry;
            final int folderCount = folders.size() - 1;
            assert folderCount <= Short.MAX_VALUE;
            file.folderIndex = (short)folderCount;
            file.uncompressedOffset = currentFolder.offset;
            currentFolder.offset += file.uncompressedSize;
            currentFolder.add(file);
        }
    }

    void changeFolder(String name) throws IOException {
        currentFolder.close();
        CabCfFolder folder = new CabCfFolder(normalizePath(name, true));
        folder.method = 1;
        folders.add(folder);
        currentFolder = folder;
        frontStream = currentFolder.bos;
    }

    void closeFolder(CabCfFolder folder) throws IOException {
        ensureOpen();
        flush();
        folder.close();
        frontStream = out;
    }

    String getParentPath(CabEntry entry) {
        final String path = normalizePath(entry.getName(), false);
        return path.replaceFirst("/[^/]+$", "/");
    }

    static String normalizePath(String path, boolean isDirectory) {
        String s = path.replace('\\', '/');
        if (isDirectory && !s.endsWith("/"))
            s += "/";
        return s;
    }

    public void closeEntry() throws IOException {
        // do nothing
    }

    @Override
    public void close() throws IOException {
        try {
            currentFolder.close();
            header.write(out, folders);
            for (CabCfFolder folder : folders)
                folder.writeCfDataInto(out, header);
            out.flush();
            flush();
            folders.clear();
        }
        finally {
            super.close();
        }
    }

}
