package jp.sfjp.armadillo.archive.lzh;

import static jp.sfjp.armadillo.archive.lzh.LzhHeader.HEADER_LEVEL_2;
import java.io.*;
import jp.sfjp.armadillo.archive.*;
import jp.sfjp.armadillo.io.*;

public final class LzhArchiveCreator implements ArchiveCreator {

    private LzhOutputStream os;

    public LzhArchiveCreator(OutputStream os) {
        this.os = new LzhOutputStream(os);
    }

    @Override
    public ArchiveEntry newEntry(String name) {
        LzhEntry entry = newLzhEntry();
        entry.setName(name);
        return entry;
    }

    @Override
    public void addEntry(ArchiveEntry entry, File file) throws IOException {
        if (file.isDirectory()) {
            final LzhEntry en = toLzhEntry(entry);
            en.setMethod(LzhMethod.LHD);
            os.putNextEntry(en);
            os.closeEntry();
        }
        else {
            InputStream is = new FileInputStream(file);
            try {
                addEntry(entry, is, file.length());
            }
            finally {
                is.close();
            }
        }
        entry.setAdded(true);
    }

    @Override
    public void addEntry(ArchiveEntry entry, InputStream is, long length) throws IOException {
        LzhEntry fileEntry = toLzhEntry(entry);
        fileEntry.setSize(length);
        os.putNextEntry(fileEntry);
        final long size = IOUtilities.transferAll(is, os);
        os.closeEntry();
        assert size == fileEntry.getSize() : "file size";
        assert size == length : "file size";
        entry.setSize(size);
        entry.setAdded(true);
    }

    static LzhEntry toLzhEntry(ArchiveEntry entry) {
        if (entry instanceof LzhEntry)
            return (LzhEntry)entry;
        LzhEntry newEntry = newLzhEntry();
        entry.copyTo(newEntry);
        return newEntry;
    }

    private static LzhEntry newLzhEntry() {
        return new LzhEntry(HEADER_LEVEL_2);
    }

    @Override
    public void close() throws IOException {
        os.close();
    }

}
