package jp.sfjp.armadillo.archive.lzh;

import jp.sfjp.armadillo.archive.*;
import jp.sfjp.armadillo.time.*;

/**
 * LZH archive entry.
 */
public final class LzhEntry extends ArchiveEntry {

    private static final TimeT TIME_T = new TimeT();
    private static final FTime FTIME = new FTime();
    private static final FileTime FILETIME = new FileTime();

    int headerLength;
    String method;
    long compressedSize;
    long size;
    private int ftime;
    private int timeT;
    private long fileTime;
    byte type;
    final byte headerLevel;
    short crc;
    int checksum;
    int calculatedChecksum;
    byte attribute;
    byte osIdentifier;

    public LzhEntry(int headerLevel) {
        super(false);
        this.headerLevel = (byte)headerLevel;
        this.method = LzhMethod.LH5;
        this.osIdentifier = 'J'; // not an official ?
    }

    @Override
    public boolean isDirectory() {
        return LzhMethod.LHD.equals(method);
    }

    public LzhMethod getMethod() throws LzhException {
        if (method == null)
            throw new IllegalStateException("method is null");
        return new LzhMethod(method);
    }

    public void setMethod(String method) {
        this.method = method;
    }

    @Override
    public long getSize() {
        return size;
    }

    @Override
    public void setSize(long size) {
        this.size = size;
    }

    @Override
    public long getCompressedSize() {
        return compressedSize;
    }

    @Override
    public void setCompressedSize(long compressedSize) {
        this.compressedSize = compressedSize;
    }

    @Override
    public long getLastModified() {
        TimeConverter tc = getTimeConverter(headerLevel);
        if (tc == FTIME)
            return FTIME.toMilliseconds(ftime);
        else if (tc == FILETIME)
            return tc.toMilliseconds(fileTime);
        else
            return tc.toMilliseconds(timeT);
    }

    @Override
    public void setLastModified(long time) {
        TimeConverter tc = getTimeConverter(headerLevel);
        if (tc == FTIME)
            ftime = FTIME.int32From(time);
        else if (tc == FILETIME)
            fileTime = tc.int64From(time);
        else
            timeT = tc.int32From(time);
    }

    @Override
    public String getMethodName() {
        return method;
    }

    private static TimeConverter getTimeConverter(int level) {
        switch (level) {
            case 0:
            case 1:
                return FTIME;
            case 2:
                return FILETIME;
            default:
                return TIME_T;
        }
    }

    int getFTime() {
        return FTIME.int32From(getLastModified());
    }

    void setFtime(short mdate, short mtime) {
        this.ftime = (mdate << 16) | mtime & 0xFFFF;
        setLastModified(FTIME.toMilliseconds(ftime));
    }

    long getFileTime() {
        return FILETIME.int64From(getLastModified());
    }

    void setFileTime(long filetime) {
        this.fileTime = filetime;
        setLastModified(FILETIME.toMilliseconds(filetime));
    }

    int getTimeT() {
        return TIME_T.int32From(getLastModified());
    }

    void setTimeT(int timet) {
        this.timeT = timet;
        setLastModified(TIME_T.toMilliseconds(timet));
    }

}
