package jp.sfjp.armadillo.archive.lzh;

import java.io.*;
import java.nio.channels.*;
import jp.sfjp.armadillo.archive.*;
import jp.sfjp.armadillo.io.*;

public final class LzhFile extends ArchiveFile {

    private File afile;
    private RandomAccessFile raf;
    private LzhHeader header;
    private LzhEntry ongoingEntry;
    private final LzhChecksum cksum;
    private final byte[] buffer;

    public LzhFile(File afile) {
        this.afile = afile;
        this.header = new LzhHeader();
        this.cksum = new LzhChecksum();
        this.buffer = new byte[8192];
    }

    public LzhFile(File afile, boolean withOpen) throws IOException {
        this(afile);
        open();
    }

    @Override
    public ArchiveEntry newEntry(String name) {
        LzhEntry entry = new LzhEntry(LzhHeader.HEADER_LEVEL_2);
        entry.setName(name);
        if (name.endsWith("/"))
            entry.method = LzhMethod.LHD;
        return entry;
    }

    @Override
    public void open() throws IOException {
        if (raf != null)
            throw new IOException("the file has been already opened");
        if (!afile.exists())
            afile.createNewFile();
        this.raf = new RandomAccessFile(afile, "rw");
        this.opened = true;
    }

    @Override
    public void reset() throws IOException {
        ongoingEntry = null;
        currentPosition = 0L;
        raf.seek(0L);
    }

    @Override
    public ArchiveEntry nextEntry() throws IOException {
        ensureOpen();
        if (ongoingEntry == null)
            currentPosition = 0L;
        else
            currentPosition += ongoingEntry.compressedSize;
        raf.seek(currentPosition);
        ongoingEntry = readCurrentEntry();
        currentPosition = raf.getFilePointer();
        return ongoingEntry;
    }

    @Override
    public void addEntry(ArchiveEntry entry, InputStream is, long length) throws IOException {
        final long p;
        if (raf.length() == 0)
            p = 0;
        else
            raf.seek(p = raf.length() - 1);
        OutputStream os = Channels.newOutputStream(raf.getChannel());
        LzhEntry newEntry = LzhArchiveCreator.toLzhEntry(entry);
        header.write(os, newEntry);
        if (length > 0) {
            final long p1 = raf.getFilePointer();
            writeData(newEntry, is, os, length);
            raf.seek(p);
            header.write(os, newEntry);
            assert raf.getFilePointer() == p1;
            raf.seek(p1 + newEntry.compressedSize);
        }
        raf.write(0);
        raf.seek(p);
        currentPosition = p;
    }

    private void writeData(LzhEntry entry, InputStream is, OutputStream os, long length) throws LzhException, IOException {
        final long p = raf.getFilePointer();
        cksum.reset();
        // XXX better way to flush remaining bits
        OutputStream los = LzhOutputStream.openStream(new AntiCloseOutputStream(os),
                                                      entry.getMethod());
        OutputStream ios = new InspectionOutputStream(los, cksum);
        final long written = IOUtilities.transfer(buffer, is, ios, length);
        assert written == length;
        los.close();
        entry.compressedSize = raf.getFilePointer() - p;
        entry.crc = cksum.getShortValue();
    }

    @Override
    public void updateEntry(ArchiveEntry entry, InputStream is, long length) throws IOException {
        removeEntry(entry);
        addEntry(entry, is, length);
    }

    @Override
    public void removeEntry(ArchiveEntry entry) throws IOException {
        if (!seek(entry))
            throw new LzhException("entry " + entry + " not found");
        assert ongoingEntry != null;
        final int headerLength = ongoingEntry.headerLength;
        final long bodyLength = ongoingEntry.compressedSize;
        truncate(currentPosition - headerLength, headerLength + bodyLength);
    }

    LzhEntry readCurrentEntry() throws IOException {
        return header.read(Channels.newInputStream(raf.getChannel()));
    }

    void truncate(final long offset, final long length) throws IOException {
        long p2 = offset;
        long p1 = p2 + length;
        while (true) {
            raf.seek(p1);
            final int r = raf.read(buffer);
            if (r <= 0)
                break;
            raf.seek(p2);
            raf.write(buffer, 0, r);
            p2 += r;
            p1 += r;
        }
        raf.setLength(raf.length() - length);
        reset();
    }

    @Override
    public void close() throws IOException {
        try {
            raf.close();
        }
        finally {
            super.close();
            afile = null;
            header = null;
        }
    }

    private static final class AntiCloseOutputStream extends BufferedOutputStream {

        public AntiCloseOutputStream(OutputStream os) {
            super(os, 32768);
        }

        @Override
        public void close() throws IOException {
            // ignore
        }

    }

}
