package jp.sfjp.armadillo.archive.tar;

import java.io.*;
import jp.sfjp.armadillo.archive.*;
import jp.sfjp.armadillo.io.*;

public final class TarArchiveCreator implements ArchiveCreator {

    private final TarOutputStream os;

    public TarArchiveCreator(OutputStream os) {
        this.os = new TarOutputStream(os);
    }

    @Override
    public ArchiveEntry newEntry(String name) {
        return new TarEntry(name);
    }

    @Override
    public void addEntry(ArchiveEntry entry, File file) throws IOException {
        if (file.isDirectory()) {
            os.putNextEntry(toTarEntry(entry));
            os.closeEntry();
        }
        else {
            InputStream is = new FileInputStream(file);
            try {
                addEntry(entry, is, file.length());
            }
            finally {
                is.close();
            }
        }
        entry.setAdded(true);
    }

    @Override
    public void addEntry(ArchiveEntry entry, InputStream is, long length) throws IOException {
        TarEntry fileEntry = toTarEntry(entry);
        fileEntry.setSize(length);
        os.putNextEntry(fileEntry);
        final long size = IOUtilities.transferAll(is, os);
        os.closeEntry();
        assert size == fileEntry.getSize() : "file size";
        assert size == length : "file size";
        entry.setSize(size);
        entry.setAdded(true);
    }

    static TarEntry toTarEntry(ArchiveEntry entry) {
        if (entry instanceof TarEntry)
            return (TarEntry)entry;
        TarEntry newEntry = new TarEntry(entry.getName());
        entry.copyTo(newEntry);
        return newEntry;
    }

    @Override
    public void close() throws IOException {
        os.close();
    }

}
