package jp.sfjp.armadillo.archive.tar;

import java.io.*;
import java.nio.channels.*;
import jp.sfjp.armadillo.archive.*;

public final class TarFile extends ArchiveFile { //implements ArchiveCreator, ArchiveExtractor {

    private File afile;
    private RandomAccessFile raf;
    private TarHeader header;
    private TarEntry ongoingEntry;
    private final byte[] buffer;

    public TarFile(File afile) {
        if (afile.isDirectory())
            throw new IllegalArgumentException("not a file: " + afile.getPath());
        this.afile = afile;
        this.header = new TarHeader();
        this.buffer = new byte[TarHeader.BLOCK_SIZE];
    }

    @Override
    public void open() throws IOException {
        if (raf != null)
            throw new IOException("the file has been already opened");
        if (!afile.exists())
            afile.createNewFile();
        this.raf = new RandomAccessFile(afile, "rw");
        this.opened = true;
    }

    @Override
    public void reset() throws IOException {
        ongoingEntry = null;
        currentPosition = 0L;
        raf.seek(0L);
    }

    @Override
    public ArchiveEntry nextEntry() throws IOException {
        ensureOpen();
        if (ongoingEntry == null)
            currentPosition = 0L;
        else {
            final long size = ongoingEntry.size;
            final long totalLength = TarHeader.BLOCK_SIZE + size + TarHeader.getSkipSize(size);
            assert totalLength % TarHeader.BLOCK_SIZE == 0;
            currentPosition += totalLength;
        }
        raf.seek(currentPosition);
        ongoingEntry = readCurrentEntry();
        raf.seek(currentPosition + TarHeader.BLOCK_SIZE);
        return ongoingEntry;
    }

    @Override
    public boolean seek(ArchiveEntry entry) throws IOException {
        ensureOpen();
        reset();
        while (true) {
            ArchiveEntry nextEntry = nextEntry();
            if (nextEntry == null)
                break;
            if (nextEntry.getName().equals(entry.getName()))
                return true;
        }
        reset();
        return false;
    }

    public void seekEndOfLastEntry() throws IOException {
        ensureOpen();
        reset();
        while (true)
            if (nextEntry() == null)
                break;
    }

    @Override
    public void addEntry(ArchiveEntry entry, InputStream is, long length) throws IOException {
        final long blockCount = calculateBlockCount(length);
        FileChannel fc = raf.getChannel();
        seekEndOfLastEntry();
        raf.seek(currentPosition);
        insertEmptyBlock(raf.getFilePointer(), blockCount + 1);
        TarEntry newEntry = toTarEntry(entry);
        header.write(Channels.newOutputStream(fc), newEntry);
        if (blockCount > 0) {
            final long p = currentPosition + TarHeader.BLOCK_SIZE;
            long written = fc.transferFrom(Channels.newChannel(is), p, length);
            assert written == length;
        }
        raf.seek(currentPosition);
        ongoingEntry = newEntry;
    }

    @Override
    public void updateEntry(ArchiveEntry entry, InputStream is, long length) throws IOException {
        final int blockSize = TarHeader.BLOCK_SIZE;
        if (!seek(entry))
            throw new TarException("entry " + entry + " not found");
        raf.seek(currentPosition);
        header.write(Channels.newOutputStream(raf.getChannel()), toTarEntry(entry));
        if (length > 0) {
            final long oldCount = calculateBlockCount(ongoingEntry.size);
            final long newCount = calculateBlockCount(length);
            assert newCount > 0;
            final long p = currentPosition + blockSize;
            if (newCount < oldCount)
                truncateBlock(p, oldCount - newCount);
            if (newCount <= oldCount) {
                raf.seek(p + (newCount - 1) * blockSize);
                raf.write(buffer);
            }
            else
                insertEmptyBlock(p, newCount - oldCount);
            raf.getChannel().transferFrom(Channels.newChannel(is), p, length);
        }
        raf.seek(currentPosition);
        ongoingEntry = TarArchiveCreator.toTarEntry(entry);
    }

    @Override
    public void removeEntry(ArchiveEntry entry) throws IOException {
        if (!seek(entry))
            throw new TarException("entry " + entry + " not found");
        raf.seek(currentPosition);
        assert ongoingEntry != null;
        final long size = ongoingEntry.size;
        if (size == 0)
            truncate(currentPosition, TarHeader.BLOCK_SIZE);
        else {
            final long totalLength = TarHeader.BLOCK_SIZE + size + TarHeader.getSkipSize(size);
            assert totalLength >= 0 && totalLength <= Integer.MAX_VALUE;
            assert totalLength % TarHeader.BLOCK_SIZE == 0;
            truncate(currentPosition, totalLength);
        }
    }

    static TarEntry toTarEntry(ArchiveEntry entry) {
        if (entry instanceof TarEntry)
            return (TarEntry)entry;
        TarEntry newEntry = new TarEntry(entry.getName());
        entry.copyTo(newEntry);
        return newEntry;
    }

    @Override
    public ArchiveEntry newEntry(String name) {
        return new TarEntry(name);
    }

    public void insertEmptyBlock(long offset, long blockCount) throws IOException {
        final int blockSize = TarHeader.BLOCK_SIZE;
        final long insertLength = blockCount * blockSize;
        raf.seek(raf.length() + insertLength - 1);
        raf.write(0);
        final long x = raf.getFilePointer();
        long p2 = x - blockSize; // last block
        long p1 = p2 - insertLength;
        assert p1 > 0 && p2 > 0;
        byte[] buffer = this.buffer;
        for (int i = 0; i < blockCount && p1 >= offset; i++) {
            raf.seek(p1);
            raf.readFully(buffer);
            raf.seek(p2);
            raf.write(buffer);
            p1 -= blockSize;
            p2 -= blockSize;
        }
        raf.seek(currentPosition = offset);
    }

    void truncateBlock(long offset, long blockCount) throws IOException {
        truncate(offset, blockCount * TarHeader.BLOCK_SIZE);
    }

    void truncate(final long offset, final long length) throws IOException {
        final int blockSize = TarHeader.BLOCK_SIZE;
        long p2 = offset;
        long p1 = p2 + length;
        final long restLength = raf.length() - p1;
        assert restLength >= blockSize && restLength % blockSize == 0;
        byte[] bytes = new byte[blockSize];
        final long blockCount = calculateBlockCount(restLength);
        for (int i = 0; i < blockCount; i++) {
            raf.seek(p1);
            raf.readFully(bytes);
            raf.seek(p2);
            raf.write(bytes);
            p2 += blockSize;
            p1 += blockSize;
        }
        raf.setLength(raf.length() - length);
        reset();
    }

    TarEntry readCurrentEntry() throws IOException {
        return header.read(Channels.newInputStream(raf.getChannel()));
    }

    static long calculateBlockCount(long size) {
        if (size == 0L)
            return 0L;
        final long r = size / TarHeader.BLOCK_SIZE;
        final long mod = size - r * TarHeader.BLOCK_SIZE;
        return r + (mod > 0 ? 1 : 0);
    }

    @Override
    public void close() throws IOException {
        try {
            raf.close();
        }
        finally {
            super.close();
            afile = null;
            header = null;
        }
    }

}
