package jp.sfjp.armadillo.archive.zip;

import java.io.*;
import java.nio.*;
import java.nio.channels.*;
import java.util.*;
import java.util.zip.*;
import jp.sfjp.armadillo.archive.*;
import jp.sfjp.armadillo.io.*;

/**
 * Dump ZIP archive header.
 */
public final class DumpZipHeader extends DumpArchiveHeader {

    private static final String fmt1 = "  * %s = %s%n";
    private static final String fmt2 = "  %1$-16s = [0x%2$08X] ( %2$d )%n";
    private static final int siglen = 4;
    private static final int siglen_m1 = siglen - 1;

    static final int SIGN_LOC = 0x04034B50;
    static final int SIGN_CEN = 0x02014B50;
    static final int SIGN_END = 0x06054B50;
    static final int SIGN_EXT = 0x08074B50;
    static final int LENGTH_LOC = 30;
    static final int LENGTH_CEN = 46;
    static final int LENGTH_END = 22;
    static final int LENGTH_EXT = 16;

    private final ByteBuffer buffer;

    public DumpZipHeader() {
        this.buffer = ByteBuffer.allocate(LENGTH_CEN).order(ByteOrder.LITTLE_ENDIAN);
    }

    @Override
    public void dump(InputStream is, PrintWriter out) throws IOException {
        final int bufferSize = 65536;
        byte[] bytes = new byte[bufferSize];
        int p = 0;
        RewindableInputStream pis = new RewindableInputStream(is, bufferSize);
        while (true) {
            Arrays.fill(bytes, (byte)0);
            final int readSize = pis.read(bytes);
            if (readSize <= 0)
                break;
            final int offset = findPK(bytes);
            if (offset < 0) {
                if (readSize < siglen)
                    break;
                pis.rewind(siglen_m1);
                p += readSize - siglen_m1;
                continue;
            }
            if (offset == 0)
                pis.rewind(readSize);
            else if (offset < bufferSize - siglen)
                pis.rewind(readSize - offset);
            p += offset;
            printOffset(out, p);
            ByteBuffer buffer = ByteBuffer.wrap(bytes, offset, siglen)
                                          .order(ByteOrder.LITTLE_ENDIAN);
            final int len;
            final int signature = buffer.getInt();
            switch (signature) {
                case SIGN_LOC:
                    len = readLOC(pis, out);
                    break;
                case SIGN_CEN:
                    len = readCEN(pis, out);
                    break;
                case SIGN_END:
                    len = readEND(pis, out);
                    break;
                case SIGN_EXT:
                    len = readEXT(pis, out);
                    break;
                default:
                    warn(out, "[%08X] is not a signature", signature);
                    len = 0;
            }
            assert len >= 0;
            if (len > 0)
                p += len;
            else {
                pis.read();
                ++p;
            }
        }
        printEnd(out, "ZIP", p);
    }

    static int findPK(byte[] bytes) {
        // PK=0x504B
        final int n = bytes.length - siglen_m1;
        for (int i = 0; i < n; i++)
            if (bytes[i] == 0x50)
                if (bytes[i + 1] == 0x4B)
                    return i;
        return -1;
    }

    int readLOC(InputStream is, PrintWriter out) throws IOException {
        int readSize = 0;
        // Local file header (LOC)
        final int signature; // local file header signature
        final short version; // version needed to extract
        final short flags; // general purpose bit flag
        final short method; // compression method
        final short mtime; // last mod file time
        final short mdate; // last mod file date
        final int crc; // crc-32
        final int compsize; // compressed size
        final int uncompsize; // uncompressed size
        final short namelen; // file name length
        final short extlen; // extra field length
        buffer.clear();
        buffer.limit(LENGTH_LOC);
        Channels.newChannel(is).read(buffer);
        readSize += buffer.position();
        buffer.rewind();
        signature = buffer.getInt();
        version = buffer.getShort();
        flags = buffer.getShort();
        method = buffer.getShort();
        mtime = buffer.getShort();
        mdate = buffer.getShort();
        crc = buffer.getInt();
        compsize = buffer.getInt();
        uncompsize = buffer.getInt();
        namelen = buffer.getShort();
        extlen = buffer.getShort();
        final String name;
        assert namelen >= 0;
        if (namelen == 0)
            name = "";
        else { // if (namelen > 0)
            byte[] nameBuffer = new byte[namelen];
            final int read = is.read(nameBuffer);
            name = new String(nameBuffer);
            if (read != namelen)
                warn(out, "namelen=%d, read=%d, name=%s", namelen, read, name);
            readSize += read;
        }
        if (extlen > 0) {
            final long read = is.skip(extlen);
            if (read != extlen)
                throw new ZipException("invalid LOC header (extra length)");
            readSize += read;
        }
        printHeaderName(out, "LOC header");
        out.printf(fmt1, "name", name);
        out.printf(fmt1, "mtime as date", toDate(mdate, mtime));
        out.printf(fmt2, "signature", signature);
        out.printf(fmt2, "version", version);
        out.printf(fmt2, "flags", flags);
        out.printf(fmt2, "method", method);
        out.printf(fmt2, "mtime", mtime);
        out.printf(fmt2, "mdate", mdate);
        out.printf(fmt2, "crc", crc);
        out.printf(fmt2, "compsize", compsize);
        out.printf(fmt2, "uncompsize", uncompsize);
        out.printf(fmt2, "namelen", namelen);
        out.printf(fmt2, "extlen", extlen);
        return readSize;
    }

    int readCEN(InputStream is, PrintWriter out) throws IOException {
        int readSize = 0;
        // Central file header (CEN)
        final int signature; // central file header signature
        final short madever; // version made by
        final short needver; // version needed to extract
        final short flags; // general purpose bit flag
        final short method; // compression method
        final short mtime; // last mod file time
        final short mdate; // last mod file date
        final int crc; // crc-32
        final int compsize; // compressed size
        final int uncompsize; // uncompressed size
        final short namelen; // file name length
        final short extlen; // extra field length
        final short fcmlen; // file comment length
        final short dnum; // disk number start
        final short inattr; // internal file attributes
        final int exattr; // external file attributes
        final int reloff; // relative offset of local header
        buffer.clear();
        buffer.limit(LENGTH_CEN);
        Channels.newChannel(is).read(buffer);
        readSize += buffer.position();
        if (buffer.position() == 0)
            return readSize;
        buffer.rewind();
        signature = buffer.getInt();
        madever = buffer.getShort();
        needver = buffer.getShort();
        flags = buffer.getShort();
        method = buffer.getShort();
        mtime = buffer.getShort();
        mdate = buffer.getShort();
        crc = buffer.getInt();
        compsize = buffer.getInt();
        uncompsize = buffer.getInt();
        namelen = buffer.getShort();
        extlen = buffer.getShort();
        fcmlen = buffer.getShort();
        dnum = buffer.getShort();
        inattr = buffer.getShort();
        exattr = buffer.getInt();
        reloff = buffer.getInt();
        final String name;
        if (namelen > 0) {
            final int reallen = namelen;
            byte[] nameBuffer = new byte[reallen];
            final int read = is.read(nameBuffer);
            name = new String(nameBuffer);
            if (read != namelen)
                warn(out, "namelen=%d, read=%d, name=%s", namelen, read, name);
        }
        else
            name = "";
        if (extlen > 0) {
            final long read = is.skip(extlen);
            if (read != extlen)
                throw new ZipException("invalid CEN header (extra length)");
            readSize += read;
        }
        if (fcmlen > 0)
            System.out.println();
        printHeaderName(out, "CEN header");
        out.printf(fmt1, "name", name);
        out.printf(fmt1, "mtime as date", toDate(mdate, mtime));
        out.printf(fmt2, "signature", signature);
        out.printf(fmt2, "madever", madever);
        out.printf(fmt2, "needver", needver);
        out.printf(fmt2, "flags", flags);
        out.printf(fmt2, "method", method);
        out.printf(fmt2, "mtime", mtime);
        out.printf(fmt2, "mdate", mdate);
        out.printf(fmt2, "crc", crc);
        out.printf(fmt2, "compsize", compsize);
        out.printf(fmt2, "uncompsize", uncompsize);
        out.printf(fmt2, "namelen", namelen);
        out.printf(fmt2, "extlen", extlen);
        out.printf(fmt2, "fcmlen", fcmlen);
        out.printf(fmt2, "dnum", dnum);
        out.printf(fmt2, "infattr", inattr);
        out.printf(fmt2, "exfattr", exattr);
        out.printf(fmt2, "reloff", reloff);
        return readSize;
    }

    int readEND(InputStream is, PrintWriter out) throws IOException {
        int readSize = 0;
        // End of central dir header (END)
        final int signature; // end of central dir signature
        final short ndisk; // number of this disk
        final short ndiskCEN; // number of the disk with the start of the central directory
        final short countDiskCENs; // total number of entries in the central directory on this disk
        final short countCENs; // total number of entries in the central directory
        final int sizeCENs; // size of the central directory
        final int offsetCENs; // offset of start of central directory with respect to the starting disk number
        final short commentlen; // .ZIP file comment length
        final byte[] comment; // .ZIP file comment
        buffer.clear();
        buffer.limit(LENGTH_END);
        Channels.newChannel(is).read(buffer);
        readSize += buffer.position();
        buffer.rewind();
        signature = buffer.getInt();
        ndisk = buffer.getShort();
        ndiskCEN = buffer.getShort();
        countDiskCENs = buffer.getShort();
        countCENs = buffer.getShort();
        sizeCENs = buffer.getInt();
        offsetCENs = buffer.getInt();
        commentlen = buffer.getShort();
        final String commentString;
        if (commentlen > 0) {
            comment = new byte[commentlen];
            final int read = is.read(comment);
            if (read != commentlen)
                throw new ZipException("invalid END header (comment length)");
            readSize += read;
            commentString = new String(comment);
        }
        else
            commentString = "";
        printHeaderName(out, "END header");
        out.printf(fmt2, "signature", signature);
        out.printf(fmt2, "ndisk", ndisk);
        out.printf(fmt2, "ndiskCEN", ndiskCEN);
        out.printf(fmt2, "countDiskCENs", countDiskCENs);
        out.printf(fmt2, "countCENs", countCENs);
        out.printf(fmt2, "sizeCENs", sizeCENs);
        out.printf(fmt2, "offsetCENs", offsetCENs);
        out.printf(fmt2, "commentlen", commentlen);
        out.printf(fmt1, "comment", commentString);
        return readSize;
    }

    int readEXT(InputStream is, PrintWriter out) throws IOException {
        int readSize = 0;
        // Extend header (EXT)
        final int signature; // extend header signature
        final int crc; // crc-32
        final int compsize; // compressed size
        final int uncompsize; // uncompressed size
        buffer.clear();
        buffer.limit(LENGTH_EXT);
        Channels.newChannel(is).read(buffer);
        readSize += buffer.position();
        buffer.rewind();
        signature = buffer.getInt();
        crc = buffer.getInt();
        compsize = buffer.getInt();
        uncompsize = buffer.getInt();
        printHeaderName(out, "EXT header");
        out.printf(fmt2, "signature", signature);
        out.printf(fmt2, "crc", crc);
        out.printf(fmt2, "compsize", compsize);
        out.printf(fmt2, "uncompsize", uncompsize);
        return readSize;
    }

    static Date toDate(short mdate, short mtime) {
        ZipEntry entry = new ZipEntry();
        entry.mdate = mdate;
        entry.mtime = mtime;
        return new Date(entry.getLastModified());
    }

}
