package jp.sfjp.armadillo.archive.zip;

import java.io.*;
import java.util.*;
import java.util.zip.*;
import jp.sfjp.armadillo.io.*;

final class ZipEndEntry {

    int signature;
    short disknum;
    short disknumCEN;
    short countDiskCENs;
    short countCENs;
    int sizeCENs;
    int offsetStartCEN;
    short commentlen;
    byte[] comment;

    static ZipEndEntry create(long offset, List<ZipEntry> entries) throws IOException {
        // End of central dir header (END)
        final int signature; // end of central dir signature
        final short disknum; // number of this disk
        final short disknumCEN; // number of the disk with the start of the central directory
        final short countDiskCENs; // total number of entries in the central directory on this disk
        final short countCENs; // total number of entries in the central directory
        final int sizeCENs; // size of the central directory
        final int offsetStartCEN; // offset of start of central directory with respect to the starting disk number
        final short commentlen; // .ZIP file comment length
        final byte[] comment; // .ZIP file comment
        // ---
        final int iEntryCount = entries.size();
        if (iEntryCount > 0xFFFF)
            throw new ZipException("overflow: entry-count=" + iEntryCount);
        final short entryCount = (short)(iEntryCount & 0xFFFF);
        if (offset > 0xFFFFFFFFL)
            throw new ZipException("overflow: offset=" + offset);
        assert offset >= 0L;
        final long p = offset;
        ZipHeader header = new ZipHeader();
        VolumetricOutputStream vos = new VolumetricOutputStream();
        for (final ZipEntry entry : entries)
            header.writeCEN(vos, entry);
        final long lsizeCENs = vos.getSize();
        if (lsizeCENs > 0xFFFFFFFFL)
            throw new ZipException("overflow: size of CEN headers=" + lsizeCENs);
        // ... prepared
        signature = ZipHeader.SIGN_END;
        disknum = 0; // not supported
        disknumCEN = 0; // not supported
        countDiskCENs = entryCount; // ignore disk number
        countCENs = entryCount;
        sizeCENs = (int)(lsizeCENs & 0xFFFFFFFFL);
        offsetStartCEN = (int)(p & 0xFFFFFFFF);
        commentlen = 0; // not supported
        comment = new byte[0]; // not supported
        ZipEndEntry entry = new ZipEndEntry();
        entry.signature = signature;
        entry.disknum = disknum;
        entry.disknumCEN = disknumCEN;
        entry.countDiskCENs = countDiskCENs;
        entry.countCENs = countCENs;
        entry.sizeCENs = sizeCENs;
        entry.offsetStartCEN = offsetStartCEN;
        entry.commentlen = commentlen;
        entry.comment = comment;
        return entry;
    }

}
