package jp.sfjp.armadillo.compression.lzhuf;

import java.io.*;

public final class LzssOutputStream extends FilterOutputStream {

    private final int dictionarySize;
    private final int matchSize;
    private final int threshold;

    private boolean closed;
    private LzssEncoderWritable output;
    private byte[] buffer;
    private int index;
    private int limit;

    public LzssOutputStream(LzssEncoderWritable output,
                            int dictionarySize,
                            int matchSize,
                            int threshold) {
        super(null);
        this.closed = false;
        this.output = output;
        this.dictionarySize = dictionarySize;
        this.matchSize = matchSize;
        this.threshold = threshold;
        this.buffer = new byte[dictionarySize];
    }

    @Override
    public void write(int b) throws IOException {
        write(new byte[]{(byte)b}, 0, 1);
    }

    @Override
    public void write(byte[] b, int off, int len) throws IOException {
        if (closed)
            throw new IOException("stream already closed");
        int remaining = len;
        int offset = off;
        while (remaining > 0) {
            final int rest = dictionarySize - limit;
            final int length = (rest < remaining) ? rest : remaining;
            System.arraycopy(b, offset, buffer, limit, length);
            remaining -= length;
            offset += length;
            limit += length;
            if (length == rest)
                encode();
        }
    }

    @Override
    public void flush() throws IOException {
        if (closed)
            throw new IOException("stream already closed");
        encode();
        output.flush();
    }

    private void encode() throws IOException {
        while (index < limit) {
            /*
             * linear search (and the performance is not good enough.)
             */
            int mark = -1;
            int length = 0;
            for (int i = index - 1; i >= 0; i--) {
                if (i + length >= limit
                    || index + length >= limit
                    || buffer[i] != buffer[index]
                    || buffer[i + length] != buffer[index + length])
                    continue;
                int matched = 0;
                while (matched < matchSize && index + matched < limit)
                    if (buffer[i + matched] == buffer[index + matched])
                        ++matched;
                    else
                        break;
                if (matched > length) {
                    mark = i;
                    length = matched;
                }
            }
            final int offset = (length >= threshold) ? index - mark : -1;
            if (offset >= 0) {
                output.writeMatched(offset, length);
                index += length;
            }
            else
                output.write(buffer[index++] & 0xFF);
        }
        if (limit > dictionarySize - matchSize) {
            index -= matchSize;
            limit -= matchSize;
            System.arraycopy(buffer, matchSize, buffer, 0, limit);
        }
    }

    @Override
    public void close() throws IOException {
        if (closed)
            throw new IOException("stream already closed");
        try {
            flush();
        }
        finally {
            buffer = null;
            output.close();
            output = null;
        }
    }

}
