package jp.sfjp.armadillo.ui.window;

import static jp.sfjp.armadillo.ui.window.MainWindow.ActionMode.*;
import static jp.sfjp.armadillo.ui.window.MainWindow.ActionMode.List;
import java.awt.*;
import java.awt.datatransfer.*;
import java.awt.dnd.*;
import java.awt.event.*;
import java.io.*;
import java.lang.Thread.UncaughtExceptionHandler;
import java.util.*;
import java.util.List;
import java.util.concurrent.*;
import javax.swing.*;
import jp.sfjp.armadillo.*;
import jp.sfjp.armadillo.archive.*;

public final class MainWindow extends JFrame {

    static Logger log = Logger.getLogger(MainWindow.class);
    static ResourceManager res = ResourceManager.getInstance("jp.sfjp.armadillo.ui.messages");
    static Color doneBGC = Color.decode("#99ff99");

    private final Executor executor;
    private final Box box;
    private final JComboBox archiveTypeCombo;
    private final JCheckBox listMode;

    public MainWindow() {
        setTitle(getTitleString());
        setDefaultCloseOperation(EXIT_ON_CLOSE);
        this.executor = Executors.newCachedThreadPool();
        this.box = Box.createVerticalBox();
        this.archiveTypeCombo = new JComboBox(getArchiveExtensions());
        this.listMode = new JCheckBox();
        setDropTarget(new DropTarget(this, new DropTargetAdapter() {
            @Override
            public void drop(DropTargetDropEvent dtde) {
                Transferable t = dtde.getTransferable();
                if (t.isDataFlavorSupported(DataFlavor.javaFileListFlavor)) {
                    dtde.acceptDrop(DnDConstants.ACTION_COPY_OR_MOVE);
                    List<File> fileList;
                    try {
                        @SuppressWarnings("unchecked")
                        List<File> o = (List<File>)t.getTransferData(DataFlavor.javaFileListFlavor);
                        fileList = o;
                    }
                    catch (Exception ex) {
                        if (ex instanceof RuntimeException)
                            throw (RuntimeException)ex;
                        throw new RuntimeException(ex);
                    }
                    for (final File file : fileList)
                        startAction(file);
                }
            }
        }));
        JScrollPane sp = new JScrollPane(box,
                                         ScrollPaneConstants.VERTICAL_SCROLLBAR_ALWAYS,
                                         ScrollPaneConstants.HORIZONTAL_SCROLLBAR_ALWAYS);
        JPanel toolbar = new JPanel(new FlowLayout());
        toolbar.add(new JButton(new AbstractAction(res.get("button.closeFinishedProgressPanel")) {
            @Override
            public void actionPerformed(ActionEvent e) {
                clearFinishedProgressPanel();
            }
        }));
        toolbar.add(new JLabel("    "));
        toolbar.add(new JLabel(res.get("label.extensionToCreateArchive")));
        toolbar.add(archiveTypeCombo);
        toolbar.add(new JLabel("    "));
        toolbar.add(new JLabel(res.get("label.listMode")));
        toolbar.add(listMode);
        archiveTypeCombo.setPreferredSize(new Dimension(60, 20));
        add(toolbar, BorderLayout.NORTH);
        add(sp, BorderLayout.CENTER);
    }

    static String getTitleString() {
        return res.get(".title", ArmadilloCommands.getVersionString());
    }

    static Object[] getArchiveExtensions() {
        String[] extensions = res.get("extensionsToCreateArchive").split(",", -1);
        List<String> a = new ArrayList<String>(extensions.length);
        for (String extension : extensions)
            a.add("." + extension);
        return a.toArray();
    }

    void createArchive(File srcdir, File dst, ProgressNotifier notifier) throws IOException {
        log.setEnteredMethodName("createArchive");
        log.debug("start to create archive %s into %s", dst, srcdir);
        ArmadilloCommands.createArchive(srcdir, dst, notifier);
        log.trace("done");
        log.atExit();
    }

    void showInfoListWindow(File src, ProgressNotifier progressNotifier) throws IOException {
        log.setEnteredMethodName("showInfoListWindow");
        log.debug("start to show list of archive %s", src);
        ArchiveType type = ArchiveType.of(src.getName());
        ListDialog d = new ListDialog(this, src, type, progressNotifier);
        d.setSize(600, 400);
        d.setModal(false);
        d.setVisible(true);
        log.trace("done");
        log.atExit();
    }

    void extractArchive(File src, File dstdir, ProgressNotifier progressNotifier) throws IOException {
        log.setEnteredMethodName("extractArchive");
        log.debug("start to extract all files from archive %s into %s", src, dstdir);
        ArmadilloCommands.extractAll(src, dstdir, progressNotifier);
        log.trace("done");
        log.atExit();
    }

    void extractArchive(File src,
                        File dstdir,
                        Set<String> targets,
                        ProgressNotifier progressNotifier) throws IOException {
        log.setEnteredMethodName("extractArchive(+targets)");
        log.debug("start to extract %d files from archive %s into %s", targets.size(), src, dstdir);
        ArmadilloCommands.extract(src, dstdir, targets, progressNotifier);
        log.trace("done");
        log.atExit();
    }

    ActionMode detectMode(File file) {
        if (listMode.isSelected()) {
            if (file.isDirectory())
                return Unknown;
            return List;
        }
        if (file.isDirectory())
            return Create;
        if (ArchiveType.of(file.getName()) != ArchiveType.Unknown)
            return Extract;
        return Unknown;
    }

    void startAction(File... files) {
        Set<String> emptySet = Collections.emptySet();
        for (File file : files)
            executor.execute(new ActionWorker(file, detectMode(file), emptySet, file.length()));
    }

    void startExtractAction(File src, long totalSize, Set<String> targets) {
        executor.execute(new ActionWorker(src, Extract, targets, totalSize));
    }

    String getSelectedArchiveType() {
        return String.valueOf(archiveTypeCombo.getSelectedItem());
    }

    void clearFinishedProgressPanel() {
        final JComponent parent = box;
        for (Component c : parent.getComponents())
            if (c instanceof ProgressInfoPanel) {
                final ProgressInfoPanel infoPanel = (ProgressInfoPanel)c;
                if (infoPanel.isFinished()) {
                    parent.remove(c);
                    EventQueue.invokeLater(new Runnable() {
                        @Override
                        public void run() {
                            ((JComponent)parent.getParent()).updateUI();
                        }
                    });
                }
            }
    }

    ProgressInfoPanel addProgressInfoPanel(File src, File dst, ActionMode mode, Set<String> targets) {
        ProgressInfoPanel infoPanel = new ProgressInfoPanel(box, mode, src, dst, targets);
        synchronized (box) {
            infoPanel.setMaximumSize(new Dimension(0x8000, infoPanel.getPreferredSize().height));
            box.add(infoPanel);
            box.revalidate();
            return infoPanel;
        }
    }

    private final class ActionWorker extends SwingWorker<Integer, Integer> {

        final File file;
        final ActionMode mode;
        final Set<String> targets;
        final long totalSize;
        File dst;
        ProgressInfoPanel infoPanel;
        boolean errorOccurred;
        String errorMessage;

        ActionWorker(File file, ActionMode mode, Set<String> targets, long totalSize) {
            this.file = file;
            assert mode != AutoDetects;
            this.mode = mode;
            this.targets = targets;
            this.totalSize = totalSize;
            switch (this.mode) {
                case Create:
                    final String extension = getSelectedArchiveType();
                    this.dst = new File(file.getParentFile(), file.getName() + extension);
                    break;
                case Extract:
                    this.dst = ArmadilloCommands.getDestinationDirectory(file);
                    break;
                default:
            }
            this.infoPanel = addProgressInfoPanel(file, dst, this.mode, targets);
            this.errorOccurred = false;
            this.errorMessage = "(unknown error)";
        }

        @Override
        protected Integer doInBackground() throws Exception {
            log.setEnteredMethodName("ActionWorker#doInBackground");
            try {
                switch (mode == Unknown ? detectMode(file) : mode) {
                    case Create:
                        final long totalSizeInDir = ArmadilloCommands.getTotalSize(file);
                        createArchive(file, dst, createProgressNotifier(totalSizeInDir));
                        break;
                    case List:
                        showInfoListWindow(file, createProgressNotifier(totalSize));
                        break;
                    case Extract:
                        if (targets.isEmpty())
                            extractArchive(file, dst, createProgressNotifier(totalSize));
                        else
                            extractArchive(file, dst, targets, createProgressNotifier(totalSize));
                        break;
                    default:
                }
            }
            catch (Throwable ex) {
                log.error(ex);
                errorOccurred = true;
                errorMessage = String.format("error at %s: %s", file, ex);
                if (ex instanceof Exception)
                    throw (Exception)ex;
                if (ex instanceof Error)
                    throw (Error)ex;
            }
            return 0;
        }

        private ProgressNotifier createProgressNotifier(long totalSize) {
            return infoPanel.getProgressNotifier(totalSize);
        }

        @Override
        protected void done() {
            infoPanel.setFinished(true);
            if (errorOccurred)
                infoPanel.setMessage(errorMessage);
        }

    }

    enum ActionMode {
        AutoDetects, Create, List, Extract, Unknown;
    }

    private final class ProgressInfoPanel extends JPanel {

        private final JComponent parent;
        private final JProgressBar pbar;
        private final JLabel lTimeInfo;
        private final JLabel lMessage;
        private final StringBuilder messageBuffer;

        private boolean finished;
        private final long startTime;

        ProgressInfoPanel(final JComponent parent,
                          ActionMode mode,
                          File src,
                          File dst,
                          Set<String> targets) {
            this.startTime = System.currentTimeMillis();
            this.parent = parent;
            this.pbar = new JProgressBar(0, 1000);
            this.messageBuffer = new StringBuilder();
            this.lTimeInfo = new JLabel();
            this.lTimeInfo.setText(String.format("time: started at %tT", startTime));
            this.lMessage = new JLabel();
            setLayout(new GridLayout(5, 1));
            // JButton closeButton = new JButton(new ImageIcon(getClass().getResource("close.png")));
            // closeButton.addActionListener(new ActionListener() {
            //     @Override
            //     public void actionPerformed(ActionEvent e) {
            //         if (isFinished()) {
            //             parent.remove(thisPanel);
            //             EventQueue.invokeLater(new Runnable() {
            //                 @Override
            //                 public void run() {
            //                     ((JComponent)parent.getParent()).updateUI();
            //                 }
            //             });
            //         }
            //     }
            // });
            setMessage("");
            setBorder(BorderFactory.createLineBorder(Color.GRAY));
            JPanel p = new JPanel(new BorderLayout());
            p.setOpaque(false);
            p.add(new JLabel(res.get("mode.caption." + mode)), BorderLayout.CENTER);
            // p.add(closeButton, BorderLayout.WEST);
            add(p, BorderLayout.NORTH);
            final String dstString = (mode == List) ? res.get("word.dialog") : String.valueOf(dst);
            if (targets.isEmpty())
                add(new JLabel(String.format("%s -> %s", src, dstString)));
            else
                add(new JLabel(String.format("%d files in %s -> %s", targets.size(), src, dstString)));
            add(this.pbar);
            add(this.lTimeInfo);
            add(this.lMessage);
        }

        void setMessage(String msg) {
            messageBuffer.append(msg);
            lMessage.setText("message: " + messageBuffer);
            parent.updateUI();
        }

        void setRate(int rate) {
            pbar.setValue(rate);
            parent.updateUI();
        }

        void setFinished(boolean finished) {
            this.finished = finished;
            final long endTime = System.currentTimeMillis();
            final long elapsed = (endTime - startTime) / 1000L;
            setMessage(res.get("status.done"));
            lTimeInfo.setText(String.format("elapsed: %d secs ( %tT to %tT )",
                                            elapsed,
                                            startTime,
                                            endTime));
            setBackground(doneBGC);
        }

        boolean isFinished() {
            return finished;
        }

        ProgressNotifier getProgressNotifier(long total) {
            return new ItsProgressNotifier(total);
        }

        private final class ItsProgressNotifier extends ProgressNotifier implements Runnable {
            protected ItsProgressNotifier(long total) {
                super(total);
            }
            @Override
            public void progressNotified(long part, long total) {
                EventQueue.invokeLater(this);
            }
            @Override
            public void run() {
                setRate(getRateAsIntPermill());
                updateUI();
            }
        }

    }

    public static void main(final String[] args) {
        Thread.setDefaultUncaughtExceptionHandler(new UncaughtExceptionHandler() {
            @Override
            public void uncaughtException(Thread t, Throwable e) {
                log.fatal(e, "at thread %s", t);
                e.printStackTrace();
            }
        });
        EventQueue.invokeLater(new Runnable() {
            @Override
            public void run() {
                setLaF();
                MainWindow w = new MainWindow();
                w.setSize(640, 400);
                w.setLocationByPlatform(true);
                w.setVisible(true);
                for (final String arg : args)
                    w.startAction(new File(arg));
            }
            void setLaF() {
                try {
                    UIManager.setLookAndFeel(UIManager.getSystemLookAndFeelClassName());
                }
                catch (RuntimeException ex) {
                    throw ex;
                }
                catch (Exception ex) {
                    throw new IllegalStateException(ex);
                }
            }
        });
    }

}
