#ifndef __CODEC_H__
#define __CODEC_H__

#include "Common.h"

namespace AScript {

//-----------------------------------------------------------------------------
// Codec
//-----------------------------------------------------------------------------
class DLLDECLARE Codec {
public:
	enum Result {
		RESULT_None,
		RESULT_Complete,
		RESULT_Error,
	};
	typedef std::map<unsigned short, unsigned short> Map;
protected:
	int _idxBuff;
	bool _processEOLFlag;
	char _buffOut[8];
public:
	inline Codec(bool processEOLFlag) : _idxBuff(0), _processEOLFlag(processEOLFlag) {}
	bool FollowChar(char &chConv);
	inline void SetProcessEOLFlag(bool processEOLFlag) {
		_processEOLFlag = processEOLFlag;
	}
	inline bool IsProcessEOL() const { return _processEOLFlag; }
	virtual Result FeedChar(char ch, char &chConv) = 0;
	virtual Result Flush(char &chConv);
	static const char *EncodingFromLANG();
protected:
	inline void StoreChar(char ch) { _buffOut[_idxBuff++] = ch; }
};

//-----------------------------------------------------------------------------
// Codec_None
//-----------------------------------------------------------------------------
class DLLDECLARE Codec_None : public Codec {
public:
	inline Codec_None() : Codec(false) {}
	virtual Result FeedChar(char ch, char &chConv);
};

//-----------------------------------------------------------------------------
// Codec_Encoder
//-----------------------------------------------------------------------------
class DLLDECLARE Codec_Encoder : public Codec {
public:
	inline Codec_Encoder(bool processEOLFlag) : Codec(processEOLFlag) {}
	bool Encode(Binary &dst, const String &src);
};

//-----------------------------------------------------------------------------
// Codec_Decoder
//-----------------------------------------------------------------------------
class DLLDECLARE Codec_Decoder : public Codec {
public:
	inline Codec_Decoder(bool processEOLFlag) : Codec(processEOLFlag) {}
	bool Decode(String &dst, const Binary &src);
};

//-----------------------------------------------------------------------------
// CodecFactory
//-----------------------------------------------------------------------------
class DLLDECLARE CodecFactory {
public:
	typedef std::vector<CodecFactory *> List;
private:
	static List *_pList;
	String _name;
public:
	CodecFactory(const char *name);
	inline const char *GetName() const { return _name.c_str(); }
	virtual Codec_Encoder *CreateEncoder(bool processEOLFlag) = 0;
	virtual Codec_Decoder *CreateDecoder(bool processEOLFlag) = 0;
	static CodecFactory *Lookup(const char *name);
	static inline const List &GetList() { return *_pList; }
};

//-----------------------------------------------------------------------------
// Codec_Encoder_USASCII
//-----------------------------------------------------------------------------
class DLLDECLARE Codec_Encoder_USASCII : public Codec_Encoder {
public:
	inline Codec_Encoder_USASCII(bool processEOLFlag) : Codec_Encoder(processEOLFlag) {}
	virtual Result FeedChar(char ch, char &chConv);
};

//-----------------------------------------------------------------------------
// Codec_Decoder_USASCII
//-----------------------------------------------------------------------------
class DLLDECLARE Codec_Decoder_USASCII : public Codec_Decoder {
public:
	inline Codec_Decoder_USASCII(bool processEOLFlag) : Codec_Decoder(processEOLFlag) {}
	virtual Result FeedChar(char ch, char &chConv);
};

//-----------------------------------------------------------------------------
// Codec_Encoder_Through
//-----------------------------------------------------------------------------
class DLLDECLARE Codec_Encoder_Through : public Codec_Encoder {
public:
	inline Codec_Encoder_Through(bool processEOLFlag) : Codec_Encoder(processEOLFlag) {}
	virtual Result FeedChar(char ch, char &chConv);
};

//-----------------------------------------------------------------------------
// Codec_Decoder_Through
//-----------------------------------------------------------------------------
class DLLDECLARE Codec_Decoder_Through : public Codec_Decoder {
public:
	inline Codec_Decoder_Through(bool processEOLFlag) : Codec_Decoder(processEOLFlag) {}
	virtual Result FeedChar(char ch, char &chConv);
};

//-----------------------------------------------------------------------------
// Codec_Encoder_Base64
//-----------------------------------------------------------------------------
class DLLDECLARE Codec_Encoder_Base64 : public Codec_Encoder {
private:
	String _buff;
	static const char _chars[];
public:
	inline Codec_Encoder_Base64(bool processEOLFlag) : Codec_Encoder(processEOLFlag) {
		_buff.reserve(3);
	}
	virtual Result FeedChar(char ch, char &chConv);
	virtual Result Flush(char &chConv);
};

//-----------------------------------------------------------------------------
// Codec_Decoder_Base64
//-----------------------------------------------------------------------------
class DLLDECLARE Codec_Decoder_Base64 : public Codec_Decoder {
private:
	int _nChars;
	int _nInvalid;
	unsigned long _accum;
public:
	inline Codec_Decoder_Base64(bool processEOLFlag) :
			Codec_Decoder(processEOLFlag), _nChars(0), _nInvalid(0), _accum(0) {}
	virtual Result FeedChar(char ch, char &chConv);
};

//-----------------------------------------------------------------------------
// Codec_Encoder_UTF
//-----------------------------------------------------------------------------
class DLLDECLARE Codec_Encoder_UTF : public Codec_Encoder {
protected:
	int _cntChars;
	unsigned long _codeUTF32;
public:
	inline Codec_Encoder_UTF(bool processEOLFlag) :
			Codec_Encoder(processEOLFlag), _cntChars(0), _codeUTF32(0x00000000) {}
	inline unsigned long GetUTF32() const { return _codeUTF32; }
	virtual Result FeedChar(char ch, char &chConv);
	virtual Result FeedUTF32(unsigned long codeUTF32, char &chConv) = 0;
};

//-----------------------------------------------------------------------------
// Codec_Decoder_UTF
//-----------------------------------------------------------------------------
class DLLDECLARE Codec_Decoder_UTF : public Codec_Decoder {
public:
	inline Codec_Decoder_UTF(bool processEOLFlag) : Codec_Decoder(processEOLFlag) {}
	Result FeedUTF32(unsigned long codeUTF32, char &chConv);
};

//-----------------------------------------------------------------------------
// Codec_Encoder_UTF16LE
//-----------------------------------------------------------------------------
class DLLDECLARE Codec_Encoder_UTF16LE : public Codec_Encoder_UTF {
public:
	inline Codec_Encoder_UTF16LE(bool processEOLFlag) : Codec_Encoder_UTF(processEOLFlag) {}
	virtual Result FeedUTF32(unsigned long codeUTF32, char &chConv);
};

//-----------------------------------------------------------------------------
// Codec_Decoder_UTF16LE
//-----------------------------------------------------------------------------
class DLLDECLARE Codec_Decoder_UTF16LE : public Codec_Decoder_UTF {
private:
	bool _firstByteFlag;
	unsigned long _codeUTF32;
public:
	inline Codec_Decoder_UTF16LE(bool processEOLFlag) : Codec_Decoder_UTF(processEOLFlag),
						_codeUTF32(0x00000000), _firstByteFlag(true) {}
	virtual Result FeedChar(char ch, char &chConv);
};

}

//-----------------------------------------------------------------------------
// macros
//-----------------------------------------------------------------------------
#define AScript_RegisterCodecFactory(symbol, dispname, encoderClass, decoderClass) \
class CodecFactory_##symbol : public CodecFactory {										\
public:																					\
	inline CodecFactory_##symbol(const char *name = (dispname)) : CodecFactory(name) {}	\
	virtual Codec_Encoder *CreateEncoder(bool processEOLFlag);									\
	virtual Codec_Decoder *CreateDecoder(bool processEOLFlag);									\
};																						\
CodecFactory_##symbol _codecFactory_##symbol;											\
Codec_Encoder *CodecFactory_##symbol::CreateEncoder(bool processEOLFlag) { return new encoderClass(processEOLFlag); } \
Codec_Decoder *CodecFactory_##symbol::CreateDecoder(bool processEOLFlag) { return new decoderClass(processEOLFlag); }

#endif
