<?php
/**
 * AutoSOAP - Expanded SOAP Server
 * 
 * PHP version 5
 * 
 * @package jp.servlet.AutoSOAP
 * @author Sakamoto Kouichi <sakamoto@servlet.sakura.ne.jp> 
 * @copyright 2006 Sakamoto Kouichi
 * @license http://www.apache.org/licenses/LICENSE-2.0 Apache Software License 2.0
 * 
 * $Id: ReflectionAccept.php 93 2006-04-16 15:23:57Z sakamoto $
 */

/**
 * AutoSOAP_ReflectionAccept インターフェース
 * 
 * @package jp.servlet.AutoSOAP
 * @author Sakamoto Kouichi <sakamoto@servlet.sakura.ne.jp> 
 */
interface AutoSOAP_ReflectionAccept {
    /**
     * AutoSOAP_ReflectionVisitorを受け入れる(Visitorパターン)
     * 
     * @param AutoSOAP_ReflectionVisitor $visitor
     */
    public function accept(AutoSOAP_ReflectionVisitor $visitor);
} 

/**
 * ReflectorにAcceptに対して処理を行います。
 * 
 * @package jp.servlet.AutoSOAP
 * @author Sakamoto Kouichi <sakamoto@servlet.sakura.ne.jp> 
 */
class AutoSOAP_ReflectionAcceptImpl implements AutoSOAP_ReflectionAccept, Reflector
{
    private $_reflection;
    private $_visit_method;

    /**
     * コンストラクタ
     * 
     * @final 
     * @param Reflector $reflection 
     */
    final public function __construct(Reflector $reflection)
    {
        $this->_reflection = $reflection;
        if ($reflection instanceof ReflectionParameter) {
            $this->_visit_method = "visitParameter";
        } else if ($reflection instanceof ReflectionMethod) {
            $this->_visit_method = "visitMethod";
        } else if ($reflection instanceof ReflectionObject) {
            $this->_visit_method = "visitObject";
        } else if ($reflection instanceof ReflectionProperty) {
            $this->_visit_method = "visitProperty";
        } else if ($reflection instanceof ReflectionExtension) {
            $this->_visit_method = "visitExtension";
        } else if ($reflection instanceof ReflectionFunction) {
            $this->_visit_method = "visitFunction";
        } else if ($reflection instanceof ReflectionClass) {
            $this->_visit_method = "visitClass";
        } else {
            $this->_visit_method = "";
        } 
    } 

    /**
     * エクスポート
     * 
     * @static 
     * @param mixed $obj 
     */
    static public function export($obj)
    {
        return $this->_reflection->export($obj);
    } 

    /**
     * __toString
     */
    public function __toString()
    {
        return $this->_reflection->__toString();
    } 

    /**
     * AutoSOAP_ReflectionVisitorを受け入れる(Visitorパターン)
     * 
     * @final 
     * @param AutoSOAP_ReflectionVisitor $visitor
     */
    final public function accept(AutoSOAP_ReflectionVisitor $visitor)
    {
        $method = $this->_visit_method;
        return $visitor->$method($this);
    } 

    /**
     * リフレクションの型を返す
     * 
     * @final 
     */
    final public function getReflectionType()
    {
        return get_class($this->_reflection);
    } 

    /**
     * Reflectorを継承したクラスすべてをAutoSOAP_ReflectionAcceptへ変換する
     * 
     * @final 
     * @param mixed $target 
     */
    final private function &convertAutoSOAP_ReflectionAcceptImpl($target)
    {
        if (is_array($target)) {
            $tmp = new AutoSOAP_ReflectionChildren();
            foreach($target as $index => &$val) {
                if (is_object($val)) {
                    $tmp->add($this->convertAutoSOAP_ReflectionAcceptImpl($val), $index);
                } else {
                    $tmp->add($val, $index);
                } 
            } 
            return $tmp;
        } else if ($target instanceof Reflector) {
            $tmp = new AutoSOAP_ReflectionAcceptImpl($target);
            return $tmp;
        } 
        return $target;
    } 

    /**
     * プロパティの設定はReflectorに任せる
     * 
     * @final 
     * @param mixed $propety プロパティ名
     * @param mixed $value 値
     */
    final public function __set($propety, $value)
    {
        $this->_reflection->$propety = $value;
    } 
    /**
     * プロパティの取得はReflectorに任せる
     * 
     * @final 
     * @param mixed $propety プロパティ名
     */
    final public function __get($propety)
    {
        return $this->_reflection->$propety;
    } 
    /**
     * メソッドの処理はReflectorに任せる
     * 
     * @final 
     * @param mixed $method 関数名
     * @param mixed $args 引数の配列
     */
    final public function __call($method, $args)
    {
        $result = call_user_func_array(array($this->_reflection, $method), $args);
        return $this->convertAutoSOAP_ReflectionAcceptImpl($result);
    } 
} 

/**
 * AutoSOAP_ReflectionChildren
 * 
 * @package jp.servlet.AutoSOAP
 * @author Sakamoto Kouichi <sakamoto@servlet.sakura.ne.jp> 
 */
class AutoSOAP_ReflectionChildren implements AutoSOAP_ReflectionAccept {
    private $_children = array();

    /**
     * AutoSOAP_ReflectionVisitorを受け入れる(Visitorパターン)
     * 
     * @final 
     * @param AutoSOAP_ReflectionVisitor $visitor
     */
    final public function accept(AutoSOAP_ReflectionVisitor $visitor)
    {
        return $visitor->visitChildren($this);
    } 

    /**
     * 子の追加
     * 
     * @final 
     * @param AutoSOAP_ReflectionAcceptImpl $child
     * @param integer $index 
     */
    final public function add(AutoSOAP_ReflectionAcceptImpl $child, $index = -1)
    {
        $count = count($this->_children);
        if (is_numeric($index) && -1 < $index && $count > $index) {
            if (0 == $index) {
                array_unshift($this->_children, $child);
            } else {
                for($i = $count-1;$i >= $index;$i--) {
                    $this->_children[$index + 1] = $this->_children[$index];
                } 
                $this->_children[$index] = $child;
            } 
        } else if ('' == $index) {
            $this->_children[] = $child;
        } else if (is_string($index)) {
            $this->_children[$index] = $child;
        } else {
            $this->_children[] = $child;
        } 
        return $child;
    } 

    /**
     * イタレータを返す
     * 
     * @final 
     */
    final public function getIterator()
    {
        $tmp = new ArrayObject($this->_children);
        return $tmp->getIterator();
    } 
} 

/**
 * AutoSOAP_ReflectionAcceptに対して処理を行います。
 * 
 * @package jp.servlet.AutoSOAP
 * @author Sakamoto Kouichi <sakamoto@servlet.sakura.ne.jp> 
 */
interface AutoSOAP_ReflectionVisitor {
    /**
     * 結果を返す
     */
    public function getResult();
    /**
     * クラスを処理する
     * 
     * @param AutoSOAP_ReflectionAccept $class
     */
    public function visitClass(AutoSOAP_ReflectionAccept $class);
    /**
     * プロパティを処理する
     * 
     * @param AutoSOAP_ReflectionAccept $property
     */
    public function visitProperty(AutoSOAP_ReflectionAccept $property);
    /**
     * メソッドを処理する
     * 
     * @param AutoSOAP_ReflectionAccept $method
     */
    public function visitMethod(AutoSOAP_ReflectionAccept $method);
    /**
     * 関数を処理する
     * 
     * @param AutoSOAP_ReflectionAccept $func
     */
    public function visitFunction(AutoSOAP_ReflectionAccept $func);
    /**
     * 引数を処理する
     * 
     * @param AutoSOAP_ReflectionAccept $param
     */
    public function visitParameter(AutoSOAP_ReflectionAccept $param);
    /**
     * オブジェクトを処理する
     * 
     * @param AutoSOAP_ReflectionAccept $object
     */
    public function visitObject(AutoSOAP_ReflectionAccept $object);
    /**
     * エクステンションを処理する
     * 
     * @param AutoSOAP_ReflectionAccept $extension
     */
    public function visitExtension(AutoSOAP_ReflectionAccept $extension);
    /**
     * 子を処理する
     * 
     * @param AutoSOAP_ReflectionChildren $children
     */
    public function visitChildren(AutoSOAP_ReflectionChildren $children);
} 

?>