<?php
/**
 * AutoSOAP - Expanded SOAP Server
 * 
 * PHP version 5
 * 
 * @package jp.servlet.AutoSOAP
 * @author Sakamoto Kouichi <sakamoto@servlet.sakura.ne.jp> 
 * @copyright 2006 Sakamoto Kouichi
 * @license http://www.apache.org/licenses/LICENSE-2.0 Apache Software License 2.0
 * 
 * $Id: Service.class.php 103 2006-04-24 14:05:08Z sakamoto $
 */

/**
 * AutoSOAPサービス定義クラス
 * 
 * @package jp.servlet.AutoSOAP
 * @author Sakamoto Kouichi <sakamoto@servlet.sakura.ne.jp> 
 */
class AutoSOAP_Service {
    /**
     * タイムスタンプ(Y/m/d H:i:s)
     * nullの場合、WSDLファイルの更新日時から取得する
     * 
     * @var string 
     */
    private $timestamp = null;

    /**
     * クラス名
     * 
     * @var string 
     */
    private $class_name;
    /**
     * クラスのファイルパス名
     * 
     * @var string 
     */
    private $class_path;
    /**
     * WSDLファイルパス名
     * 
     * @var string 
     */
    private $wsdl_path;
    /**
     * XSDファイルパス名
     * 
     * @var string 
     */
    private $xsd_path;
    /**
     * その他のデータファイルパス名
     * 
     * @var string 
     */
    private $option_path;
    /**
     * ロケーション（クライアントがアクセスするURL）
     * 
     * @var string 
     */
    private $location;
    /**
     * ネームスペース
     * 
     * @var string 
     */
    private $namespace;

    /**
     * コンストラクタ
     * 
     * @access public 
     * @param AutoSOAPEnvironment $env 
     * @return void 
     */
    public function __construct($class_name, $class_path, $wsdl_path, $xsd_path, $option_path, $location, $namespace, $timestamp=null)
    {
        $this->class_name = $class_name;
        $this->class_path = $class_path;
        $this->wsdl_path = $wsdl_path;
        $this->xsd_path = $xsd_path;
        $this->option_path = $option_path;
        $this->location = $location;
        $this->namespace = $namespace;
        $this->timestamp = $timestamp;
    } 

    /**
     * ロケーションを返す
     * 
     * @return string 
     */
    public function getLocation()
    {
        return $this->location;
    } 

    /**
     * ネームスペースを返す
     * 
     * @return string 
     */
    public function getNamespace()
    {
        return $this->namespace;
    } 

    /**
     * クラス名を返す
     * 
     * @return string 
     */
    public function getClassName()
    {
        return $this->class_name;
    } 

    /**
     * クラスをインクルードする
     * 
     * @return void 
     */
    public function include_class()
    {
        if (!is_null($this->class_path)) {
            include_once($this->class_path);
        } 
    } 

    /**
     * WSDLファイルパスを返す
     * 
     * @return string 
     */
    public function getWsdlPath()
    {
        return $this->wsdl_path;
    } 

    /**
     * XSDファイルパスを返す
     * 
     * @return string 
     */
    public function getXsdPath()
    {
        return $this->xsd_path;
    } 

    /**
     * クラスマップのファイルパスを返す
     * 
     * @return string 
     */
    public function getOptionPath()
    {
        return $this->option_path;
    } 

    /**
     * タイムスタンプを返す
     * 
     * @access public 
     * @return integer 
     */
    public function getTimestamp()
    {
        $timezone = date_default_timezone_get();
        if(!$timezone) {
            date_default_timezone_get('GMT');
        }
        $timestamp = (is_null($this->timestamp))
                   ? filemtime($this->class_path)
                   : preg_replace("/^([0-9]{4})?\/?([0-9]{1,2})?\/?([0-9]{1,2})? ?([0-9]{1,2})?:?([0-9]{1,2})?:?([0-9]{1,2})?/e", "mktime(((int)'$4'),((int)'$5'),((int)'$6'),((int)'$2'),((int)'$3'),((int)'$1'))", $this->timestamp);

        if(!$timezone) {
            date_default_timezone_get($timezone);
        }
        return $timestamp;
    }
    
    /**
     * WSDLファイルの更新が必要か返す
     * 
     * @access public 
     * @return boolean 
     */
    public function need_update_wsdl_file()
    {
        return (is_file($this->getWsdlPath()))
             ? ($this->getTimestamp() != filemtime($this->getWsdlPath()))
             : true ;
    } 

    /**
     * WSDLファイルの更新が必要か返す
     * 
     * @access public 
     * @return boolean 
     */
    public function need_update_xsd_file()
    {
        return (is_file($this->getXsdPath()))
              ? ($this->getTimestamp() != filemtime($this->getXsdPath()))
              : true ;
    } 
} 
?>