<?php
/**
 * AutoSOAP - Expanded SOAP Server
 * 
 * PHP version 5
 * 
 * @package jp.servlet.AutoSOAP
 * @author Sakamoto Kouichi <sakamoto@servlet.sakura.ne.jp> 
 * @copyright 2006 Sakamoto Kouichi
 * @license http://www.apache.org/licenses/LICENSE-2.0 Apache Software License 2.0
 * 
 * $Id: class2wsdlVisitor.class.php 97 2006-04-23 10:05:16Z sakamoto $
 */

/**
 * クラス→WSDLデータへの変換用ビジタークラス
 * 
 * @package jp.servlet.AutoSOAP
 * @author Sakamoto Kouichi <sakamoto@servlet.sakura.ne.jp> 
 */
class AutoSOAP_class2wsdlVisitor implements AutoSOAP_ReflectionVisitor {
    // SOAPデータ生成管理
    private $_class2soap; 
    // WSDLの各パーツ
    private $_portType;
    private $_binding;
    private $_service; 
    // 検証時のエラーメッセージマップ
    private $_ValidateErrorMap = array();

    // エンコード
    private $_encoding = null;

    /**
     * 初期化
     */
    public function init($target_uri, $wsdl_name, $location)
    {
        $wsdl = AutoSOAP_XMLogic_WsdlFactory :: getInstance(); 
        // XMLスキーマタグ生成クラス
        $schema = AutoSOAP_XMLogic_SchemaFactory :: getInstance();

        /**
         * WSDLの大枠の作成
         */ 
        // Root
        $root = $wsdl -> createRoot($target_uri); 
        // Types
        $types = $wsdl -> createTypes();
        $root -> add($types); 
        // Types_Schema
        $types_schema = $schema -> createRoot($target_uri);
        $types -> add($types_schema); 
        // portType
        $this -> _portType = $wsdl -> createPortType($wsdl_name . '_portType');
        $root -> add($this -> _portType); 
        // Binding
        $this -> _binding = $wsdl -> createBinding($wsdl_name . '_Binding', $wsdl_name . '_portType', $target_uri, AutoSOAP_XMLogic_WsdlFactory :: XMLNS_HTTP);
        $root -> add($this -> _binding); 
        // Service
        $this -> _service = $wsdl -> createService($wsdl_name . '_Service', $wsdl_name . '_Binding', $target_uri, $location);
        $root -> add($this -> _service);

        $class2xsd = new AutoSOAP_class2xsdVisitor();
        $class2xsd -> init($types_schema, $wsdl); 
        
        $this -> _class2soap = new AutoSOAP_class2soap($target_uri, $root, $class2xsd);
    } 

    /**
     * 結果を返す
     */
    public function getResult($to_encoding = null, $from_encoding = null)
    {
        $encoding = (is_null($from_encoding)) ? $this->_encoding : $from_encoding;
        return $this -> _class2soap -> getResult($to_encoding, $encoding);
    }

    /**
     * 子を処理する
     */
    public function visitChildren(AutoSOAP_ReflectionChildren $children)
    {
        return AutoSOAP_class2soap :: visitChildren($this, $children);
    } 

    /**
     * クラスを処理する
     */
    public function visitClass(AutoSOAP_ReflectionAccept $class)
    {
        $wsdl = AutoSOAP_XMLogic_WsdlFactory :: getInstance();

        /**
         * インターフェースから新しい型をインクルードする
         */
        $this -> _class2soap -> createNewTypes($class);

        /**
         * アノテーションの取得
         */ 
        $annotation = AutoSOAP_Anno :: parseReflector($class);
        // コメントの取得
        $this->_class2soap->appendXML($wsdl->createDoc(trim($annotation->getComment())),0);
        //$this->_service->add($wsdl->createDoc($annotation->getComment()),0);

        // エンコードを取得
        $this->_encoding = $annotation->execute(new AutoSOAP_Anno_EncodingCmd()); 

        // 新しい型の取得
        $elemTypes = AutoSOAP_ElementType :: createByClass($annotation);
        foreach($elemTypes as $newtype => $elemType) {
            $this -> _class2soap -> getElementInfo('', $elemType, array('new_type' => $newtype));
        }
        
        // メソッド追加
        $class -> getMethods() -> accept($this);
    } 

    /**
     * メソッドを処理する
     */
    public function visitMethod(AutoSOAP_ReflectionAccept $method)
    {
        if (!$this -> _class2soap -> createMethodInfo($method)) {
            return ;
        } 

        $wsdl = AutoSOAP_XMLogic_WsdlFactory :: getInstance();
        $method_name = $method -> getName();

        /**
         * 検証時のエラーメッセージ取得
         */
        $validate_error = $this -> _class2soap -> getValidateError();
        if (!is_null($validate_error))
            $this -> _ValidateErrorMap[$method_name] = $validate_error;

        /**
         * リクエストメソッド追加
         */
        $request_name = $method_name . "Request";
        $request = $wsdl -> createMessage($request_name); 
        // パラメータ追加
        $parameters = $method -> getParameters() -> accept($this);
        if (0 < count($parameters)) {
            foreach($parameters as $param) {
                $request -> add($param);
            } 
        } 
        $this -> _class2soap -> appendXML($request, 2);

        /**
         * レスポンスメソッド追加
         */
        $response_name = $method_name . "Response";
        $response = $wsdl -> createMessage($response_name);
        $response -> add($this -> getMessagePort('return', $this -> _class2soap -> getReturnInfo()));
        $this -> _class2soap -> appendXML($response, 3);

        /**
         * portTypeへ追加
         */
        $this -> _portType -> add($wsdl -> createOperationOfPortType($method_name,
                $request_name,
                $response_name,
                $this -> _class2soap -> getTargetURI())
            );

        /**
         * bindingへ追加
         */
        $this -> _binding -> add($wsdl -> createOperationOfBinding($method_name,
                (0 < strlen($request_name)),
                (0 < strlen($response_name)),
                $this -> _class2soap -> getTargetURI())
            );
    } 

    /**
     * 引数を処理する
     */
    public function visitParameter(AutoSOAP_ReflectionAccept $param)
    {
        $elemType = $this -> _class2soap -> getParameterInfo($param -> getName());
        $options = AutoSOAP_class2soap :: getParamOptions($param);

        return $this -> getMessagePort($param -> getName(), $elemType, $options);
    } 

    /**
     * プロパティを処理する
     */
    public function visitProperty(AutoSOAP_ReflectionAccept $property)
    {
    } 
    /**
     * 関数を処理する
     */
    public function visitFunction(AutoSOAP_ReflectionAccept $func)
    {
    } 
    /**
     * オブジェクトを処理する
     */
    public function visitObject(AutoSOAP_ReflectionAccept $object)
    {
    } 
    /**
     * エクステンションを処理する
     */
    public function visitExtension(AutoSOAP_ReflectionAccept $extension)
    {
    } 

    /**
     * メッセージポートのエレメントを返す。
     */
    private function getMessagePort($name, AutoSOAP_ElementType $elemType, $options = null)
    {
        $wsdl = AutoSOAP_XMLogic_WsdlFactory :: getInstance();
        $info = $this -> _class2soap -> getElementInfo($name, $elemType, $options);
        return $wsdl -> createPart($info['name'], $info['datatype'], $info['namespace'], $options);
    } 

    /**
     * SOAPデータ生成管理を返す
     */
    public function getClass2soap()
    {
        return $this -> _class2soap;
    } 

    /**
     * 検証時のエラーメッセージマップを返す
     */
    public function getValidateErrorMap()
    {
        return $this -> _ValidateErrorMap;
    } 

    /**
     * エンコードを返す
     */
    public function getEncoding()
    {
        return $this->_encoding;
    }
} 

?>